/*
 * #%L
 * IsisFish
 * 
 * $Id: AnalysePlanStorage.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2005 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.simulator.AnalysePlan;
import fr.ifremer.isisfish.util.Doc;
import fr.ifremer.isisfish.util.DocHelper;
import fr.ifremer.isisfish.util.Docable;
import fr.ifremer.isisfish.vcs.VCSException;

import static org.nuiton.i18n.I18n._;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.lang.reflect.Field;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Class permettant de representer un plan d'analyse.
 * Un plan d'analyse est un fichier Java que l'on compile si besoin.
 * <p/>
 * Gere les fichiers VCS de type {@link AnalysePlan} (package analyseplans)
 *
 * Created: 17 août 2005 11:11:51 CEST
 *
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 *
 * @version $Revision: 3124 $
 *
 * Last update: $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $ by : $Author: chatellier $
 */
public class AnalysePlanStorage extends JavaSourceStorage implements Docable { // AnalysePlanStorage

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(AnalysePlanStorage.class);
    
    public final static String ANALYSE_PLAN_PATH = "analyseplans";

    /** Template freemarker pour les plans d'analyse . */
    public static final String ANALYSE_PLAN_TEMPLATE = "templates/script/analyseplan.ftl";

    @SuppressWarnings("unchecked")
    static private Map<String, AnalysePlanStorage> plansCache =
            (Map<String, AnalysePlanStorage>) new ReferenceMap();

    /**
     * Construit un nouveau AnalysePlan storage.
     *
     * @param rootSrc   repertoire root de stockage des plans
     * @param directory le repertoire ou devrait se trouver le plan
     * @param name      le nom du plan
     */
    protected AnalysePlanStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name);
    }

    static public File getAnalysePlanDirectory() {
        File result = new File(getContextDatabaseDirectory(), ANALYSE_PLAN_PATH);
        result.mkdirs();
        return result;
    }

    /**
     * Retourne le nom de toutes les plans existantes
     *
     * @return les noms de tous les plans existant en local
     */
    static public List<String> getAnalysePlanNames() {
        File dir = getAnalysePlanDirectory();
        return AnalysePlanStorage.getStorageNames(dir);
    }

    /**
     * Retourne le storage pour le plan demandé.
     *
     * @param name le nom du plan souhaité
     * @return Le storage pour le plan
     */
    static public AnalysePlanStorage getAnalysePlan(String name) {
        String cacheName = getContextDatabaseCacheKey(name);
        AnalysePlanStorage result = plansCache.get(cacheName);
        if (result == null) {
            result = new AnalysePlanStorage(getContextDatabaseDirectory(),
                    getAnalysePlanDirectory(), name);
            plansCache.put(cacheName, result);
        }
        return result;
    }

    /**
     * Retourne une nouvelle instance du plan. Compile le fichier si besoin
     *
     * @return retourne une nouvelle instance du plan
     * @throws IsisFishException s'il y a un probleme d'instanciation
     */
    public AnalysePlan getNewAnalysePlanInstance() throws IsisFishException {
        Object result = getNewInstance();
        return (AnalysePlan) result;
    }

    /**
     * Permet de recuperer les parametres du plan.
     *
     * @return Une map contenant le nom du parametre et son type
     * @throws IsisFishException s'il est impossible de retourner une map
     *                           convenable
     */
    public Map<String, Class<?>> getParameterNames() throws IsisFishException {
        Map<String, Class<?>> result = null;
        // On essai de recuperer les paramètres depuis la classe compilé
        // si c possible        
        try {
            if (0 == compile(false, null)) {
                AnalysePlan plan = getNewAnalysePlanInstance();
                result = AnalysePlanStorage.getParameterNames(plan);
            }
        } catch (Exception eee) {
            log.info(_("isisfish.error.plan.parameter"), eee);
        }
        // si on ne reussi pas avec la classe compilé on essai en parsant le
        // source
        if (result == null) {
            try {
                //String code = getContent();
                log.fatal("FIXME a faire recherche des parametre dans le source");
                throw new IsisFishException(_("isisfish.error.source.parameter"));
                // TODO implanter la recherche des parametres.
                // Se sont les attributs commencant par 'param_' et ils doivent 
                // etre public non static, non transient.
                // ex: public Date param_Date.
                // Lorsqu'on ajoute un champs dans la map il ne faut pas mettre
                // le prefix param.
            } catch (Exception eee) {
                throw new IsisFishException(_("isisfish.error.source.parameter"), eee);
            }
        }
        return result;
    }


    /**
     * Recherche par introspection tous les parametres de la classe
     * commencant par param_
     *
     * @param plan le plan dont on souhaite les infos de parametre
     * @return retourne le nom et le type des parametres du plan
     */
    public static Map<String, Class<?>> getParameterNames(AnalysePlan plan) {
        Map<String, Class<?>> result = new LinkedHashMap<String, Class<?>>();
        for (Field field : plan.getClass().getFields()) {
            if (field.getName().startsWith(PARAM_PREFIX)) {
                result.put(field.getName().substring(PARAM_PREFIX.length()),
                        field.getType());
            }
        }
        return result;
    }

    /**
     * Donne la valeur d'un parametre par introspection
     *
     * @param name le nom du parametre
     * @param plan le plan dont on souhaite la valeur du parametre
     * @return la valeur courante du parametre
     * @throws IsisFishException s'il y a un probleme pour recuperer la valeur
     */
    static public Object getParameterValue(AnalysePlan plan, String name) throws IsisFishException {
        if (plan == null || name == null || "".equals(name)) {
            return null;
        }
        try {
            String fieldName = PARAM_PREFIX + name;
            Field field = plan.getClass().getDeclaredField(fieldName);
            return field.get(plan);
        } catch (IllegalAccessException eee) {
            throw new IsisFishException("Can't get plan parameter: " + name, eee);
        } catch (NoSuchFieldException eee) {
            throw new IsisFishException("Can't get plan parameter: " + name, eee);
        }
    }

    /**
     * Modifie la valeur d'un attribut pas introspection
     *
     * @param name  le nom de l'attribut
     * @param value la valeur de l'attribut
     * @param plan  le plan dont on souhaite modifier la valeur de parametre
     * @throws IsisFishException s'il y a un probleme
     */
    static public void setParameterValue(AnalysePlan plan, String name, Object value) throws IsisFishException {
        try {
            String fieldName = PARAM_PREFIX + name;
            Field field = plan.getClass().getDeclaredField(fieldName);
            field.set(plan, value);
        } catch (IllegalAccessException eee) {
            throw new IsisFishException("Can't modify plan parameter: " + name + " with '" + value + "'(" + ObjectUtils.identityToString(value) + ")", eee);
        } catch (NoSuchFieldException eee) {
            throw new IsisFishException("Can't modify plan parameter: " + name + " with '" + value + "'(" + ObjectUtils.identityToString(value) + ")", eee);
        } catch (IllegalArgumentException eee) {
            throw new IsisFishException("Can't modify plan parameter: " + name + " with '" + value + "'(" + ObjectUtils.identityToString(value) + ")", eee);
        }
    }

    /**
     * Recupere les parametres et leur valeur pour les retourner sous forme
     * de chaine. Pour pouvoir par exemple les afficher a l'utilisateur
     *
     * @param plan le plan dont on souhaite la valeur du parametre en string
     * @return la valeur sous forme de string
     * @throws IsisFishException
     */
    public static String getParamAsString(AnalysePlan plan) throws IsisFishException {
        StringBuffer result = new StringBuffer();

        for (String name : getParameterNames(plan).keySet()) {
            Object value = getParameterValue(plan, name);

            result.append(name).append(" : ").append(value);
            result.append("\n");
        }
        return result.toString();
    }

    static public void checkout() throws VCSException {
        checkout(IsisFish.config.getDatabaseDirectory(), ANALYSE_PLAN_PATH);
    }

    /**
     * Retourne la liste des noms de tous les plans disponibles en local qui
     * ne sont pas encore sur le serveur VCS.
     *
     * @return liste de noms de plans
     */
    public static List<String> getNewAnalysePlanNames() {
        List<String> result = getAnalysePlanNames();
        result.removeAll(getRemoteAnalysePlanNames());
        return result;
    }

    /**
     * Retourne la liste des noms de tous les plans disponibles sur le
     * serveur VCS.
     *
     * @return la liste des noms de tous les plans disponibles sur le serveur
     *         VCS. Si le serveur n'est pas disponible la liste retournée est
     *         vide.
     */
    public static List<String> getRemoteAnalysePlanNames() {
        File dir = getAnalysePlanDirectory();
        return getRemoteStorageNames(dir);
    }

    /**
     * Retourne la liste des noms de tous les plans disponibles sur le
     * serveur VCS qui ne sont pas encore en local
     *
     * @return liste de noms de regions
     * @throws VCSException
     */
    public static List<String> getNewRemoteAnalysePlanNames() throws VCSException {
        List<String> result = getRemoteAnalysePlanNames();
        result.removeAll(getAnalysePlanNames());
        return result;
    }

    /**
     * @return the @Doc of the underlied AnalysePlan class
     * @see DocHelper
     * @see Doc
     * @see Docable
     */
    public Doc getClassDoc() {
        Doc result = null;
        try {
            Class<?> klazz = getCodeClass();
            result = DocHelper.getClassDoc(klazz);
        } catch (IsisFishException e) {
            log.warn(_("isisfish.error.not.found.code", this));
        }
        return result;
    }

    /**
     * @param fieldName name of the field to inspect
     * @return the @Doc of the given field of the underlied AnalysePlan class
     * @see DocHelper
     * @see Doc
     * @see Docable
     */
    public Doc getFieldDoc(String fieldName) {
        Doc result = null;
        try {
            Class<?> klazz = getCodeClass();
            result = DocHelper.getFieldDoc(klazz, fieldName);
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.field", fieldName, this));
        }
        return result;
    }

    /**
     * @param paramName the name of the param to inspect
     * @return the doc associated with the param,
     *         (says the field PARAM_PREFIX+fieldName)
     */
    public Doc getParamDoc(String paramName) {
        Doc doc = getFieldDoc(PARAM_PREFIX + paramName);
        return doc;
    }

    /**
     * <b>Be ware this method require to instanciate a AnalysePlan, so
     * it would be better to call as often as possible.</b>
     *
     * @return the descript of the instanciate AnalysePlan
     * @see Doc
     * @see Docable
     */
    public String getDescription() {
        String result = null;
        try {
            AnalysePlan analysePlan = getNewAnalysePlanInstance();
            result = analysePlan == null ? null : analysePlan.getDescription();
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.description", this));
        }
        return result;
    }

} // AnalysePlanStorage

