/*
 * #%L
 * IsisFish
 * 
 * $Id: CodeSourceStorage.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;

import org.nuiton.util.FileUtil;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.vcs.VCSException;

/**
 * Classes abstraite permettant la gestion des fichiers de code source.
 * 
 * {@link fr.ifremer.isisfish.datastore.ScriptStorage},
 * {@link fr.ifremer.isisfish.datastore.RuleStorage},
 * {@link fr.ifremer.isisfish.datastore.ExportStorage},
 * {@link fr.ifremer.isisfish.datastore.AnalysePlanStorage},
 * {@link fr.ifremer.isisfish.datastore.SensitivityStorage}
 * 
 * Created: 21 janv. 2006 15:20:24
 *
 * @author poussin
 *
 * @version $Revision: 3124 $
 *
 * Last update: $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $ by : $Author: chatellier $
 */
public abstract class CodeSourceStorage extends VersionStorage {

    /**
     * Le nom de la classe sans le package.
     */
    protected String name = null;

    /**
     * Le contenu du fichier.
     */
    protected String content = null;

    /**
     * Date de derniere mise a jour de content, sert a detecter si content doit etre relu.
     */
    protected long lastContentUpdate = 0;

    /**
     * La derniere taille connu du content, sert a detecter si content doit etre relu.
     */
    protected long lastContentLength = 0;

    /**
     * Contruit un nouveau rule storage.
     *
     * @param rootSrc   repertoire racine des sources
     * @param directory le repertoire ou devrait se trouver la classe. Ce
     *                  répertoire doit etre un sous répertoire de rootSrc
     * @param name      le nom de la classe
     * @param suffix    l'extension des fichiers
     */
    protected CodeSourceStorage(File rootSrc, File directory, String name,
            String suffix) {
        // if name end with suffix we don't add suffix to filename
        super(rootSrc, name.endsWith(suffix) ? new File(directory, name)
                : new File(directory, name + suffix));
        this.name = name.endsWith(suffix) ? name : name + suffix;
    }

    /*
     * @see fr.ifremer.isisfish.datastore.VersionStorage#prepare()
     */
    @Override
    protected void prepare() {
        // Il n'y a rien a faire pour les code sources
    }

    /*
     * @see fr.ifremer.isisfish.datastore.VersionStorage#getFiles(boolean)
     */
    @Override
    protected List<File> getFiles(boolean withParent) {
        
        // don't work in svn, commiting each parent
        // just return current script
        List<File> result = Collections.singletonList(getFile());

        return result;
    }

    /**
     * Check if script exists.
     * 
     * @return {@code true} if storage file exists
     */
    public boolean exists() {
        return getFile().exists();
    }

    /**
     * Get storage name.
     * 
     * @return the name.
     */
    public String getName() {
        return this.name;
    }

    /**
     * Retourne le contenu du fichier. Si le fichier reel est plus recent
     * que la derniere lecture, alors il est relu automatiquement. Ce
     * mecanisme de relecture automatique peut ne pas fonctionner si entre
     * la derniere lecture de content et la modification directement du fichier
     * a ete fait dans la meme seconde et que le fichier modifier a la meme
     * taille que le precedent. Pour reellement forcer la relecture on peut
     * utiliser la methode {@link #reload()}}
     *
     * @return le contenu du fichier
     */
    public String getContent() {
        if (content == null || getFile().lastModified() > lastContentUpdate
                || getFile().length() != lastContentLength) {
            if (!exists()) {
                content = "";
            } else {
                try {
                    content = FileUtil.readAsString(getFile());
                    lastContentUpdate = System.nanoTime();
                    lastContentLength = getFile().length();
                } catch (IOException eee) {
                    throw new IsisFishRuntimeException(eee);
                }
            }
        }
        return content;
    }

    /**
     * Set file content.
     * 
     * @param content new content
     * @throws IOException
     */
    public void setContent(String content) throws IOException {
        this.content = content;
        // FIXME hard coded encoding ?
        FileUtil.writeString(getFile(), content, "utf-8");
        lastContentUpdate = System.nanoTime();
        lastContentLength = getFile().length();
    }

    /**
     * Permet de forcer la relecture du fichier sur le disque.
     */
    public void reload() {
        content = null;
    }

    /**
     * Return diff between current file content and VCS file content.
     * 
     * @return diff as string
     * @throws VCSException
     * @throws IOException
     */
    public String diff() throws VCSException, IOException {
        return getVCS().getDiff(getFile());
    }
}
