/*
 * #%L
 * IsisFish
 * 
 * $Id: Trace.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.aspect;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.aspectwerkz.annotation.After;
import org.codehaus.aspectwerkz.annotation.AfterThrowing;
import org.codehaus.aspectwerkz.annotation.Aspect;
import org.codehaus.aspectwerkz.annotation.Before;
import org.codehaus.aspectwerkz.annotation.Expression;
import org.codehaus.aspectwerkz.definition.Pointcut;
import org.codehaus.aspectwerkz.joinpoint.JoinPoint;
import org.codehaus.aspectwerkz.joinpoint.MethodSignature;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.simulator.SimulationContext;

/**
 * Permet de tracer les appels aux methodes utilisateur ainsi que l'execution
 * a ces methodes. La difference entre les deux est lors de l'utilisation du
 * cache les appels seront superieur a l'execution car certaine valeur seront
 * reutilisé dans le cache
 * 
 * Created: 25 août 06 22:19:21
 *
 * @author poussin
 * @version $Revision: 3124 $
 *
 * Last update: $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * by : $Author: chatellier $
 */
@Aspect("perJVM")
public class Trace {

    /** to use log facility, just put in your code: log.info("..."); */
    static private Log log = LogFactory.getLog(Trace.class);

    static private List<Trace> instances = new ArrayList<Trace>();

    @Expression("execution(* scripts..*(..))"
            + " || execution(* simulators..*(..))"
            + " || execution(* rules..*(..)) "
            + " || execution(* analyseplans..*(..)) "
            + " || execution(* formules..*(..))")
    Pointcut executeMethod;

    public Trace() {
        instances.add(this);
    }

    /**
     * Return trace object from context.
     * 
     * @return trace object from context
     */
    protected fr.ifremer.isisfish.util.Trace getTrace() {
        SimulationContext context = SimulationContext.get();

        fr.ifremer.isisfish.util.Trace result = (fr.ifremer.isisfish.util.Trace) context
                .getValue(fr.ifremer.isisfish.util.Trace.class.getName());

        if (result == null) {
            throw new IsisFishRuntimeException("No trace object found in context");
        }

        return result;
    }

    @Before("executeMethod")
    public void traceBeforeExecute(JoinPoint jp) {
        getTrace().traceBefore();
    }

    @AfterThrowing(type = "Exception", pointcut = "executeMethod")
    public void traceAfterThrowingExecute(JoinPoint jp) {
        // si une exeption est leve, il faut faire la meme chose
        traceAfterExecute(jp);
    }

    @After("executeMethod")
    public void traceAfterExecute(JoinPoint jp) {
        Method method = ((MethodSignature) jp.getSignature()).getMethod();
        getTrace().traceAfterCall(method);
    }

    /**
     * Affiche les statistiques
     *
     * @return statistics as string
     */
    public static String printStatistiqueAndClear() {
        StringBuffer result = new StringBuffer();
        for (Trace trace : instances) {
            result.append(trace.getTrace().printStatisticAndClear());
        }
        instances.clear();
        return result.toString();
    }
}
