/*
 * #%L
 * IsisFish
 * 
 * $Id: Cache.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.aspect;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.aspectwerkz.annotation.Around;
import org.codehaus.aspectwerkz.annotation.Aspect;
import org.codehaus.aspectwerkz.annotation.Expression;
import org.codehaus.aspectwerkz.definition.Pointcut;
import org.codehaus.aspectwerkz.joinpoint.JoinPoint;
import org.codehaus.aspectwerkz.joinpoint.MethodRtti;
import org.codehaus.aspectwerkz.joinpoint.MethodSignature;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.simulator.SimulationContext;

/**
 * Cache aspect.
 * 
 * Created: 25 août 06 22:42:47
 *
 * @author poussin
 * @version $Revision: 3124 $
 *
 * Last update: $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * by : $Author: chatellier $
 */
@Aspect("perJVM")
public class Cache {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(Cache.class);

    static private List<Cache> instances = new ArrayList<Cache>();
    
    protected long totalCall = 0;
    protected long cacheUsed = 0;
    
    protected Map cache = new ReferenceMap(ReferenceMap.HARD, ReferenceMap.SOFT);
//    protected Map cache = new AdaptaptativeCache(1000, 95);
    
    @Expression("execution(* scripts..*(..))")
    Pointcut scriptsMethod;

    public Cache() {
        instances.add(this);
    }

    /**
     * Return trace object from context.
     * 
     * @return trace object from context
     */
    protected fr.ifremer.isisfish.util.Trace getTrace() {
        SimulationContext context = SimulationContext.get();

        fr.ifremer.isisfish.util.Trace result = (fr.ifremer.isisfish.util.Trace) context
                .getValue(fr.ifremer.isisfish.util.Trace.class.getName());

        if (result == null) {
            throw new IsisFishRuntimeException("No trace object found in context");
        }

        return result;
    }
    
//    @Before("scriptsMethod")
//    public void traceBeforeExecute (JoinPoint jp) {
//        System.out.println("before !!!");
//    }
//
//    @AfterThrowing("scriptsMethod")
//    public void traceAfterThrowingExecute (JoinPoint jp) {
//        System.out.println("throwing !!!");
//    }
//    
//    @After("scriptsMethod")
//    public void traceAfterExecute (JoinPoint jp) {
//        System.out.println("After !!!");
//    }

    
    @Around("scriptsMethod")
    public Object call(final JoinPoint jp) throws Throwable {
        totalCall++;
        Object key = computeKey(jp);
        Object result = cache.get(key);
        if (result == null) {
            
            // computation increment (/ by 0)
            // FIXME need to be called, but fail with empty stack
            //Method method = ((MethodSignature)jp.getSignature()).getMethod();
            //getTrace().traceAfterComputation(method);

            result = realCall(jp);
//            addListenerFor(key); // pas necessaire car on a la date et pour une date donnée rien ne peut changer
            if (result != null) { // util pour les methodes retournant void, ne fonctionne pas si on met AND !execute(void *(..)) dans l'aspect. En fait fonction seulement si utilisé avec les traces :(
                cache.put(key, result);
            }
        } else {
            cacheUsed++;
        }
        if (log.isTraceEnabled()) {
            log.trace(((MethodSignature)jp.getSignature()).getMethod()
                    + " args " + Arrays.toString(((MethodRtti)jp.getRtti()).getParameterValues())
                    + " result = " + result);
        }
        return result;
    }
    
    /**
     * On fait l'appel reel dans une autre methode pour pouvoir le savoir
     * dans les traces
     * 
     * @param jp
     * @return ?
     * @throws Throwable
     */
    protected Object realCall(final JoinPoint jp) throws Throwable {
        Object result = jp.proceed();
        return result;
    }

    /**
     * Attention pour avoir une chaine en sortie on prend la reprensentation
     * toString des arguments. Mais en utilisant la methode toString implanté
     * dans Object.
     * <p>
     * Pour les objets de style Number ou String, il faut prendre le vrai
     * toString, pour que 2 soit bien egal a 2. 
     * 
     * @param jp
     * @return ?
     */
    protected Object computeKey(JoinPoint jp) {            
        Method method = ((MethodSignature)jp.getSignature()).getMethod();
        Object[] args = ((MethodRtti)jp.getRtti()).getParameterValues();
        
        String result = method.toString();
        for (Object o : args) {
            result += ";";
            if (o instanceof Number || o instanceof String) {
                result += o.toString();
            } else {
                result += ObjectUtils.identityToString(o);
            }
        }

        return result;
        
//        Method method = ((MethodSignature)jp.getSignature()).getMethod();
//        Object[] args = ((MethodRtti)jp.getRtti()).getParameterValues();
//        
//        Object[] keys = new Object[args.length + 1];
//        keys[0] = method;
//        System.arraycopy(args, 0, keys, 1, args.length);
//        
//        MultiKey result = new MultiKey(keys, false);
//        
//        return result;
    }  
//    protected HashMapMultiKey.Key computeKey(JoinPoint jp) {            
//        Method method = ((MethodSignature)jp.getSignature()).getMethod();
//        Object[] args = ((MethodRtti)jp.getRtti()).getParameterValues();
//        
//        HashMapMultiKey.Key result = new HashMapMultiKey.Key();
//        result.add(method);
//        for (Object arg : args) {
//            result.add(arg);
//        }
//        
//        return result;
//    }  
    
    
    /**
     * @return Returns the cacheUsed.
     */
    static public long getCacheUsed() {
        long result = 0;
        for (Cache cache : instances) {
            result += cache.cacheUsed;
        }
        return result;
    }
    
    /**
     * @return Returns the totalCall.
     */
    static public long getTotalCall() {
        long result = 0;
        for (Cache cache : instances) {
            result += cache.totalCall;
        }
        return result;
    }
    
    /**
     * Affiche les statistiques
     *
     */
    static public String printStatistiqueAndClear() {
        StringBuffer result = new StringBuffer();
        for (Cache cache : instances) {
            result.append("--- Cache Statistiques ---\n");
            result.append("Total call: " + cache.totalCall + "\n");
            result.append("Cache used: " + cache.cacheUsed + "\n");
            result.append("Cache usage: " + (100*cache.cacheUsed/cache.totalCall) + "%" + "\n");
            result.append("--------------------\n");
            cache.cache.clear();
        }
        System.out.println(result.toString());
        instances.clear();
        return result.toString();
    }
    
//    /**
//     * Parcours les elements de la cle et pour ceux du type Entities se
//     * met listener pour pouvoir supprimer l'entre du cache lors de leur
//     * modification
//     * 
//     * @param key
//     */
//    protected void addListenerFor(HashMapMultiKey.Key key) {
//    }
    
    
    class AdaptaptativeCache extends LinkedHashMap<String, Object> {
        protected int maxMemory = 95;
        /**
         * 
         * @param capacity initial capacity
         * @param maxMemory maximum memory used (0-100)
         */
        public AdaptaptativeCache(int capacity, int maxMemory) {
            super(capacity, 0.75f, true);
            this.maxMemory = maxMemory;
        }
        
        /* (non-Javadoc)
         * @see java.util.LinkedHashMap#removeEldestEntry(java.util.Map.Entry)
         */
        @Override
        protected boolean removeEldestEntry(Entry<String, Object> eldest) {
            double free = 100.0 * Runtime.getRuntime().freeMemory() / Runtime.getRuntime().maxMemory() ;
            boolean result = 100 - free > maxMemory;
            return result;
        }
    }
}


