/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font.freetype;

import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.PathIterator;

import de.intarsys.cwt.common.ShapeWrapper;
import de.intarsys.cwt.environment.IGraphicsContext;
import de.intarsys.cwt.freetype.Freetype;
import de.intarsys.cwt.freetype.GlyphSlot;
import de.intarsys.cwt.freetype.Outline;
import de.intarsys.pdf.content.GraphicsState;
import de.intarsys.pdf.content.TextState;
import de.intarsys.pdf.font.PDGlyphs;
import de.intarsys.pdf.platform.cwt.font.IPlatformFont;
import de.intarsys.pdf.platform.cwt.font.IPlatformGlyphs;
import de.intarsys.pdf.platform.cwt.font.PlatformFontException;
import de.intarsys.pdf.platform.cwt.rendering.ICSPlatformDevice;
import de.intarsys.tools.geometry.ApplyTransformationShape;
import de.intarsys.tools.geometry.TransformedShape;

public class FreetypeGlyphs implements IPlatformGlyphs {

	final private static Shape EMPTY = new GeneralPath();

	final private FreetypeFont font;

	final private PDGlyphs glyphs;

	private Shape shape;

	private int width;

	protected FreetypeGlyphs(FreetypeFont font, PDGlyphs glyphs)
			throws PlatformFontException {
		super();
		this.font = font;
		this.glyphs = glyphs;
		init();
	}

	protected void createContour(GeneralPath path, Outline outline, int start,
			int end) {
		boolean reverse = outline.isReverseFill();
		float initialX = 0;
		float initialY = 0;
		float cx1 = 0;
		float cy1 = 0;
		float cx2 = 0;
		float cy2 = 0;
		int cCount = 0;
		boolean newContour = true;
		int pointIndex = start;
		while (true) {
			byte tag = outline.getTag(pointIndex);
			int px = outline.getPointX(pointIndex);
			int py = outline.getPointY(pointIndex);
			float x = px;
			float y = py;
			if (newContour) {
				initialX = x;
				initialY = y;
				path.moveTo(x, y);
			} else {
				if ((tag & Freetype.CURVE_TAG_ON) != 0) {
					if (cCount == 0) {
						path.lineTo(x, y);
					} else if (cCount == 1) {
						path.quadTo(cx1, cy1, x, y);
					} else {
						path.curveTo(cx1, cy1, cx2, cy2, x, y);
					}
					cCount = 0;
				} else {
					if ((tag & Freetype.CURVE_TAG_CUBIC) == 0) {
						if (cCount == 1) {
							float midX = cx1 + ((x - cx1) / 2f);
							float midY = cy1 + ((y - cy1) / 2f);
							path.quadTo(cx1, cy1, midX, midY);
							cCount = 0;
						}
					}
					if (cCount == 0) {
						cx1 = x;
						cy1 = y;
						cCount = 1;
					} else {
						cx2 = x;
						cy2 = y;
						cCount = 2;
					}
				}
			}
			newContour = false;
			if (reverse) {
				pointIndex--;
				if (pointIndex < start) {
					pointIndex = end;
				}
				if (pointIndex == start) {
					break;
				}
			} else {
				if (pointIndex == end) {
					break;
				}
				pointIndex++;
			}
		}

		if (cCount == 0) {
			path.lineTo(initialX, initialY);
		} else if (cCount == 1) {
			path.quadTo(cx1, cy1, initialX, initialY);
		} else {
			path.curveTo(cx1, cy1, cx2, cy2, initialX, initialY);
		}
		path.closePath();
	}

	protected Shape createShape() {
		synchronized (font) {
			GlyphSlot slot;
			try {
				// ensure correct slot is selected
				slot = loadGlyphSlot();
			} catch (PlatformFontException e) {
				// this should not happen any more
				return EMPTY;
			}
			Outline outline = slot.getOutline();
			int numContours = outline.getNumContours();
			if (numContours == 0) {
				// empty shape
				return EMPTY;
			}
			int rule = outline.isEvenOddFill() ? PathIterator.WIND_EVEN_ODD
					: PathIterator.WIND_NON_ZERO;
			GeneralPath tempShape = new GeneralPath(rule, 40);
			ShapeWrapper iShape = new ShapeWrapper(tempShape);
			int contourStart = 0;
			for (int i = 0; i < numContours; i++) {
				int contourEnd = outline.getContour(i);
				createContour(tempShape, outline, contourStart, contourEnd);
				contourStart = contourEnd + 1;
			}
			return iShape;
		}
	}

	public FreetypeFont getFont() {
		return font;
	}

	public PDGlyphs getGlyphs() {
		return glyphs;
	}

	public IPlatformFont getPlatformFont() {
		return font;
	}

	public Shape getShape() {
		if (shape == null) {
			shape = createShape();
		}
		return shape;
	}

	public int getWidth() {
		return width;
	}

	private void init() throws PlatformFontException {
		synchronized (font) {
			// ensure correct glyph is active
			GlyphSlot slot = loadGlyphSlot();
			// get width
			long advance = slot.getLinearHoriAdvance();
			if (advance == 0) {
				// try harder
				advance = slot.getGlyphMetrics().getHoriAdvance();
			}
			// advance is integer expressed in pdf text space units
			// font units are encoded in "unitsPerEM"
			int glyphWidth = (int) (advance * getFont().getTextSpaceFactor());
			setWidth(glyphWidth);
		}
	}

	protected GlyphSlot loadGlyphSlot() throws PlatformFontException {
		return getFont().loadGlyphSlot(getGlyphs());
	}

	public void render(ICSPlatformDevice device) {
		Shape glyphShape = getShape();
		if (glyphShape == EMPTY) {
			// shape may be omitted, for example with whitespace
			return;
		}
		GraphicsState graphicsState = device.getGraphicsState();
		IGraphicsContext gc = device.getGraphicsContext();
		AffineTransform fontTransform = (AffineTransform) graphicsState
				.getAttribute(FreetypeFont.ATTR_TRANSFORM);
		TransformedShape tempShape = ApplyTransformationShape.create(
				glyphShape, fontTransform);
		int mode = graphicsState.textState.renderingMode & 3;
		if (mode == TextState.RENDERING_MODE_FILL) {
			gc.fill(tempShape);
		} else if (mode == TextState.RENDERING_MODE_STROKE) {
			gc.draw(tempShape);
		} else if (mode == TextState.RENDERING_MODE_FILL_STROKE) {
			gc.fill(tempShape);
			gc.draw(tempShape);
		} else {
			// no rendering
		}
		if ((graphicsState.textState.renderingMode & 4) == 4) {
			device.addTextClip(tempShape);
		}
	}

	public void setWidth(int width) {
		this.width = width;
	}
}
