/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import de.intarsys.cwt.font.FontStyle;
import de.intarsys.cwt.font.FontTools;
import de.intarsys.cwt.font.IFont;
import de.intarsys.pdf.encoding.Encoding;
import de.intarsys.pdf.encoding.WinAnsiEncoding;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.font.PDFontStyle;
import de.intarsys.pdf.font.PDFontType1;
import de.intarsys.pdf.font.outlet.FontFactoryException;
import de.intarsys.pdf.font.outlet.FontQuery;
import de.intarsys.pdf.font.outlet.IFontFactory;
import de.intarsys.pdf.font.outlet.IFontQuery;

/**
 * The standard {@link IFontFactory} managing and creating fonts.
 * <p>
 * The factory tries first to resolve the request within the cache of already
 * loaded fonts. If not found, a new {@link IFont} is looked up.
 */
public class StandardFontFactory implements IFontFactory {

	private static final String BUILTIN_SUFFIX_BOLD = "-Bold"; //$NON-NLS-1$

	private static final String BUILTIN_SUFFIX_BOLDOBLIQUE = "-BoldOblique"; //$NON-NLS-1$

	private static final String BUILTIN_SUFFIX_OBLIQUE = "-Oblique"; //$NON-NLS-1$

	/** The collection of fonts already used within the document */
	private final List cache;

	/** flag if new fonts should be embedded */
	private boolean embedNew = true;

	public StandardFontFactory() {
		super();
		this.cache = new ArrayList(10);
	}

	protected PDFont createBuiltinFontAlias(IFontQuery query) {
		String builtinName = createBuiltinName(query);
		if (!PDFontType1.isBuiltinAlias(builtinName)) {
			return null;
		}
		return PDFontType1.createNew(builtinName);
	}

	protected PDFont createBuiltinFontStandard(IFontQuery query) {
		String builtinName = createBuiltinName(query);
		if (!PDFontType1.isBuiltin(builtinName)) {
			return null;
		}
		return PDFontType1.createNew(builtinName);
	}

	protected String createBuiltinName(IFontQuery query) {
		String fontName = query.getFontName();
		if (fontName != null) {
			return fontName;
		}
		String family = query.getFontFamilyName();
		if (PDFontType1.FONT_Courier.equals(family)
				|| PDFontType1.FONT_Helvetica.equals(family)) {
			PDFontStyle style = query.getFontStyle();
			if (style == PDFontStyle.BOLD) {
				return family + BUILTIN_SUFFIX_BOLD;
			}
			if (style == PDFontStyle.ITALIC) {
				return family + BUILTIN_SUFFIX_OBLIQUE;
			}
			if (style == PDFontStyle.BOLD_ITALIC) {
				return family + BUILTIN_SUFFIX_BOLDOBLIQUE;
			}
		} else if (PDFontType1.FONT_Times_Roman.equals(family)) {
			PDFontStyle style = query.getFontStyle();
			if (style == PDFontStyle.BOLD) {
				return PDFontType1.FONT_Times_Bold;
			}
			if (style == PDFontStyle.ITALIC) {
				return PDFontType1.FONT_Times_Italic;
			}
			if (style == PDFontStyle.BOLD_ITALIC) {
				return PDFontType1.FONT_Times_BoldItalic;
			}
		}
		return family;
	}

	protected PDFont createExternalFont(IFontQuery query)
			throws FontFactoryException {
		IFont font;
		de.intarsys.cwt.font.FontQuery cwtQuery = new de.intarsys.cwt.font.FontQuery();
		cwtQuery.setFontType(query.getFontType());
		cwtQuery.setFontName(query.getFontName());
		cwtQuery.setFontFamilyName(query.getFontFamilyName());
		cwtQuery.setFontStyle(FontStyle.getFontStyle(query.getFontStyle()
				.getLabel()));
		font = FontTools.lookupFont(cwtQuery);
		if (font != null) {
			FontConverterCwt2Pdf converter = new FontConverterCwt2Pdf(font);
			converter.setEmbed(isEmbedNew());
			return converter.getPdFont();
		}
		return null;
	}

	public PDFont getBoldFlavor(PDFont font) throws FontFactoryException {
		PDFont result;
		PDFontStyle style = font.getLookupFontStyle().getBoldFlavor();
		FontQuery template = new FontQuery(font);
		template.setOverrideFontStyle(style);
		result = lookupFont(template);
		if (result == null) {
			result = loadFont(template);
		}
		return (result == null) ? font : result;
	}

	/**
	 * The collection of already loaded fonts.
	 * 
	 * @return The collection of already loaded fonts.
	 */
	protected List getCache() {
		return cache;
	}

	public PDFont getFont(IFontQuery query) throws FontFactoryException {
		PDFont result = lookupFont(query);
		if (result == null) {
			result = loadFont(query);
		}
		if (result == null) {
			throw new FontFactoryException("can't find font");
		}
		return result;
	}

	public PDFont getItalicFlavor(PDFont font) throws FontFactoryException {
		PDFont result = null;
		PDFontStyle style = font.getLookupFontStyle().getItalicFlavor();
		FontQuery template = new FontQuery(font);
		template.setOverrideFontStyle(style);
		result = lookupFont(template);
		if (result == null) {
			result = loadFont(template);
		}
		return (result == null) ? font : result;
	}

	public PDFont getRegularFlavor(PDFont font) throws FontFactoryException {
		PDFont result = null;
		PDFontStyle style = PDFontStyle.REGULAR;
		FontQuery template = new FontQuery(font);
		template.setOverrideFontStyle(style);
		result = lookupFont(template);
		if (result == null) {
			result = loadFont(template);
		}
		return (result == null) ? font : result;
	}

	public boolean isEmbedNew() {
		return embedNew;
	}

	/**
	 * <code>true</code> when <code>font</code> is considered safe to be used by
	 * others within the document.
	 * 
	 * @param font
	 * 
	 * @return
	 */
	protected boolean isReusable(PDFont font) {
		if (font.isStandardFont()) {
			return true;
		}
		if (font.isSubset()) {
			return false;
		}

		// check if there is some undefined code
		Encoding encoding = font.getEncoding();
		for (int i = font.getFirstChar(); i < font.getLastChar(); i++) {
			if (encoding.getDecoded(i) == -1) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Load a font satisfying the constraints defined in the query.
	 * 
	 * @param query
	 *            The object defining constraints on the result.
	 * 
	 * @return a font satisfying the constraints defined in in the query.
	 * @throws FontFactoryException
	 */
	protected PDFont loadFont(IFontQuery query) throws FontFactoryException {
		PDFont result = null;
		IFont font = null;
		if (IFontQuery.TYPE_BUILTIN.equals(query.getFontType())) {
			// we explicitly request a builtin font
			result = createBuiltinFontAlias(query);
		} else if (IFontQuery.TYPE_TYPE1.equals(query.getFontType())) {
			// lookup standard builtin type 1 fonts
			result = createBuiltinFontAlias(query);
			if (result == null) {
				result = createExternalFont(query);
			}
		} else if (IFontQuery.TYPE_TRUETYPE.equals(query.getFontType())) {
			result = createExternalFont(query);
		} else {
			result = createBuiltinFontStandard(query);
			if (result == null) {
				result = createExternalFont(query);
				if (result == null) {
					result = createBuiltinFontAlias(query);
				}
			}
		}
		if (result != null) {
			if (query.getEncoding() != null) {
				result.setEncoding(query.getEncoding());
			} else {
				if (result.getFontDescriptor().isNonsymbolic()) {
					result.setEncoding(WinAnsiEncoding.UNIQUE);
				}
			}
			if (isEmbedNew()) {
				try {
					// now look for the font program...
					PlatformFontTools.embedFontFile(result);
				} catch (IOException e) {
					throw new FontFactoryException(e);
				}
			}
			// attach query values to registered fonts to enable cache lookup
			if (query.getFontName() != null) {
				// force PDF font name
				if (result.getBaseFont() == null) {
					result.setBaseFont(query.getFontName());
					result.getFontDescriptor()
							.setFontName(result.getFontName());
				}
				result.setLookupFontName(query.getFontName());
			}
			if (query.getFontFamilyName() != null) {
				result.setLookupFontFamilyName(query.getFontFamilyName());
			}
			if (query.getFontStyle() != PDFontStyle.UNDEFINED) {
				result.setLookupFontStyle(query.getFontStyle());
			}
			registerFont(result);
		}
		return result;
	}

	/**
	 * Lookup a font that is compatible to the template.
	 * 
	 * @param query
	 *            The template defining properties of the font. A property that
	 *            is null is a "wildcard".
	 * 
	 * @return A font that satisfies the conditions defined int the template.
	 */
	protected PDFont lookupFont(IFontQuery query) {
		for (Iterator it = getCache().iterator(); it.hasNext();) {
			PDFont font = (PDFont) it.next();
			if (matches(query, font)) {
				return font;
			}
		}
		return null;
	}

	/**
	 * Answer <code>true</code> if font is "compatible" with query. Compatible
	 * means, that the font returned has at least all of the required features
	 * defined in the query.
	 * 
	 * @param font
	 *            The font to examine.
	 * 
	 * @return Answer <code>true</code> if font is "compatible" with template.
	 */
	protected boolean matches(IFontQuery query, PDFont font) {
		if (query.getFontType() != null && !"Any".equals(query.getFontType())) {
			if (!query.getFontType().equals(font.getFontType())) {
				return false;
			}
		}
		if (query.getFontName() != null) {
			if (!query.getFontName().equals(font.getLookupFontName())) {
				return false;
			}
		} else {
			if (query.getFontFamilyName() != null) {
				if (!query.getFontFamilyName().equals(
						font.getLookupFontFamilyName())) {
					return false;
				}
			}
			if (query.getFontStyle() != PDFontStyle.UNDEFINED) {
				if (font.getLookupFontStyle() != query.getFontStyle()) {
					return false;
				}
			}
		}
		if (query.getEncoding() != null) {
			if (font.getEncoding() != query.getEncoding()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * register a font that can be reused in an other context.
	 * 
	 * @param font
	 *            The PDFont to be registered
	 */
	public void registerFont(PDFont font) {
		getCache().add(font);
	}

	public void reset() {
		cache.clear();
	}

	public void setEmbedNew(boolean embedNew) {
		this.embedNew = embedNew;
	}

	protected FontStyle toFontStyle(PDFontStyle pdFontStyle) {
		return FontStyle.getFontStyle(pdFontStyle.getLabel());
	}
}
