/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import de.intarsys.cwt.font.truetype.CreateSubset;
import de.intarsys.cwt.font.truetype.TTFont;
import de.intarsys.cwt.font.truetype.TTFontSerializer;
import de.intarsys.cwt.font.truetype.TrueTypeException;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.font.PDFontDescriptor;
import de.intarsys.pdf.font.PDFontDescriptorEmbedded;
import de.intarsys.pdf.font.PDFontTrueType;
import de.intarsys.pdf.font.PDGlyphs;
import de.intarsys.tools.attribute.Attribute;
import de.intarsys.tools.locator.ILocator;
import de.intarsys.tools.randomaccess.RandomAccessByteArray;
import de.intarsys.tools.stream.StreamTools;

public class PDFontTrueTypeTools {

	private static final Attribute ATTR_CWTFONT = new Attribute("cwtfont");

	protected static TTFont basicSubset(PDFont pdFont, TTFont ttFont,
			Set<Integer> indices) throws IOException {
		CreateSubset tool = new CreateSubset(ttFont, indices);
		TTFont newFont;
		try {
			newFont = tool.compute();
		} catch (TrueTypeException e) {
			newFont = ttFont;
		}
		//
		RandomAccessByteArray random = new RandomAccessByteArray(null);
		TTFontSerializer serializer = new TTFontSerializer();
		serializer.write(random, newFont);
		pdFont.getFontDescriptor().setFontFile2(random.toByteArray());
		return newFont;
	}

	static protected void embedFontFile(PDFont font, ILocator locator)
			throws IOException {
		// compress();
		InputStream is = locator.getInputStream();
		try {
			byte[] data = StreamTools.toByteArray(is);
			((PDFontDescriptorEmbedded) font.getFontDescriptor())
					.setFontFile2(data);
		} finally {
			StreamTools.close(is);
		}
		// restrict font program to characters from 0-255
		subset(font);
	}

	public static TTFont getTTFont(PDFont pdFont) {
		TTFont ttFont = (TTFont) pdFont.getAttribute(ATTR_CWTFONT);
		if (ttFont != null) {
			return ttFont;
		}
		ILocator locator = PlatformFontTools.getFontFile(pdFont);
		if (locator == null) {
			return null;
		}
		try {
			ttFont = TTFont.createFromLocator(locator);
			setTTFont(pdFont, ttFont);
			return ttFont;
		} catch (IOException e) {
			return null;
		}
	}

	static protected Map getTTFontCMap(PDFont pdFont) throws TrueTypeException {
		TTFont ttFont = getTTFont(pdFont);
		Map result = null;
		PDFontDescriptor fontDescriptor = null;
		try {
			fontDescriptor = pdFont.getFontDescriptor();
		} catch (RuntimeException e) {
			// better support invalid documents around...
		}
		if (fontDescriptor == null) {
			result = ttFont.getCMapsAt(3, 1);
		} else {
			if (fontDescriptor.isSymbolic()) {
				result = ttFont.getCMapsAt(3, 0);
			} else {
				result = ttFont.getCMapsAt(3, 1);
			}
		}
		if (result == null) {
			throw new TrueTypeException("No suitable CMap found"); //$NON-NLS-1$
		}
		return result;
	}

	public static void setTTFont(PDFont pdFont, TTFont ttFont) {
		pdFont.setAttribute(ATTR_CWTFONT, ttFont);
	}

	/**
	 * Subset the font program for <code>pdFont</code> and restrict to the range
	 * of chars defined in the font meta information.
	 * 
	 * @param pdFont
	 * @throws IOException
	 * @throws TrueTypeException
	 */
	static public void subset(PDFont pdFont) throws IOException {
		if (!(pdFont instanceof PDFontTrueType)) {
			throw new IllegalArgumentException("subsetting "
					+ pdFont.getClass().getName() + " not supported");
		}
		TTFont ttFont = getTTFont(pdFont);
		Map cmap = null;
		try {
			cmap = getTTFontCMap(pdFont);
		} catch (TrueTypeException e) {
			return;
		}
		Set<Integer> indices = new HashSet<Integer>();
		int first = pdFont.getFirstChar();
		int last = pdFont.getLastChar();
		for (int i = first; i <= last; i++) {
			int decoded = pdFont.getEncoding().getDecoded(i);
			Integer index = (Integer) cmap.get(new Integer(decoded));
			if (index != null) {
				indices.add(index);
			}
		}
		TTFont newFont = basicSubset(pdFont, ttFont, indices);
		setTTFont(pdFont, newFont);
	}

	/**
	 * Subset the font program for <code>pdFont</code> and restrict to the
	 * glyphs in <code>used</code>
	 * 
	 * @param pdFont
	 * @param used
	 * @throws IOException
	 * @throws TrueTypeException
	 */
	static public void subset(PDFont pdFont, Set<PDGlyphs> used)
			throws IOException, TrueTypeException {
		if (!(pdFont instanceof PDFontTrueType)) {
			throw new IllegalArgumentException("subsetting "
					+ pdFont.getClass().getName() + " not supported");
		}
		TTFont ttFont = getTTFont(pdFont);
		Map cmap = null;
		try {
			cmap = getTTFontCMap(pdFont);
		} catch (TrueTypeException e) {
			return;
		}
		Set<Integer> indices = new HashSet<Integer>();
		for (Iterator it = used.iterator(); it.hasNext();) {
			PDGlyphs pdGlyphs = (PDGlyphs) it.next();
			int decoded = pdGlyphs.getDecoded();
			Integer index = (Integer) cmap.get(new Integer(decoded));
			if (index != null) {
				indices.add(index);
			}
		}
		TTFont newFont = basicSubset(pdFont, ttFont, indices);
		setTTFont(pdFont, newFont);
	}
}
