/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font;

import java.awt.geom.Rectangle2D;

import de.intarsys.cwt.font.IFont;
import de.intarsys.pdf.cds.CDSRectangle;
import de.intarsys.pdf.cos.COSArray;
import de.intarsys.pdf.cos.COSInteger;
import de.intarsys.pdf.encoding.SymbolEncoding;
import de.intarsys.pdf.encoding.WinAnsiEncoding;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.font.PDFontDescriptorEmbedded;
import de.intarsys.pdf.font.PDFontStyle;
import de.intarsys.pdf.font.PDGlyphs;
import de.intarsys.pdf.font.outlet.FontFactoryException;
import de.intarsys.tools.locator.ILocator;

/**
 * An abstract converter from a platform font program to a PDF font.
 */
abstract public class FontConverterPlatform2Pdf extends CommonFontConverter {

	private PDFont pdFont;

	private IPlatformFont pfFont;

	public FontConverterPlatform2Pdf(IFont font) {
		super(font);
	}

	abstract protected PDFont basicCreatePdFont() throws FontFactoryException;

	protected IPlatformFont basicCreatePfFont() throws FontFactoryException {
		try {
			return PlatformFontFactory.get().createPlatformFont(pdFont);
		} catch (PlatformFontException e) {
			throw new FontFactoryException(e);
		}
	}

	protected void createPdFont() throws FontFactoryException {
		pdFont = basicCreatePdFont();
		pdFont.setBaseFont(getCwtFont().getFontNamePostScript());
		//
		// provide meta information that otherwise might get lost on the way
		PlatformFontTools.setCWTFont(pdFont, getCwtFont());
		pdFont.setLookupFontName(getCwtFont().getFontName());
		pdFont.setLookupFontFamilyName(getCwtFont().getFontFamilyName());
		pdFont.setLookupFontStyle(PDFontStyle.getFontStyle(getCwtFont()
				.getFontStyle().getId()));
		//
		// provide an empty font descriptor
		PDFontDescriptorEmbedded result = (PDFontDescriptorEmbedded) PDFontDescriptorEmbedded.META
				.createNew();
		pdFont.setFontDescriptor(result);
		pfFont = basicCreatePfFont();
		// now derive all values from the information available
		fillPdFont();
		// if possible, we provide the font program, too
		if (isEmbed()) {
			embedFontFile(getCwtFont().getFontProgram().getLocator());
		}
	}

	protected void embedFontFile(ILocator locator) throws FontFactoryException {
		// do nothing
	}

	protected void fillFontDescriptor() throws FontFactoryException {
		PDFontDescriptorEmbedded result = (PDFontDescriptorEmbedded) getPdFont()
				.getFontDescriptor();
		result.setAscent(pfFont.getAscent());
		result.setDescent(pfFont.getDescent());
		result.setCapHeight(pfFont.getHeight());
		Rectangle2D rect = pfFont.getBBox();
		CDSRectangle cdsRect = new CDSRectangle(rect);
		result.setFontBB(cdsRect);
		result.setFontName(pdFont.getBaseFont().stringValue());
		result.getFlags().setSymbolic(pfFont.isSymbolFont());
		result.getFlags().setItalic(pfFont.isItalicStyle());
		result.getFlags().setForceBold(pfFont.isBoldStyle());
		result.getFlags().setFixedPitch(pfFont.isMonospaced());
		// result.setItalicAngle((float) italicAngle);
		// result.setStemV(80);
		// result.setFlags(0);
	}

	protected void fillPdFont() throws FontFactoryException {
		if (pfFont.isSymbolFont()) {
			pdFont.setEncoding(SymbolEncoding.UNIQUE);
		} else {
			pdFont.setEncoding(WinAnsiEncoding.UNIQUE);
		}
		fillFontDescriptor();
		fillWidths();
	}

	protected void fillWidths() throws FontFactoryException {
		int last = 255;
		int first = 0;
		try {
			COSArray widths = COSArray.create(last - first + 1);
			for (int i = first; i <= last; i++) {
				PDGlyphs pdglyphs = pdFont.getGlyphsEncoded(i);
				IPlatformGlyphs pfglyphs = pfFont
						.createPlatformGlyphs(pdglyphs);
				widths.add(COSInteger.create(pfglyphs.getWidth()));
			}
			pdFont.cosSetField(PDFont.DK_FirstChar, COSInteger.create(first));
			pdFont.cosSetField(PDFont.DK_LastChar, COSInteger.create(last));
			pdFont.cosSetField(PDFont.DK_Widths, widths);
		} catch (PlatformFontException e) {
			throw new FontFactoryException(e);
		}
	}

	@Override
	public PDFont getPdFont() throws FontFactoryException {
		if (pdFont == null) {
			createPdFont();
		}
		return pdFont;
	}
}
