/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.adapter;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import de.intarsys.cwt.environment.IGraphicsEnvironment;
import de.intarsys.tools.provider.ProviderTools;

/**
 * A common implementation for {@link IGraphicsEnvironmentAdapterRegistry}.
 * <p>
 * The providers of {@link IGraphicsEnvironmentAdapter} are looked up using the
 * standard Java mechanics for provider lookup and registered.
 */
public class StandardGraphicsEnvironmentAdapterRegistry implements
		IGraphicsEnvironmentAdapterRegistry {

	private boolean lookupProviders = true;

	private Map<IGraphicsEnvironment, IGraphicsEnvironmentAdapter> adapters = new HashMap<IGraphicsEnvironment, IGraphicsEnvironmentAdapter>();

	private boolean initialized = false;

	protected IGraphicsEnvironmentAdapter[] findProviders() {
		Iterator<IGraphicsEnvironmentAdapter> ps = ProviderTools
				.providers(IGraphicsEnvironmentAdapter.class);
		List<IGraphicsEnvironmentAdapter> adapters = new ArrayList<IGraphicsEnvironmentAdapter>();
		while (ps.hasNext()) {
			try {
				adapters.add(ps.next());
			} catch (Throwable e) {
				// ignore and try on
			}
		}
		return adapters
				.toArray(new IGraphicsEnvironmentAdapter[adapters.size()]);
	}

	protected void init() {
		if (!lookupProviders || initialized) {
			return;
		}
		initialized = true;
		IGraphicsEnvironmentAdapter[] adapters = findProviders();
		for (int i = 0; i < adapters.length; i++) {
			registerGraphicsEnvironmentAdapter(
					adapters[i].getGraphicsEnvironment(), adapters[i]);
		}
	}

	public boolean isLookupProviders() {
		return lookupProviders;
	}

	synchronized public IGraphicsEnvironmentAdapter lookupGraphicsEnvironmentAdapter(
			IGraphicsEnvironment environment) {
		init();
		IGraphicsEnvironmentAdapter adapter = adapters.get(environment);
		return adapter;
	}

	synchronized public void registerGraphicsEnvironmentAdapter(
			IGraphicsEnvironment environment,
			IGraphicsEnvironmentAdapter adapter) {
		adapters.put(environment, adapter);
	}

	public void setLookupProviders(boolean pLookupProviders) {
		lookupProviders = pLookupProviders;
	}
}
