/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi.ui.models;

import static org.nuiton.i18n.I18n._;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JOptionPane;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.navigation.NavigationNode;
import jaxx.runtime.swing.navigation.treetable.NavigationTreeTableNode;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;
import com.jurismarches.vradi.ui.ThesaurusHandler;
import com.jurismarches.vradi.ui.admin.content.AdminThesaurusUI;
import com.jurismarches.vradi.ui.helpers.ThesaurusTreeHelper;

/**
 * @author letellier
 */

public class ModifThesaurusModel extends ModifModel<ModifThesaurusModel.ModifThesaurus> {

    static String I18N_PATTERN = "vradi.thesaurus.attribute.";

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ModifThesaurusModel.class);

    public abstract class ModifThesaurus implements ModifModel.Modif {

        protected ModifThesaurus linked;

        public ModifThesaurus getLink(){
            return linked;
        }

        public void link(ModifThesaurus linked){
            this.linked = linked;
        }

        public boolean isLinked(){
            return linked != null;
        }

        public abstract VradiThesaurusDTO getConcernedThesaurus();

        public abstract boolean canImpactRequest();

    }

    public class CreateModif extends ModifThesaurus {

        VradiThesaurusDTO added;

        public CreateModif(VradiThesaurusDTO added){
            this.added = added;
        }

        @Override
        public String getMsg() {
            return _("vradi.thesaurus.addModifMsg", added.getName());
        }

        @Override
        public boolean revert() {
            // Get path
            String path = added.getPath(ThesaurusHandler.PREFIX_EDIT, helper.getPathSeparator());

            if (log.isDebugEnabled()){
                log.debug("Finding parent " + path + "to add " + added.getName());
            }
            // Find node
            NavigationTreeTableNode node = (NavigationTreeTableNode)helper.findNode(context, path);

            // Ask to revert linked
            if (!askToCancelModifLinked(this)){
                 return false;
            }

            // Remove it
            NavigationTreeTableNode resultNode = helper.getTreeTableBuilder().removeChildNode(node);

            added.setToDelete(true);

            return resultNode != null;
        }

        @Override
        public VradiThesaurusDTO getConcernedThesaurus(){
            return added;
        }

        @Override
        public boolean canImpactRequest() {
            return false;
        }

    }

    public class DeleteModif extends ModifThesaurus{

        VradiThesaurusDTO deleted;

        public DeleteModif(VradiThesaurusDTO deleted){
            this.deleted = deleted;
        }

        @Override
        public String getMsg() {
            return _("vradi.thesaurus.deleteModifMsg", deleted.getName());
        }

        @Override
        public boolean revert() {
            String parentPath = deleted.getParentThesaurus().getPath(ThesaurusHandler.PREFIX_EDIT, helper.getPathSeparator());
            if (log.isDebugEnabled()){
                log.debug("Finding parent " + parentPath + "to add " + deleted.getName());
            }
            NavigationTreeTableNode parentNode = (NavigationTreeTableNode) helper.findNode(context, parentPath);
            if (parentNode == null){
                return false;
            }

            // Ask to revert linked
            if (!askToCancelModifLinked(this)){
                 return false;
            }

            // Add it
            NavigationTreeTableNode navigationTreeNode = helper.getTreeTableBuilder().addThesaurus(context, parentNode, deleted);

            deleted.setToDelete(false);

            helper.selectNode(context, navigationTreeNode);

            return true;
        }

        @Override
        public VradiThesaurusDTO getConcernedThesaurus(){
            return deleted;
        }

        @Override
        public boolean canImpactRequest() {
            return true;
        }
    }

    public class MoveModif extends ModifThesaurus{

        VradiThesaurusDTO moved;

        String oldParentNamePath;
        String oldParentPath;
        String newParentPath;

        public MoveModif(VradiThesaurusDTO moved, String oldParentNamePath, String oldParentPath, String newParentPath){
            this.moved = moved;
            this.oldParentNamePath = oldParentNamePath;
            this.oldParentPath = oldParentPath;
            this.newParentPath = newParentPath;
        }

        @Override
        public String getMsg() {
            return _("vradi.thesaurus.moveModifMsg", moved.getName(),
                    oldParentNamePath, moved.getNamePath(helper.getPathSeparator()));
        }

        @Override
        public boolean revert() {
            if (log.isDebugEnabled()){
                log.debug("Finding old path " + oldParentNamePath);
            }
            NavigationTreeTableNode parentNode = (NavigationTreeTableNode) helper.findNode(context, oldParentPath);
            String path = moved.getPath(ThesaurusHandler.PREFIX_EDIT, helper.getPathSeparator());
            NavigationTreeTableNode movedNode = (NavigationTreeTableNode) helper.findNode(context, path);

            if (parentNode == null || movedNode == null){
                if (log.isDebugEnabled()){
                    if (parentNode == null){
                        log.debug("Cant find parent node " + oldParentNamePath);
                    }
                    if (movedNode == null){
                        log.debug("Cant find moved node " + path);
                    }
                }
                return false;
            }

            // Ask to revert linked
            if (!askToCancelModifLinked(this)){
                 return false;
            }

            // Remove in tree
            helper.getModel(context).removeNodeFromParent(movedNode);

            // Remove from parent
            VradiThesaurusDTO moved = (VradiThesaurusDTO)movedNode.getBean(context);
            VradiThesaurusDTO oldParent = moved.getParentThesaurus();
            oldParent.removeChild(moved);

            // Attach child
            VradiThesaurusDTO newParent = (VradiThesaurusDTO)parentNode.getBean(context);
            moved.setParentThesaurus(newParent);
            newParent.addChild(moved);

            // Mark to save
            moved.setToSave(true);

            // Add in tree old path
            if (log.isDebugEnabled()){
                VradiThesaurusDTO parent = (VradiThesaurusDTO)parentNode.getBean(context);
                log.debug("Add thesaurus " + moved.getNamePath(helper.getPathSeparator()) + " to parent " + parent.getNamePath(helper.getPathSeparator()));
            }
            NavigationNode newNode = helper.getTreeTableBuilder().addThesaurusAndChildrenRecursivly(parentNode, moved);

            // Expend
            helper.selectNode(context, parentNode);

            return true;
        }

        @Override
        public VradiThesaurusDTO getConcernedThesaurus(){
            return moved;
        }

        @Override
        public boolean canImpactRequest() {
            return true;
        }
    }

    public class AttributeModif extends ModifThesaurus {

        protected VradiThesaurusDTO modif;

        protected String attributeName;

        protected String oldValue;

        protected String newValue;

        public AttributeModif(VradiThesaurusDTO modif, String attributeName, String oldValue){
            this.modif = modif;
            this.attributeName = attributeName;
            this.oldValue = oldValue;
            try {
                // Get new value
                newValue = BeanUtils.getProperty(modif, attributeName);
            } catch (Exception e) {
                log.error("Cant get new value for property " + attributeName + " of " + modif.getName(), e);
            }
        }

        protected String getOldValue(){
            return oldValue;
        }

        @Override
        public String getMsg() {
            return _("vradi.thesaurus.attributeModifMsg", modif.getName(), _(I18N_PATTERN + attributeName),
                    oldValue, newValue);
        }

        @Override
        public boolean revert() {
            try {
                // Set old value
                BeanUtils.setProperty(modif, attributeName, oldValue);
            } catch (Exception e) {
                log.error("Cant revert for property " + attributeName + " of " + modif.getName(), e);
                return false;
            }
            String path = modif.getPath(ThesaurusHandler.PREFIX_EDIT, helper.getPathSeparator());
            NavigationNode<?> modifNode = helper.findNode(context, path);

            if (log.isDebugEnabled()){
                log.debug("Repaint node : " + modifNode.getFullPath());
            }
            helper.repaintNode(context, modifNode);
            helper.selectNode(context, modifNode);

            return true;
        }

        @Override
        public VradiThesaurusDTO getConcernedThesaurus(){
            return modif;
        }

        @Override
        public boolean canImpactRequest() {
            return attributeName.equals("name");
        }
    }

    protected JAXXContext context;

    protected ThesaurusTreeHelper helper;

    protected Map<String, AttributeModif> cachedAttributeModif = new HashMap<String, AttributeModif>();

    protected LinkedHashMap<String, ModifThesaurus> cachedModif = new LinkedHashMap<String, ModifThesaurus>();

    public ModifThesaurusModel(JAXXContext context, ThesaurusTreeHelper helper){
        this.context = context;
        this.helper = helper;
    }

    public CreateModif addCreateModif(VradiThesaurusDTO created){
        CreateModif createdModif = new CreateModif(created);
        add(createdModif);
        findLinkedModif(createdModif);
        cachedModif.put(created.getPath(helper.getPathSeparator()), createdModif);
        return createdModif;
    }

    public DeleteModif addDeleteModif(VradiThesaurusDTO deleted){
        DeleteModif deleteModif = new DeleteModif(deleted);
        add(deleteModif);
        DeleteModif lastDeleteModif = deleteModif;
        for (VradiThesaurusDTO child : deleted.getChildren()){
           lastDeleteModif = addDeleteModif(child);
           lastDeleteModif.link(deleteModif);
        }
        cachedModif.put(lastDeleteModif.getConcernedThesaurus().getPath(helper.getPathSeparator()), lastDeleteModif);
        return deleteModif;
    }

    public MoveModif addMoveModif(VradiThesaurusDTO moved, String parentNamePath, String parentPath, String newPath){
        MoveModif moveModif = new MoveModif(moved, parentNamePath, parentPath, newPath);
        add(moveModif);
        findLinkedModif(moveModif);
        cachedModif.put(newPath, moveModif);
        return moveModif;
    }

    protected void findLinkedModif(ModifThesaurus toCheck) {
        // Check linked modif
        List<String> keyList = new ArrayList<String>(cachedModif.keySet());
        Collections.reverse(keyList);
        for (String oldParentPath : keyList){
//            for (String parentId : oldParentPath.split(helpers.getPathSeparator())){
//                if (log.isDebugEnabled()){
//                    log.debug("Compare " + parentId + " with " +
//                            toCheck.getConcernedThesaurus().getWikittyId() +
//                            " for modif " + toCheck.getMsg());
//                }
//
//                if (toCheck.getConcernedThesaurus().getWikittyId().equals(parentId)){
//                    ModifThesaurus toLink = cachedModif.get(oldParentPath);
//                    if (log.isDebugEnabled()){
//                        log.debug("Link " + toCheck.getMsg() + " with " +
//                                toLink.getMsg());
//                    }
//
//                    toCheck.link(toLink);
//                }
//            }
        }
    }

    public String getAllMsgs(){
        StringBuffer buf = new StringBuffer();
        for (Modif m : values){
            buf.append(m.getMsg()).append("\n");
        }
        return buf.toString();
    }

    protected boolean askToCancelModifLinked(ModifThesaurus modif) {

        if (!modif.isLinked()){
            return true;
        }

        ModifThesaurus linked = modif.getLink();

        int result = JOptionPane.CANCEL_OPTION;
        result = JOptionPane.showConfirmDialog((AdminThesaurusUI)context,
                _("vradi.thesaurus.askRevertModifThesaurus", linked.getMsg()));

        if (result == JOptionPane.NO_OPTION || result == JOptionPane.CANCEL_OPTION) {
            return false;
        }

        // Impact parent modif
        if (!linked.revert()){
            return false;
        }
        remove(linked);

        return true;
    }

    public AttributeModif prepareAttributeModif(VradiThesaurusDTO modif, String attributeName, String oldValue) {
        // Get cached modif
        AttributeModif oldModif = cachedAttributeModif.get(attributeName);

        if (oldModif != null){
            // Get the oldre value
            oldValue = oldModif.getOldValue();
        }
        // Add in cache
        cachedAttributeModif.put(attributeName, new AttributeModif(modif, attributeName, oldValue));
        return oldModif;
    }

    public void confirmAttributeModif(){
        addAll(cachedAttributeModif.values());
        cachedAttributeModif.clear();
    }

    public void cancelAttributeModif(){
        cachedAttributeModif.clear();
    }

    protected VradiThesaurusDTO getConcernedThesaurus(int index){
        ModifThesaurus value = get(index);
        return value.getConcernedThesaurus();
    }

    public List<VradiThesaurusDTO> getConcernedThesaurus(int[] index){
        List<VradiThesaurusDTO> result = new ArrayList<VradiThesaurusDTO>();
        for (int i : index){
            result.add(getConcernedThesaurus(i));
        }
        return result;
    }
}
