/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.helpers;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.*;

import javax.swing.*;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.ui.VradiComparators;
import com.jurismarches.vradi.ui.models.ThesaurusTreeTableModel;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.decorator.Decorator;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.navigation.*;

import jaxx.runtime.swing.navigation.handler.NavigationHandler;
import jaxx.runtime.swing.navigation.handler.NavigationOneClicSelectionHandler;
import jaxx.runtime.swing.navigation.tree.NavigationTreeModel;
import jaxx.runtime.swing.navigation.tree.NavigationTreeModelBuilder;
import jaxx.runtime.swing.navigation.tree.NavigationTreeNode;
import jaxx.runtime.swing.navigation.treetable.NavigationTreeTableModel;
import jaxx.runtime.swing.navigation.treetable.NavigationTreeTableModelBuilder;
import jaxx.runtime.swing.navigation.treetable.NavigationTreeTableNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTreeTable;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;
import com.jurismarches.vradi.ui.ThesaurusHandler;
import com.jurismarches.vradi.ui.ThesaurusUI;
import com.jurismarches.vradi.ui.admin.ThesaurusPathChooserUI;
import com.jurismarches.vradi.ui.admin.content.AdminThesaurusUI;
import com.jurismarches.vradi.ui.renderer.ThesaurusHighlighter;

/**
 * @author letellier
 */
public class ThesaurusTreeHelper extends AbstractNavigationHelper {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ThesaurusTreeHelper.class);

    public static String PATH_SEPARATOR = "/";

    private static final String THESAURUS_NAME = "vradi.thesaurus.name";
    private static final String THESAURUS_NB_FORMS = "vradi.thesaurus.nbforms";
    private static final String THESAURUS_TAGS = "vradi.thesaurus.tags";

    protected ThesaurusModelBuilder builder;

    protected VradiThesaurusDTO rootThesaurus;

    public JAXXContext context;

    protected String refName;

    protected JAXXContextEntryDef<VradiThesaurusDTO> THESAURUS;
    protected List<String> columnsName;

    protected NavigationModel<?> cachedModel;

    // Current listener
    protected TreeSelectionListener listener;

    // Selection mode
    protected int selectionMode = ListSelectionModel.SINGLE_SELECTION;

    // Use default ref and default root thesaurus
    public ThesaurusTreeHelper(JAXXContext context, int selectionMode) {
        this(context, null, null, selectionMode);
    }

    // Use new ref with default root thesaurus
    public ThesaurusTreeHelper(JAXXContext context, String refName, int selectionMode) {
        this(context, null, refName, selectionMode);
    }

    // Use default ref with custom root thesaurus
    public ThesaurusTreeHelper(JAXXContext context,  VradiThesaurusDTO rootThesaurus, int selectionMode) {
        this(context, rootThesaurus, null, selectionMode);
    }

    // Use new ref (if refName is not null) and custom root thesaurus
    public ThesaurusTreeHelper(JAXXContext context, VradiThesaurusDTO rootThesaurus, String refName, int selectionMode) {
        super(refName == null ? ThesaurusHandler.PREFIX_THESAURUS + rootThesaurus.getWikittyId() : refName);
        this.refName = refName;
        this.context = context;
        this.selectionMode = selectionMode;

        columnsName = new ArrayList<String>();

        // If is to attach thesaurus to form
        if (isCopy()){
            columnsName.add(_(THESAURUS_NAME));
            columnsName.add(_(THESAURUS_NB_FORMS));
            columnsName.add(_(THESAURUS_TAGS));
        } else {
            columnsName.add("");
        }

        if (rootThesaurus != null){
            this.rootThesaurus = rootThesaurus;
        } else if (isCopy()){
            this.rootThesaurus = getRootRefThesaurus(context).clone();
        } else {
            this.rootThesaurus = getRootRefThesaurus(context);
        }

        // Create ENTRY def with unique name
        if (isCopy()){

            // Create custom referenciel
            THESAURUS = SwingUtil.newContextEntryDef(refName, VradiThesaurusDTO.class);

            // Create the model and store it in the given context.
            THESAURUS.setContextValue(VradiContext.get(), this.rootThesaurus);
        }
    }

    protected void reloadTreeTable() {
        JXTreeTable treeTable = getTreeTable(context);

        NavigationNode selected = getSelectedNode(context);

        // Get selected
        String selectedPath = null;
        if (selected != null){
            VradiThesaurusDTO bean = (VradiThesaurusDTO)selected.getBean();

            selectedPath = bean.getPath(getPathSeparator());
            if (log.isDebugEnabled()){
                log.debug("Searching node whith path : " + selectedPath);
            }
        }

        // Reload model
        treeTable.setTreeTableModel((NavigationTreeTableModel)reloadModel());

        // Select
        if (selectedPath != null){
            NavigationTreeTableNode toSelect = (NavigationTreeTableNode) findNode(context, selectedPath);
            selectNode(context, toSelect);
        }
    }

    protected NavigationModel reloadModel() {

        if (isCopy()){
            builder = new ThesaurusTreeTableModelBuilder(context);
        } else {
            builder = new ThesaurusTreeModelBuilder(context);
        }

        // construction du noeud root
        // il ne contient pas de context et ne sera pas visible
        NavigationNode rootNode = builder.buildRoot();

        builder.buildChildren(context, rootNode);

        cachedModel = builder.getModel();

        // save tree model in context
        setModel(context, cachedModel);

        if (!isCopy()){
            Map<String, NavigationModel> map = VradiContext.THESAURUS_NAVIGATION_MODEL.getContextValue(VradiContext.get());

            // Store in vradi cache
            map.put(rootThesaurus.getWikittyId(), cachedModel);
            VradiContext.THESAURUS_NAVIGATION_MODEL.setContextValue(VradiContext.get(), map);
        }

        return cachedModel;
    }

    // If this helpers use an other referenciel
    protected boolean isCopy(){
        return refName != null;
    }

    public void registerListener(TreeSelectionListener listener){
        this.listener = listener;
    }

    public TreeSelectionListener getListener(){
        return listener;
    }

    public void desactiveListener(){
        getTree(context).removeTreeSelectionListener(listener);
    }

    public void activeListener(){
        getTree(context).addTreeSelectionListener(listener);
    }

    // Get concerned ref
    protected JAXXContextEntryDef<VradiThesaurusDTO> getThesaurusRef() {
        return getThesaurusRef(isCopy());
    }

    /**
     * Get root thesaurus from VradiContext
     *
     * @param context
     * @return vradiThesaurusDTO from VradiContext
     */
    protected VradiThesaurusDTO getRootRefThesaurus(JAXXContext context){
        return getThesaurusRef(false).getContextValue(context);
    }

    /**
     * Get thesaurus ref
     *
     * @param copy
     * @return entryDef
     */
    protected JAXXContextEntryDef<VradiThesaurusDTO> getThesaurusRef(boolean copy) {
        if (copy){
             return THESAURUS;
        }
        return VradiContext.getThesaurusEntryDef();
    }

    // Get thesaurus handler
    protected ThesaurusHandler getHandler(JAXXContext context) {
        return UIHelper.getHandler(context, ThesaurusHandler.class);
    }

    // Get the builder
    public ThesaurusModelBuilder<?> getBuilder() {
        return builder;
    }

    public ThesaurusModelBuilder<NavigationTreeTableNode> getTreeTableBuilder() {
        return (ThesaurusModelBuilder<NavigationTreeTableNode>)builder;
    }

    // Get this root thesaurus
    public VradiThesaurusDTO getRootThesaurus() {
        return rootThesaurus;
    }

    // Get this root node
    public NavigationNode getRootNode() {
        return cachedModel.getRoot();
    }

    // Get this root name
    public String getRootName() {
        return rootThesaurus.getName();
    }

    // Get this root name path (like : Thesaurus/child)
    public String getRootNamePath() {
        return rootThesaurus.getNamePath(getPathSeparator());
    }

    public String getPathSeparator(){
        return PATH_SEPARATOR;
    }

    public void reinitColors() {
        reinitColors(getHandler(context).getUI(context));
    }

    public void reinitColors(ThesaurusUI ui) {
        // Apply Highlighter
        JXTreeTable treeTable = getTreeTable(context);
        if (treeTable != null){
            treeTable.setHighlighters(new ThesaurusHighlighter());
        } else {
            ((ThesaurusHighlighter)getTree(context).getCellRenderer()).updateColors();

            // Apply on list of selected
            JList thesaurusSelected = ui.getThesaurusSelected();

            ListCellRenderer selectedRenderer = thesaurusSelected.getCellRenderer();
            if (!(selectedRenderer instanceof ThesaurusHighlighter)){
                selectedRenderer = new ThesaurusHighlighter();
                thesaurusSelected.setCellRenderer(selectedRenderer);
            }
            ((ThesaurusHighlighter) selectedRenderer).updateColors();

            // Apply in list of proposition
            JList propositionList = ui.getParentUI().getPropositionList();
            ListCellRenderer propositionRenderer = propositionList.getCellRenderer();
            if (!(propositionRenderer instanceof ThesaurusHighlighter)){
                propositionRenderer = new ThesaurusHighlighter(true);
                thesaurusSelected.setCellRenderer(propositionRenderer);
            }
            ((ThesaurusHighlighter)propositionRenderer).updateColors();
        }
    }

    @Override
    public NavigationModel createTreeModel(JAXXContext context) {
        
        // Return cached model
        if (cachedModel != null){
            setModel(context, cachedModel);
            return cachedModel;
        }

        // Search model in vradi cache
        Map<String, NavigationModel> map = null;
        if (!isCopy()){
            map = VradiContext.THESAURUS_NAVIGATION_MODEL.getContextValue(VradiContext.get());

            // Init Cache
            if (map == null){
                map = new HashMap<String, NavigationModel>();
                VradiContext.THESAURUS_NAVIGATION_MODEL.setContextValue(VradiContext.get(), map);
            }

            // Search cached model
            cachedModel = map.get(rootThesaurus.getWikittyId());
            if (cachedModel != null){
                setModel(context, cachedModel);
                return cachedModel;
            }
        }                   

        this.context = context;
        return reloadModel();
    }

    @Override
    public NavigationHandler<?> createTreeHandler(JAXXObject context) {

        if (log.isDebugEnabled()) {
            log.debug("create handler");
        }

        // save tree in context
        // TODO : remove it
        if (context instanceof ThesaurusUI) {
            setTree(context, ((ThesaurusUI) context).getThesaurus());
        } else if (context instanceof AdminThesaurusUI) {
            setTreeTable(context, ((AdminThesaurusUI) context).getThesaurus());
        } else {
            setTreeTable(context, ((ThesaurusPathChooserUI) context).getThesaurus());
        }

        if (log.isDebugEnabled()){
            log.debug("Using default thesaurus handler");
        }

        NavigationOneClicSelectionHandler<?> handler = new NavigationOneClicSelectionHandler(
                getPrefix(),
                context,
                NavigationHandler.Strategy.PER_NODE) {

            private static final long serialVersionUID = 1L;

            @Override
            public NavigationModel<?> getNavigationTreeModel() {
                NavigationModel<?> model = getSafeModel(getContext());
                return model;
            }

            @Override
            public void treateError(Exception e) {
                ErrorDialogUI.showError(e);
            }

            @Override
            public JAXXContext getContext() {
                return this.context;
            }

            @Override
            public Component getCurrentUI() {
                return null;
            }

            @Override
            public Component getUI(NavigationNode node) {
                return (Component) context;
            }

            @Override
            public boolean closeUI(Component component) throws Exception {
                return true;
            }

            @Override
            public Component createUI(NavigationNode node) throws Exception {
                return getHandler(context).getUI(context);
            }

            @Override
            public void openUI(Component newUI, NavigationNode node) throws Exception {
            }

            @Override
            public void openUI(Component newUI, List nodes) throws Exception {
            }

            @Override
            public Component createUI(List nodes) throws Exception {
                return getHandler(context).getUI(context);
            }

            @Override
            public Component getUI(List  nodes) {
                return getHandler(context).getUI(context);
            }
        };
        JXTreeTable treeTable = getTreeTable(context);
        if (treeTable != null){
            treeTable.setSelectionMode(selectionMode);
        } else {
            JTree tree = getTree(context);
            tree.getSelectionModel().setSelectionMode(selectionMode);
        }

        // save handler in ui context
        setTreeHandler(context, handler);

        reinitColors();

        return handler;
    }

    public TreePath expendNode(NavigationNode<?> node) {

        NavigationNode<?>[] treeNodes = node.getPath();
        TreePath path = new TreePath(treeNodes);
        JXTreeTable treeTable = getTreeTable(context);

        if (log.isDebugEnabled()){
            log.debug("Try to expend path : " + path);
        }

        if (treeTable != null){
            treeTable.expandPath(path);
        } else {
            getTree(context).expandPath(path);
        }

        return path;
    }

    public void tryToUnselect(List<Object> beans) {
        tryToUnselect(extractIds(beans));
    }

    public void tryToUnselect(Collection<String> thesaurusIds) {
        if (thesaurusIds != null) {
            List<NavigationNode<?>> nodes = new ArrayList<NavigationNode<?>>();

            // Find recursivly
            for (String id : thesaurusIds) {
                findNode(id, getRootNode(), nodes);
            }
            
            // Unselect all
            TreePath[] paths = new TreePath[nodes.size()];
            int i = 0;
            for (NavigationNode<?> node : nodes) {

                NavigationNode<?>[] treeNodes = node.getPath();
                TreePath path = new TreePath(treeNodes);

                if (log.isDebugEnabled()) {
                    log.debug("Try to unselect path : " + path);
                }

                paths[i++] = path;
                if (log.isDebugEnabled()) {
                    log.debug("path " + paths[i - 1] + " for node " + node);
                }
                getTree(context).removeSelectionPath(path);

                if (log.isDebugEnabled()){
                    TreePath[] treePaths = getTree(context).getSelectionPaths();
                    log.debug((treePaths == null ? 0 : treePaths.length) +
                              " path unselected : " + Arrays.toString(treePaths));
                }
            }
            if (getSelectedNodes(context) != null){
                getSelectedNodes(context).removeAll(nodes);
                getSelectedPaths(context).removeAll(extractPaths(nodes));
                getSelectedBeans(context).removeAll(extractBeans(nodes));
            }
        }
    }

    public List<String> extractIds(List<Object> beans){
        List<String> ids = new ArrayList<String>();
        if (beans != null){
            for (Object bean : beans){
                ids.add(((VradiThesaurusDTO)bean).getWikittyId());
            }
        }
        return ids;
    }

    public void tryToSelect(List<Object> beans) {
        tryToSelect(extractIds(beans));
    }

    public void tryToSelect(Collection<String> thesaurusIds) {
        if (thesaurusIds != null) {
            List<NavigationNode<?>> nodes = new ArrayList<NavigationNode<?>>();

            // Find recursivly
            for (String id : thesaurusIds) {
                findNode(id, getRootNode(), nodes);
            }

            // Select all
            TreePath[] paths = new TreePath[nodes.size()];
            int i = 0;
            for (NavigationNode<?> node : nodes) {

                // Expend
//                TreePath path = expendNode(node);

                NavigationNode<?>[] treeNodes = node.getPath();
                TreePath path = new TreePath(treeNodes);

                if (log.isDebugEnabled()) {
                    log.debug("Try to select path : " + path);
                }

                paths[i++] = path;
                if (log.isDebugEnabled()) {
                    log.debug("path " + paths[i - 1] + " for node " + node);
                }
                getTree(context).addSelectionPath(path);

                if (log.isDebugEnabled()){
                    TreePath[] treePaths = getTree(context).getSelectionPaths();
                    log.debug((treePaths == null ? 0 : treePaths.length) +
                              " path selected : " + Arrays.toString(treePaths));
                }
            }
            List<NavigationNode<?>> selectedNodes = getSelectedNodes(context);
            if (selectedNodes != null){
                selectedNodes.addAll(nodes);
                List selectedPaths = getSelectedPaths(context);
                selectedPaths.addAll(extractPaths(nodes));
                List list = getSelectedBeans(context);
                list.addAll(extractBeans(nodes));
            } else {
                setSelectedNodes(context, nodes);
                setSelectedPaths(context, extractPaths(nodes));
                setSelectedBeans(context, extractBeans(nodes));
            }
        }
    }

    protected TreeSelectionModel getSelectionModel() {
        // Get model
        JXTreeTable treeTable = getTreeTable(context);
        TreeSelectionModel selectionModel;
        if (treeTable == null){
            selectionModel = getTree(context).getSelectionModel();
        } else {
            selectionModel = treeTable.getTreeSelectionModel();
        }
        return selectionModel;

    }

    protected List<String> extractPaths(List<NavigationNode<?>> nodes) {
        List<String> paths = new ArrayList<String>();
        for (NavigationNode<?> node : nodes){
            paths.add(node.getFullPath());
        }
        return paths;
    }

    protected List<Object> extractBeans(List<NavigationNode<?>> nodes) {
        List<Object> beans = new ArrayList<Object>();
        for (NavigationNode<?> node : nodes){
            beans.add(node.getBean());
        }
        return beans;
    }

    public void findNode(String thesaurusId, NavigationNode<?> node, List<NavigationNode<?>> nodes) {
        if (thesaurusId != null) {
            if (thesaurusId.equals(getWikittyId(node))) {
                // Add to select list
                nodes.add(node);

            } else {
                // Find in child
                Enumeration<? extends NavigationNode<?>> children = node.children();
                while (children.hasMoreElements()) {
                    NavigationNode<?> child = children.nextElement();
                    findNode(thesaurusId, child, nodes);
                }
            }
        }
    }

    protected String getWikittyId(NavigationNode<?> node) {
        VradiThesaurusDTO thesaurus = (VradiThesaurusDTO) node.getBean();
        if (thesaurus == null) {
            return null;
        }
        return thesaurus.getWikittyId();
    }

    public interface ThesaurusModelBuilder<E extends NavigationNode<E>> extends NavigationModelBuilder<E>{

        E addThesaurusToSelected(JAXXContext context, VradiThesaurusDTO thesaurus) ;

        E addThesaurus(JAXXContext context, E parent, VradiThesaurusDTO thesaurus) ;

        E addThesaurusAndChildrenRecursivly(E parent, VradiThesaurusDTO thesaurus) ;

        E addThesaurus(E parent, String thesaurusId);

        // Build the root node to display
        E buildRoot() ;

        void buildChildren(JAXXContext context, E parentNode);

        NavigationModel<E> getModel();
    }

    public class ThesaurusTreeTableModelBuilder extends NavigationTreeTableModelBuilder implements ThesaurusModelBuilder<NavigationTreeTableNode>{
        public ThesaurusTreeTableModelBuilder(JAXXContext context) {
            super(null, null,
                    new NavigationTreeTableModel(new ThesaurusTreeTableModel(), PATH_SEPARATOR, context, columnsName));
        }

        // Return true if empty
        public boolean removeNodeFromParent(JAXXContext context, NavigationTreeTableNode node) {
            NavigationModel<NavigationTreeTableNode> treeModel = getModel();
            treeModel.removeNodeFromParent(node);
            return treeModel.getRoot().getChildCount() == 0;
        }

        public NavigationTreeTableNode addThesaurusToSelected(JAXXContext context, VradiThesaurusDTO thesaurus) {
            NavigationTreeTableNode selectedNode = (NavigationTreeTableNode) getSelectedNode(context);

            return addThesaurus(context, selectedNode, thesaurus);
        }

        public NavigationTreeTableNode addThesaurus(JAXXContext context, NavigationTreeTableNode parent, VradiThesaurusDTO thesaurus) {
            return addThesaurus(parent, thesaurus.getWikittyId());
        }

        public NavigationTreeTableNode addThesaurusAndChildrenRecursivly(NavigationTreeTableNode parent, VradiThesaurusDTO thesaurus) {
            NavigationTreeTableNode newNode = addThesaurus(parent, thesaurus.getWikittyId());
            for (VradiThesaurusDTO child : thesaurus.getChildren()){
                addThesaurusAndChildrenRecursivly(newNode, child);
            }
            return newNode;
        }

        public NavigationTreeTableNode addThesaurus(NavigationTreeTableNode parent, String thesaurusId) {
            NavigationTreeTableNode result;
            Decorator<VradiThesaurusDTO> tDecorator = getThesaurusDecorator();
            String path;
            if (parent == null) {
                parent = (NavigationTreeTableNode) getRootNode();
                path = "..children[@wikittyId=\"" + thesaurusId + "\"]";
            } else {
                path = ".."  + PATH_SEPARATOR + "children[@wikittyId=\"" + thesaurusId + "\"]";
            }
            result = getTreeTableBuilder().build(parent, tDecorator, path, thesaurusId, null, null);

            if (log.isDebugEnabled()) {
                log.debug("addThesaurus : " + thesaurusId + " to parent : " +
                        (parent == null ? "null" : parent.getFullPath()) + " fullPath : " +
                        (result == null ? "result is null" : result.getFullPath()));
            }
            sort(context, result);
            return result;
        }

        // Build the root node to display
        public NavigationTreeTableNode buildRoot() {
            if (log.isDebugEnabled()){
                log.debug("Building root : " + getRootName());
            }
            NavigationTreeTableNode invisibleRootNode = getTreeTableBuilder().buildEmptyRoot(getThesaurusRef(), getThesaurusRef().getName());

            // When it's the root one
            if (rootThesaurus.getParentThesaurus() == null){
                return invisibleRootNode;
            }

            // Add root node
            return getTreeTableBuilder().build(invisibleRootNode, getThesaurusDecorator(), rootThesaurus.getBuildPath(PATH_SEPARATOR), rootThesaurus.getWikittyId(), null, null);
        }

        public void buildChildren(JAXXContext context, NavigationTreeTableNode parentNode) {
            if (log.isDebugEnabled()) {
                log.debug("buildChildren of thesaurus " + parentNode.getFullPath());
            }

            VradiThesaurusDTO parent = (VradiThesaurusDTO) parentNode.getBean(context);
            if (parent != null){
                for (VradiThesaurusDTO child : parent.getChildren()) {
                    NavigationTreeTableNode childNode = addThesaurus(context, parentNode, child);

                    // Build childs
                    if (child.getWikittyId() != null) {
                        buildChildren(context, childNode);
                    }
                }
            }
        }
    }

    public NavigationTreeModelBuilder getTreeBuilder() {
        return (NavigationTreeModelBuilder) builder;
    }

    public class ThesaurusTreeModelBuilder extends NavigationTreeModelBuilder implements ThesaurusModelBuilder<NavigationTreeNode>{
        public ThesaurusTreeModelBuilder(JAXXContext context) {
            super(null, null,
                    new NavigationTreeModel(PATH_SEPARATOR, context));
        }

        // Return true if empty
        public boolean removeNodeFromParent(JAXXContext context, NavigationTreeNode node) {
            NavigationModel<NavigationTreeNode> treeModel = getModel();
            treeModel.removeNodeFromParent(node);
            return treeModel.getRoot().getChildCount() == 0;
        }

        public NavigationTreeNode addThesaurusToSelected(JAXXContext context, VradiThesaurusDTO thesaurus) {
            NavigationTreeNode selectedNode = (NavigationTreeNode) getSelectedNode(context);

            return addThesaurus(context, selectedNode, thesaurus);
        }

        public NavigationTreeNode addThesaurus(JAXXContext context, NavigationTreeNode parent, VradiThesaurusDTO thesaurus) {
            return addThesaurus(parent, thesaurus.getWikittyId());
        }

        public NavigationTreeNode addThesaurusAndChildrenRecursivly(NavigationTreeNode parent, VradiThesaurusDTO thesaurus) {
            NavigationTreeNode newNode = addThesaurus(parent, thesaurus.getWikittyId());
            for (VradiThesaurusDTO child : thesaurus.getChildren()){
                addThesaurusAndChildrenRecursivly(newNode, child);
            }
            return newNode;
        }

        public NavigationTreeNode addThesaurus(NavigationTreeNode parent, String thesaurusId) {
            NavigationTreeNode result;
            Decorator<VradiThesaurusDTO> tDecorator = getThesaurusDecorator();
            String path;
            if (parent == null) {
                parent = (NavigationTreeNode) getRootNode();
                path = "..children[@wikittyId=\"" + thesaurusId + "\"]";
            } else {
                path = ".."  + PATH_SEPARATOR + "children[@wikittyId=\"" + thesaurusId + "\"]";
            }

            result = getTreeBuilder().build(parent, tDecorator, path, thesaurusId, null, null);
            if (log.isDebugEnabled()) {
                log.debug("addThesaurus : " + thesaurusId + " to parent : " +
                        (parent == null ? "null" : parent.getFullPath()) + " fullPath : " +
                        (result == null ? "result is null" : result.getFullPath()));
            }
            sort(context, result);
            return result;
        }

        // Build the root node to display
        public NavigationTreeNode buildRoot() {
            if (log.isDebugEnabled()){
                log.debug("Building root : " + getRootName());
            }
            NavigationTreeNode invisibleRootNode = getTreeBuilder().buildEmptyRoot(getThesaurusRef(), getThesaurusRef().getName());

            // When it's the root one
            if (rootThesaurus.getParentThesaurus() == null){
                return invisibleRootNode;
            }

            // Add root node
            return getTreeBuilder().build(invisibleRootNode, getThesaurusDecorator(), rootThesaurus.getBuildPath(PATH_SEPARATOR), rootThesaurus.getWikittyId(), null, null);
        }

        public void buildChildren(JAXXContext context, NavigationTreeNode parentNode) {
            if (log.isDebugEnabled()) {
                log.debug("buildChildren of thesaurus " + parentNode.getFullPath());
            }

            VradiThesaurusDTO parent = (VradiThesaurusDTO) parentNode.getBean(context);
            if (parent != null){
                for (VradiThesaurusDTO child : parent.getChildren()) {
                    NavigationTreeNode childNode = addThesaurus(context, parentNode, child);

                    // Build childs
                    if (child.getWikittyId() != null) {
                        buildChildren(context, childNode);
                    }
                }
            }
        }
    }

    public void sort(JAXXContext context, NavigationNode node){
        NavigationNode parent = node.getParent();
        if (parent == null){
            parent = getRootNode();
        }
        Enumeration<? extends NavigationNode> childEnum = parent.children();

        List<NavigationNode> children = Collections.list((Enumeration<NavigationNode>) childEnum);

        Map<VradiThesaurusDTO, NavigationNode> toSort = new HashMap<VradiThesaurusDTO, NavigationNode>();

        List<VradiThesaurusDTO> beans = new ArrayList<VradiThesaurusDTO>();

        // Extract beans
        int cnt = 0;
        for (NavigationNode child : children){
            Object o = child.getBean(context);
            VradiThesaurusDTO casted = (VradiThesaurusDTO) o;
            beans.add(casted);
            toSort.put(casted, child);
            if (log.isDebugEnabled()){
                log.debug("Will sort : " + ++cnt + " - " + (casted == null ? "null" : casted.toString()));
            }
        }

        // Sort beans
        Collections.sort(beans, VradiComparators.THESAURUS_COMPARATOR);

        // Retrieve sorted nodes
        children.clear();
        cnt = 0;
        for (VradiThesaurusDTO bean : beans){
            NavigationNode retrieveNode = toSort.get(bean);
            if (log.isDebugEnabled()){
                log.debug("Retrieve sorted bean : " +
                        ++cnt + " - " + (bean == null ? "null" : bean.toString()));
            }
            children.add(retrieveNode);
        }

        for (NavigationNode child : children){
            int index = children.indexOf(child);
            parent.remove(child);
            parent.insert(child, index);
        }
        builder.getModel().nodeStructureChanged(parent);
    }

    protected Decorator<VradiThesaurusDTO> getThesaurusDecorator() {
        Decorator<VradiThesaurusDTO> tDecorator = VradiContext.get().getDecoratorProvider().getDecorator(VradiThesaurusDTO.class);
        return tDecorator;
    }

}
