/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi;

import static org.apache.commons.beanutils.PropertyUtils.getPropertyDescriptor;

import java.awt.Component;
import java.beans.PropertyDescriptor;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.swing.ComboBoxModel;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JList;
import javax.swing.ListCellRenderer;

import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.WikittyExtension;

import com.jurismarches.vradi.entities.FieldTypeEnum;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * @author letellier
 */
public class VradiHelper {

    private static final Log log = LogFactory.getLog(VradiHelper.class);

    public final static String QUERIES = "vradi.queries";
    public final static String XML_STREAMS = "vradi.xmlStreams";
    public final static String CRITERIAS = "vradi.criterias";
    public final static String THESAURUS = "vradi.thesaurus";
    public final static String COLUMNS = "vradi.columns";
    public final static String LAST_VERSION = "vradi.lastVersion";
    public final static String TYPE_DESCRIPTION = "description";
    public final static String TYPE_LAST_MODIFIED_DATE = "lastModifiedDate";
    public final static String TYPE_LAST_MODIFIED_TIME = "lastModifiedTime";
    public final static String TYPE_LAST_MODIFIED_BY = "lastModifiedBy";

    // Singleton sur les proprietes
    public static Properties userProperties = null;
    public static Properties querryProperties = null;

    public static VradiConfig getVradiConfig() {
        return VradiContext.get().getVradiConfig();
    }

    protected static Properties getUserProperties() {
        if (userProperties == null) {
            userProperties = getProperties(getVradiConfig().getUserFile());
        }
        return userProperties;
    }

    protected static Properties getProperties(File propertyFile) {
        Properties properties;
        try {
            properties = new Properties();
            if (getVradiConfig().getVersion().getVersion() != null
                    && !getVradiConfig().getVersion().getVersion().isEmpty()) {
                if (!propertyFile.exists()) {
                    propertyFile.createNewFile();
                }
                InputStream input = new FileInputStream(propertyFile);
                properties.load(input);
            }
        } catch (IOException ioe) {

            throw new IllegalStateException("could not get "
                    + propertyFile.getName() + " caused by : "
                    + ioe.getMessage(), ioe);
        }
        return properties;
    }

    protected static Properties store(Properties property, File file) {
        try {
            Writer fileOutputStream = new OutputStreamWriter(
                    new FileOutputStream(file), "ISO-8859-1");
            property.store(fileOutputStream, "");
        } catch (IOException e) {
            log.error("Cant save request property caused by : ", e);
            ErrorDialogUI.showError(e);
        }

        return property;
    }

    protected static void storeUserProperties(Properties properties) {

        File userFile = getVradiConfig().getUserFile();
        if (userFile != null) {
            store(properties, userFile);
        }
    }

    protected static List<String> getUserListOfStringFromProperties(
            String propertyName) {
        Properties properties = getUserProperties();
        List<String> result = new ArrayList<String>();
        // Recuperation des propriete
        if (properties != null) {
            String resultNames = properties.getProperty(propertyName);
            if (resultNames != null) {
                for (String resultName : resultNames.split(",")) {
                    String utf8ResultName = null;
                    try {
                        utf8ResultName = new String(
                                resultName.getBytes("ISO-8859-1"), "UTF-8");
                    } catch (UnsupportedEncodingException eee) {
                        utf8ResultName = resultName;
                    }
                    result.add(utf8ResultName);
                }
            } else {
                log.warn("Cant find property : " + propertyName);
            }
        }
        return result;
    }

    protected static void storeUserListProperties(String propertyName,
                                                  List<String> values) {
        Properties properties = getUserProperties();
        if (properties != null) {
            StringBuffer store = new StringBuffer();
            for (String value : values) {
                store.append(value).append(",");
            }

            // Removing last ","
            int length = store.length() - 1;
            if (length > 0) {
                store.deleteCharAt(length);
            }
            properties.setProperty(propertyName, store.toString());
            storeUserProperties(properties);
        }
    }

    protected static void addToUserListProperties(String propertyName,
                                                  String value) {
        addToUserListProperties(propertyName, value, false);
    }

    protected static void addToUserListProperties(String propertyName,
                                                  String value,
                                                  boolean replace) {
        addToUserListProperties(propertyName, value, replace, false, false);
    }

    protected static void addToUserListProperties(String propertyName,
                                                  String value, boolean replace,
                                                  boolean inverse,
                                                  boolean uniqueValues) {
        List<String> result = replace ? new ArrayList<String>() :
                getUserListOfStringFromProperties(propertyName);
        if (value != null) {
            String isoValue;
            try {
                isoValue = new String(value.getBytes("UTF-8"), "ISO-8859-1");
            } catch (UnsupportedEncodingException eee) {
                isoValue = value;
            }
            if (uniqueValues && result.contains(isoValue)) {
                result.remove(isoValue);
                if (inverse) {
                    result.add(0, isoValue);
                } else {
                    result.add(isoValue);
                }
            } else {
                if (inverse) {
                    result.add(0, isoValue);
                } else {
                    result.add(isoValue);
                }
            }
        }
        storeUserListProperties(propertyName, result);
    }

    protected static List<String> removeToUserListProperties(
            String propertyName, String value) {
        List<String> result = getUserListOfStringFromProperties(propertyName);
        result.remove(value);
        storeUserListProperties(propertyName, result);

        return result;
    }

    public static void addRequestToProperties(String request) {
        addToUserListProperties(QUERIES, request, false, true, true);
    }

    public static java.util.List<String> loadRequests() {
        return getUserListOfStringFromProperties(QUERIES);
    }

    public static void setLastItemOfXmlStream(String xmlStreamName,
                                              String value) {
        addToUserListProperties(XML_STREAMS + "." + xmlStreamName, value, true);
    }

    public static String getLastItemOfXmlStream(String xmlStreamName) {
        List<String> prop = getUserListOfStringFromProperties(XML_STREAMS + "."
                + xmlStreamName);
        if (!prop.isEmpty()) {
            return prop.get(0);
        }
        return null;
    }

    public static List<String> getVradiListCriteria() {
        return getUserListOfStringFromProperties(CRITERIAS);
    }

    public static void removeVradiListCriteria(TreeNode value) {
        removeToUserListProperties(CRITERIAS, value.getWikittyId());
    }

    public static void addVradiListCriteria(TreeNode value) {
        addToUserListProperties(CRITERIAS, value.getWikittyId());
    }

    public static List<String> getVradiListThesaurus() {
        return getUserListOfStringFromProperties(THESAURUS);
    }

    public static void removeVradiListThesaurus(TreeNode value) {
        removeToUserListProperties(THESAURUS, value.getWikittyId());
    }

    public static void addVradiListThesaurus(TreeNode value) {
        addToUserListProperties(THESAURUS, value.getWikittyId());
    }

    public static List<String> getVradiListColumns() {
        return getUserListOfStringFromProperties(COLUMNS);
    }

    public static void removeVradiListColumns(String value) {
        removeToUserListProperties(COLUMNS, value);
    }

    public static void addVradiListColumns(String value) {
        addToUserListProperties(COLUMNS, value);
    }

    public static String getEntityName(Object o) {
        if (o == null) {
            return "";
        }
        PropertyDescriptor descriptor = null;
        try {
            descriptor = getPropertyDescriptor(o, "name");

            if (descriptor == null) {
                return "";
            }
            Method getter = descriptor.getReadMethod();

            Object invoke = null;
            invoke = getter.invoke(o);
            if (invoke == null) {
                return "";
            }
            String result = (String) invoke;
            return result;
        } catch (IllegalAccessException e) {
            log.error(e);
            ErrorDialogUI.showError(e);
        } catch (InvocationTargetException e) {
            log.error(e);
            ErrorDialogUI.showError(e);
        } catch (NoSuchMethodException e) {
            log.error(e);
            ErrorDialogUI.showError(e);
        }
        return "";
    }

    // Models
    public static FieldTypeEnum getFieldTypeEnum(FieldType type) {
        for (FieldTypeEnum typeEnum : FieldTypeEnum.values()) {
            if (typeEnum.isType(type)) {
                return typeEnum;
            }
        }
        return null;
    }

    public static ComboBoxModel getFormTypesModel() {
        ComboBoxModel model = new DefaultComboBoxModel(getFormTypes());
        return model;
    }

    public static Object[] getFormTypes() {
        VradiStorageService vradiStorageService =
                ServiceHelper.getVradiStorageService();
        List<WikittyExtension> formTypes = new ArrayList<WikittyExtension>();
        try {
            formTypes = vradiStorageService.getAllFormTypes();
        } catch (TechnicalException eee) {
            ErrorDialogUI.showError(eee);
            log.error("Cant get form type : " + eee);
        }
        Object[] result = new Object[formTypes.size() + 1];
        result[0] = null;
        for (int i = 0; i < formTypes.size(); i++) {
            result[i + 1] = formTypes.get(i);
        }
        return result;
    }

    private static Map<String, Status> statusCache = new HashMap<String, Status>();
    
    public static Status getStatus(String statusId) {
        if(statusId == null) {
            return null;
        }
        
        Status status = statusCache.get(statusId);
        
        if (status == null) {
            VradiStorageService vradiStorageService =
                    ServiceHelper.getVradiStorageService();
            try {
                status = vradiStorageService.getStatus(statusId);
                statusCache.put(statusId, status);
                
            } catch (TechnicalException eee) {
                log.error("Cant get status : " + eee);
            }
        }
        
        return status;
    }

    public static ListCellRenderer getStatusesRenderer() {
        return new DefaultListCellRenderer() {

            @Override
            public Component getListCellRendererComponent(JList jList, Object o,
                                                          int i,
                                                          boolean b,
                                                          boolean b1) {
                Status status = (Status)o;
                String statusName = status == null ? "" :
                    status.getName();
                return super.getListCellRendererComponent(
                    jList, statusName, i, b, b1);
            }
        };
    }


    // Dates
    public static Date getLastDayDate() {
        Date result = new Date();
        Calendar cal = Calendar.getInstance();
        cal.setTime(result);
        cal.roll(Calendar.DAY_OF_MONTH, false);

        return cal.getTime();
    }

    public static Date getBeginMonthDate() {
        Date result = new Date();
        Calendar cal = Calendar.getInstance();
        cal.setTime(result);
        cal.set(Calendar.DAY_OF_MONTH, 1);

        return cal.getTime();
    }
    
    public static void executeQuery(String query, WikittyExtension extension,
                                    String dateType, Date beginDate,
                                    Date endDate, List<String>[] thesaurus,
                                    VradiFormPageDTO formPageDTO) {
        try {
            if (log.isDebugEnabled()) {
                log.debug(query);
            }
            
            formPageDTO.setFormsToShow(null);
            formPageDTO.setTotalFoundFormNb(0);
            
            if (formPageDTO.getNbFormsToShow() == 0) {
                formPageDTO.setNbFormsToShow(10);
            }
            
            if (formPageDTO.getPageToShow() == 0) {
                formPageDTO.setPageToShow(1);
            }
            
            ServiceHelper.getVradiStorageService().findForms(query, extension,
                    dateType, beginDate, endDate, thesaurus, formPageDTO);
            
            if (log.isDebugEnabled()) {
                log.debug(formPageDTO.getTotalFoundFormNb() + " forms found");
                for (Form form : formPageDTO.getFormsToShow()) {
                    log.debug(form.getWikittyId() + " : " + form.getObjet());
                }
            }
            
        } catch (TechnicalException eee) {
            log.error("Cant execute query : ", eee);
            ErrorDialogUI.showError(eee);
            formPageDTO = new VradiFormPageDTO();
            
        } catch (UnsupportedQueryException e) {
            log.error(e.getMessage());
            ErrorDialogUI.showError(e);
            formPageDTO = new VradiFormPageDTO();
        }
    }


    public static String getFieldTypeDescription(FieldType fieldType) {
        return fieldType.getTagValue(TYPE_DESCRIPTION);
    }

    /**
     * Get the most recent version of Vradi already run by the user
     *
     * @return the most recent version that the user has ever launched
     */
    public static String getLastVersion() {
        Properties userProperties = getUserProperties();
        return userProperties.getProperty(LAST_VERSION);
    }

    public static void setLastVersion(String lastVersion) {
        Properties userProperties = getUserProperties();
        userProperties.setProperty(LAST_VERSION, lastVersion);
        storeUserProperties(userProperties);
    }

    public static boolean isVersionNewer(String version) {
        if (getLastVersion() == null) {
            return true;
        }
        String[] versionToken = version.split("\\.");
        String[] lastVersionToken = getLastVersion().split("\\.");
        int i = 0;
        while (i < versionToken.length && i < lastVersionToken.length) {
            log.debug(versionToken[i] + " ? " + lastVersionToken[i]);
            if (Integer.valueOf(versionToken[i])
                    > Integer.valueOf(lastVersionToken[i])) {
                return true;
            } else if (Integer.valueOf(versionToken[i])
                    < Integer.valueOf(lastVersionToken[i])) {
                return false;
            }
            i++;
        }
        return false;
    }

}
