/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: VradiServiceConfiguration.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3.1/vradi-services/src/main/java/com/jurismarches/vradi/VradiServiceConfiguration.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package com.jurismarches.vradi;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;

import org.apache.commons.lang.UnhandledException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.StringUtil;
import org.nuiton.wikitty.WikittyServiceCached;
import org.nuiton.wikitty.WikittyServiceNotifier;
import org.nuiton.wikitty.XMPPNotifierTransporter;

/**
 * Configuration.
 *
 * @author schorlet
 * @version $Revision: 1715 $ $Date: 2010-10-27 21:21:28 +0200 (mer., 27 oct. 2010) $
 * @since 26 mars 2010 21:18:08
 */
public class VradiServiceConfiguration extends ApplicationConfig {

    /** log. */
    private static final Log log = LogFactory.getLog(VradiServiceConfiguration.class);

    /** After init step. */
    public static final int STEP_AFTER_INIT = 0;

    /** Singleton instance. */
    protected static VradiServiceConfiguration instance = null;

    protected VradiServiceConfiguration() {
        super();
        
        // set default option (included configuration file name : important)
        for (Option o : Option.values()) {
            if (o.defaultValue != null) {
                setDefaultOption(o.key, o.defaultValue);
            }
        }
        
        // creation des actions disponibles
        for (ActionDefinition a : ActionDefinition.values()) {
            for (String alias : a.aliases) {
                addActionAlias(alias, a.action);
            }
        }
    }

    public static synchronized VradiServiceConfiguration getInstance(String... args) {
        if (instance == null) {
            try {
                instance = new VradiServiceConfiguration();
                instance.parse(args);
                // was done in parse before (but not good for tests)
                instance.setSystemDefaultProps();
            } catch (Exception ex) {
                if (log.isErrorEnabled()) {
                    log.error("Can't intitialize configuration", ex);
                }
                throw new UnhandledException(ex);
            }
        }
        return instance;
    }

    protected void addActionAlias(String action, String... aliases) {
        for(String alias : aliases) {
            super.addActionAlias(alias, action);
        }
    }

    /**
     * Get current application version as string.
     * 
     * @return version
     */
    public String getApplicationVersion() {
        return getOption(Option.APPLICATION_VERSION.key);
    }

    /**
     * Get last saved version as string (last launch).
     * 
     * @return version
     */
    public String getServiceVersion() {
        return getOption(Option.SERVICE_VERSION.key);
    }

    /**
     * Set service version.
     * 
     * @param version version to set
     */
    public void setServiceVersion(String version) {
        setOption(Option.SERVICE_VERSION.key, version);
    }

    public String getDataDir() {
        String option = getOption(Option.DATA_DIR.key);
        return option;
    }
    public File getDataDirAsFile() {
        File option = getOptionAsFile(Option.DATA_DIR.key);
        return option;
    }

    public Locale getLocale() {
        Locale result = getOption(Locale.class, Option.LOCALE.key);
        return result;
    }
    
    public void setLocale(Locale locale) {
        setOption(Option.LOCALE.key, locale.toString());
    }

    @Deprecated
    public File getCommonsPropertiesFile() {
        return getOptionAsFile("vradi.data.dir", ".vradi-properties");
    }

    public File getAttachmentsDir() {
        return getOptionAsFile("vradi.data.dir", "attachments");
    }

    public File getEmbeddedFilesDir() {
        return getOptionAsFile("vradi.data.dir", "embeddedFiles");
    }

    public File getQueryHistoryDir() {
        return getOptionAsFile("vradi.data.dir", "queryHistory");
    }

    public File getTemplatesDir() {
        return getOptionAsFile("vradi.data.dir", "templates");
    }

    public File getPdfDir() {
        return getOptionAsFile("vradi.data.dir", "pdf");
    }

    public File getTempDir() {
        return getOptionAsFile("vradi.data.dir", "temp");
    }

    public File getExportCSVDir() {
        return getOptionAsFile("vradi.data.dir", "exportCSV");
    }

    public File getImportCSVDir() {
        return getOptionAsFile("vradi.data.dir", "importCSV");
    }

    public File getWebHarvestPreviewDir() {
        return getOptionAsFile("vradi.data.dir", "WebHarvestPreview");
    }

    public File getWebHarvestScriptDir() {
        return getOptionAsFile("vradi.data.dir", "WebHarvestScript");
    }

    public String getOpenOfficeExecDir() {
        return getOption("oOo.exec.folder");
    }

    // Email config
    /*public String getMailerDaemonEmail() {
        return getOption("vradi.mail.mailerdaemonemail");
    }*/
    
    public String getSmtpHost() {
        return getOption("vradi.smtp.host");
    }

    public int getSmtpPort() {
        return getOptionAsInt("vradi.smtp.port");
    }

    public String getImapHost() {
        return getOption("vradi.imap.host");
    }

    public int getImapPort() {
        return getOptionAsInt("vradi.imap.port");
    }

    public String getMailUser() {
        return getOption("vradi.mail.user");
    }
    
    public String getMailFrom() {
        return getOption("vradi.mail.from");
    }
    
    public String getMailFromName() {
        return getOption("vradi.mail.fromname");
    }

    public String getMailPassword() {
        return getOption("vradi.mail.password");
    }

    public String getOfferMailSubject() {
        String result = getOption("vradi.mail.offer.subject");
        return result;
    }

    public boolean isMailDebug() {
        return getOptionAsBoolean("vradi.mail.debug");
    }

    public List<String> getMailAdmin() {
        List<String> result = new ArrayList<String>();
        String admins = getOption(Option.MAIL_ADMIN.key);
        if (admins != null && !admins.isEmpty()) {
            if (admins.contains(", ")) {
                String[] strings = admins.split(", ");
                result = Arrays.asList(strings);
            } else {
                result.add(admins);
            }
        }
        return result;
    }

    public void addMailAdmin(String email) {
        List<String> mailAdmins = getMailAdmin();
        mailAdmins.add(email);

        String value = StringUtil.join(mailAdmins, ", ", false);

        setOption(Option.MAIL_ADMIN.key, value);
    }

    public String getSessionDefaultParagraph() {
        return getOption("vradi.session.defaultParagraph");
    }

    public boolean skipMigration() {
        return getOptionAsBoolean("vradi.migration.skip");
    }

    /**
     * To specify XMPP room to use
     *
     * @param xmppRoom to use
     *
     * @see XMPPNotifierTransporter#WIKITTY_XMPP_ROOM
     */
    public void setWikittyXMPPRoom(String xmppRoom) {
        setOption(Option.WIKITTY_XMPP_ROOM.key, xmppRoom);
    }

    /**
     * to specify XMPP server to use
     *
     * @param xmppServer to use
     *
     * @see XMPPNotifierTransporter#WIKITTY_XMPP_SERVER
     */
    public void setWikittyXMPPServeur(String xmppServer) {
        setOption(Option.WIKITTY_XMPP_SERVER.key, xmppServer);
    }

    /**
     * To specify class to use for notifier transporter
     *
     * @param remoteNotifierClass class to use
     *
     * @see WikittyServiceNotifier.RemoteNotifier#WIKITTY_NOTIFIER_TRANSPORTER_CLASS
     */
    public void setWikittyNotifierTransporteurClass(Class remoteNotifierClass) {
        setOption(Option.WIKITTY_EVENT_PROPAGATE_OPTION.key, remoteNotifierClass.getName());
    }

    /**
     * Change propagate events use.
     *
     * @param b new value
     *
     * @see WikittyServiceNotifier.RemoteNotifier#WIKITTY_EVENT_PROPAGATE_OPTION
     */
    public void setWikittyPropagateEvents(boolean b) {
        setOption(Option.WIKITTY_EVENT_PROPAGATE_OPTION.key, String.valueOf(b));
    }

    /**
     * Change listen events on cache use.
     * 
     * @param b new value
     * 
     * @see WikittyServiceCached#WIKITTY_CACHE_LISTENEVENTS_OPTION
     */
    public void setWikittyListenEvents(boolean b) {
        setOption(Option.WIKITTY_CACHE_LISTENEVENTS_OPTION.key, String.valueOf(b));
    }

    /**
     * Change wikitty cache policy.
     * 
     * @param b cache policy
     * 
     * @see WikittyServiceCached#WIKITTY_CACHE_ALLWAYS_RESTORE_COPIES_POLICY_OPTION
     */
    public void setWikittyCachePolicy(boolean b) {
        setOption(Option.WIKITTY_CACHE_ALLWAYS_RESTORE_COPIES.key, String.valueOf(b));
    }

    protected File getOptionAsFile(String parent, String child) {
        String parentDir = getOption(parent);
        File option = new File(parentDir, child);
        return option;
    }

    /**
     * Set {@code solr} and {@code jms} system configuration.
     *
     * This is the "only" way to configure embedded solr.
     */
    protected void setSystemDefaultProps() {
        String[] datadirs = new String[] { "solr.data.dir", "vradi.objectStore.dir"};

        for (String datadir : datadirs) {
            String value = System.getProperty(datadir, null);
            if (value == null) {
                value = getOption(datadir);
                if (log.isInfoEnabled()) {
                    log.info("Setting system property " + datadir + " : " + value);
                }
                System.setProperty(datadir, value);
                env.put(datadir, value);
            }
        }
    }

    /**
     * Vradi option definition.
     */
    public static enum Option {

        CONFIG_FILE(CONFIG_FILE_NAME, _("vradi.service.config.configFileName.description"), "vradi-services.properties", String.class, true, true),
        APPLICATION_VERSION("application.version", _("vradi.service.config.application.version.description"), null, String.class, false, false),
        SERVICE_VERSION("vradi.service.version", _("vradi.service.config.version.description"), null, String.class, false, false),
        DATA_DIR("vradi.data.dir", _("vradi.service.config.data.dir.description"), "${user.home}/.vradi/vradi-${application.version}", String.class, false, false),
        LOCALE("vradi.service.locale", _("vradi.service.config.ui.locale"), Locale.FRANCE.toString(), Locale.class, false, false),
        MAIL_ADMIN("vradi.mail.admin", _("vradi.service.config.mail.admin"), null, String.class, false, false),

        // XMPP server configuration
        WIKITTY_XMPP_ROOM(XMPPNotifierTransporter.WIKITTY_XMPP_ROOM,
                _("vradi.service.config.wikitty.xmpp.room.description"), null, String.class, false, false),
        WIKITTY_XMPP_SERVER(XMPPNotifierTransporter.WIKITTY_XMPP_SERVER,
                _("vradi.service.config.wikitty.xmpp.server.description"), null, String.class, false, false),
        WIKITTY_NOTIFIER_TRANSPORTER_CLASS(WikittyServiceNotifier.RemoteNotifier.WIKITTY_NOTIFIER_TRANSPORTER_CLASS,
                _("vradi.service.config.wikitty.notifier.transporter.class.description"),
                "org.nuiton.wikitty.XMPPNotifierTransporter", Class.class, false, false),
        WIKITTY_EVENT_PROPAGATE_OPTION(WikittyServiceNotifier.RemoteNotifier.WIKITTY_EVENT_PROPAGATE_OPTION,
                _("vradi.service.config.wikitty.propagate.propagateEvent"), "true", Boolean.class, false, false),

        // Cache configuration
        WIKITTY_CACHE_LISTENEVENTS_OPTION(WikittyServiceCached.WIKITTY_CACHE_LISTENEVENTS_OPTION,
                _("vradi.service.config.listenevents.description"), "true", Boolean.class, false, false),
        WIKITTY_CACHE_ALLWAYS_RESTORE_COPIES(WikittyServiceCached.WIKITTY_CACHE_ALLWAYS_RESTORE_COPIES_POLICY_OPTION,
                _("vradi.service.config.cacheusecopies.description"), "true", Boolean.class, false, false);

        public final String key;
        public final String description;
        public final String defaultValue;
        public final Class<?> type;
        public final boolean isTransient;
        public final boolean isFinal;

        private Option(String key, String description, String defaultValue,
                Class<?> type, boolean isTransient, boolean isFinal) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
            this.type = type;
            this.isFinal = isFinal;
            this.isTransient = isTransient;
        }
    }
    
    /**
     * Actions can be called in command line
     *
     * @author sletellier <letellier@codelutin.com>
     */
    public enum ActionDefinition {

        HELP(VradiServiceAction.class.getName() + "#help", _("vradi.service.config.help.description"), "-h", "--help"),
        REINDEX_SOLR(VradiServiceAction.class.getName() + "#reIndexSolr", _("vradi.service.config.reIndexSolr.description"), "--reIndexSolr"),
        DELETE_OBSELETE_THESAURUS(VradiServiceAction.class.getName() + "#deleteObseleteThesaurus", _("vradi.service.config.deleteObseleteThesaurus.description"), "--deleteObseleteThesaurus"),
        REINIT_STATUS(VradiServiceAction.class.getName() + "#reinitStatus", _("vradi.service.config.reinitStatus.description"), "--reinitStatus"),
        FORCE_ALL_OLD_USER_ENABLE(VradiServiceAction.class.getName() + "#forceAllOldUserEnable", _("vradi.service.config.forceAllOldUserEnable.description"), "--forceAllOldUserEnable"),
        DELETE_OBSELETE_USERS(VradiServiceAction.class.getName() + "#deleteObseleteUsers", _("vradi.service.config.deleteObseleteUsers.description"), "--deleteObseleteUsers"),
        IMPORT_FILES(VradiServiceAction.class.getName() + "#importFiles", _("vradi.service.config.importFiles.description"), "--importFiles"),
        EXPORT_CLIENTS(VradiServiceAction.class.getName() + "#exportClients", _("vradi.service.config.exportClients.description"), "--exportClients"),
        EXPORT_THESAURUS(VradiServiceAction.class.getName() + "#exportThesaurus", _("vradi.service.config.exportThesaurus.description"), "--exportThesaurus");

        protected String action;

        protected String description;

        protected String[] aliases;

        private ActionDefinition(String action, String description, String... aliases) {
            this.action = action;
            this.description = description;
            this.aliases = aliases;
        }

        public String getAction() {
            return action;
        }
        
        public String getDescription() {
            return action;
        }

        public String[] getAliases() {
            return aliases;
        }
    }
}
