/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services;

import java.io.File;
import java.util.Date;
import java.util.List;
import java.util.Map;

import com.jurismarches.vradi.entities.*;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.BusinessEntity;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.WikittyExtension;

import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiSendingDTO;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * VradiStorageService.
 *
 * @author morin
 * @version $Revision: 495 $ $Date: 2010-02-09 11:15:15 +0100 (mar., 09 févr. 2010) $
 */
public interface VradiStorageService {

    /**
     * Returns a <code>BusinessEntity</code> of the specified <code>id</code> and <code>clazz</code>.
     *
     * @param id    the id of the entity
     * @param clazz the class of the entity
     * @return a <code>BusinessEntity</code>
     * @throws TechnicalException
     */
    
    BusinessEntity getEntity(String id, Class<? extends BusinessEntity> clazz) throws TechnicalException;

    /**
     * Updates the specified <code>BusinessEntity</code>.
     *
     * @param entity the entity to update
     * @return the updated entity
     * @throws TechnicalException
     */
    BusinessEntity updateEntity(BusinessEntity entity) throws TechnicalException;

    /**
     * Delete the specified <code>BusinessEntity</code>.
     *
     * @param entity the entity to delete
     * @throws TechnicalException
     */
    void deleteEntity(BusinessEntity entity) throws TechnicalException;

    /**
     * Returns a <code>Client</code> with the specified <code>clientId</code>.
     *
     * @param clientId the id of the client
     * @return a <code>Client</code>
     * @throws TechnicalException
     */
    Client getClient(String clientId) throws TechnicalException;

    /**
     * Returns a <code>User</code> with the specified <code>userId</code>.
     *
     * @param userId the id of the user
     * @return a <code>User</code>
     * @throws TechnicalException
     */
    User getUser(String userId) throws TechnicalException;

    /**
     * Returns a <code>Group</code> with the specified <code>groupId</code>.
     *
     * @param groupId the id of the group
     * @return a <code>Group</code>
     * @throws TechnicalException
     */
    Group getGroup(String groupId) throws TechnicalException;

    /**
     * Returns a list of <code>Users</code> belonging to the specified <code>groupId</code>.
     *
     * @param groupId the id of the group
     * @return a list of <code>Users</code>
     * @throws TechnicalException
     */
    List<User> getGroupUsers(String groupId) throws TechnicalException;

    /**
     * Returns a list of <code>Users</code> belonging to the specified <code>clientId</code>.
     *
     * @param clientId the id of the client
     * @return a list of <code>Users</code>
     * @throws TechnicalException
     */
    List<User> getClientUsers(String clientId) throws TechnicalException;

    /**
     * Returns a list of <code>Clients</code> belonging to the specified <code>groupId</code>.
     *
     * @param groupId the id of the group
     * @return a list of <code>Clients</code>
     * @throws TechnicalException
     */
    List<Client> getGroupClients(String groupId) throws TechnicalException;

    /**
     * Returns a <code>Client</code> whose specified <code>userId</code> belongs.
     *
     * @param userId the id of the user
     * @return a <code>Client</code>
     * @throws TechnicalException
     */
    Client getClientByUserId(String userId) throws TechnicalException;

    /**
     * Returns a list of <code>Group</code> that the specified <code>userId</code> belongs.
     *
     * @param userId the id of the user
     * @return a list of <code>Group</code>
     * @throws TechnicalException
     */
    List<Group> getGroupsByUserId(String userId) throws TechnicalException;

    /**
     * Returns a list of <code>Group</code> that the specified <code>clientId</code> belongs.
     *
     * @param clientId the id of the client
     * @return a list of <code>Group</code>
     * @throws TechnicalException
     */
    List<Group> getGroupsByClientId(String clientId) throws TechnicalException;

    /**
     * Returns all the clients.
     *
     * @return all the clients
     * @throws TechnicalException
     */
    List<Client> getAllClients() throws TechnicalException;

    /**
     * Returns all the groups.
     *
     * @return all the groups
     * @throws TechnicalException
     */
    List<Group> getAllGroups() throws TechnicalException;

    /**
     * Update a list of <code>XmlFieldBinding</code>.
     *
     * @param bindings the list of XmlFieldBinding to update
     * @return the list of updated XmlFieldBinding
     */
    List<XmlFieldBinding> updateXmlFieldBindings(List<XmlFieldBinding> bindings) throws TechnicalException;

    /**
     * Creates or Update a form type.
     *
     * @param extension the extension designing the form type
     * @return a <code>WikittyExtension</code>
     */
    WikittyExtension updateFormType(WikittyExtension extension) throws TechnicalException;

    /**
     * Creates or Update a form type.
     *
     * @param name      the name of the form type
     * @param fields    the fields of the form type
     * @param requires  the extension requred by the extension to update
     * @param tagValues the map containing the tags and their values
     * @return a <code>WikittyExtension</code>
     */
    WikittyExtension updateFormType(String name, Map<java.lang.String, org.sharengo.wikitty.FieldType> fields, String requires, Map<java.lang.String, java.lang.String> tagValues) throws TechnicalException;

    /**
     * Creates or Update the specified <code>form</code>.
     *
     * @param form the form to create or update
     * @return the stored <code>Form</code>
     */
    Form updateForm(Form form) throws TechnicalException;

    /**
     * Returns a <code>XmlStream</code> with the specified <code>xmlStreamId</code>.
     *
     * @param xmlStreamId the id of the XmlStream
     * @return a <code>XmlStream</code>
     * @throws TechnicalException
     */
    XmlStream getXmlStream(String xmlStreamId) throws TechnicalException;

    /**
     * Returns a list of <code>XmlFieldBinding</code> belonging to the specified <code>xmlStream</code>.
     *
     * @param xmlStream the <code>XmlStream</code>
     * @return a list of <code>XmlFieldBinding</code>
     */
    List<XmlFieldBinding> getXmlFieldBindings(XmlStream xmlStream) throws TechnicalException;

    /**
     * Returns a <code>XmlFieldBinding</code> with the specified <code>xmlFieldBindingId</code>.
     *
     * @param xmlFieldBindingId the id of the XmlFieldBinding
     * @return a <code>XmlFieldBinding</code>
     */
    XmlFieldBinding getXmlFieldBinding(String xmlFieldBindingId) throws TechnicalException;

    /**
     * Returns a <code>TreeNode</code> with the specified <code>thesaurusId</code>.
     *
     * @param thesaurusId the id of the thesaurus
     * @return a <code>TreeNode</code>
     * @throws TechnicalException
     */
    TreeNode getThesaurus(String thesaurusId) throws TechnicalException;

    /**
     * Returns the root Thesaurus Node.
     * 
     * @return a <code>TreeNode</code>
     * @throws TechnicalException
     */
    TreeNode getRootThesaurus() throws TechnicalException;

    /**
     * Returns all children of the specified <code>thesaurusId</code>.
     * 
     * @return all list of <code>TreeNode</code>
     * @throws TechnicalException
     */
    List<TreeNode> getChildrenThesaurus(String thesaurusId) throws TechnicalException;

    /**
     * Finds fields of a form type.
     *
     * @param name the name of the form type
     * @return a map containing the names of the fields and their type
     */
    Map<String, FieldType> getFormTypeFields(String name) throws TechnicalException;

    /**
     * Returns a form type with the specified <code>name</code>.
     *
     * @param name the name of the form type
     * @return a <code>WikittyExtension</code>
     */
    WikittyExtension getFormType(String name) throws TechnicalException;

    /**
     * Returns a <code>Form</code> with the specified <code>formId</code>.
     *
     * @param formId the id of the form
     * @return the form whose id is id
     */
    Form getForm(String formId) throws TechnicalException;

    /**
     * Returns all <code>XmlStreams</code>.
     * 
     * @return a list of <code>XmlStream</code>
     */
    List<XmlStream> getAllXmlStreams() throws TechnicalException;

    /**
     * Returns all <code>TreeNode</code> considered as thesaurus nodes.
     * 
     * @return a list of <code>TreeNode</code>
     * @throws TechnicalException
     */
    List<TreeNode> getAllThesaurus() throws TechnicalException;

    /**
     * Returns all Form Types.
     *
     * @return a list of <code>WikittyExtension</code>
     */
    List<WikittyExtension> getAllFormTypes() throws TechnicalException;

    /**
     * Returns all <code>Forms</code>.
     *
     * @return a list of <code>Form</code>
     */
    List<Form> getAllForms() throws TechnicalException;

    /**
     * Searches for forms with the specified filters.
     * 
     * The <code>beginDate</code> and <code>endDate</code> specifies a
     * date range filter which is applied on <code>dateType</code> field.
     * The date range filter is applied only if the three parameters are not null.
     * 
     * @param query     the query filter
     * @param extension the extension filter
     * @param dateType  the date field. Must be fully qualified (extension.field)
     * @param beginDate the begin date filter
     * @param endDate   the end date filter
     * @param thesaurus the thesaurus filter
     * @param formPageDTO ...
     * @throws UnsupportedQueryException
     */
    void findForms(String query, WikittyExtension extension, String dateType, Date beginDate, Date endDate, java.util.List<String>[] theasurus, VradiFormPageDTO formPageDTO) throws TechnicalException, UnsupportedQueryException;

    /**
     * Searches for forms with the specified query.
     *
     * @param query the query to filter the forms
     * @param formPageDTO ...
     * @throws UnsupportedQueryException
     */
    void findForms(String query, VradiFormPageDTO formPageDTO) throws TechnicalException, UnsupportedQueryException;

    /**
     * Deletes a <code>Form</code> by the specified <code>formId</code>.
     *
     * @param formId the id of the Form to delete
     */
    void deleteForm(String formId) throws TechnicalException;

    /**
     * Binds all clients with the forms found by their queries
     * and not already sent.
     */
    void bindFormsToClients() throws TechnicalException;

    /**
     * Computes all forms by clients, status and sent date with a filter.
     *
     * @param dateType       the date field. Must be fully qualified (extension.field)
     * @param beginDate      the begin date filter
     * @param endDate        the end date filter
     * @param extension      the extension filter
     * @param receptionProof true if the sending must have a reception proof
     * @param paragraph      the paragraph sent with the forms
     * @param status         the status of the sending
     * @return               a list of VradiSendingDTO representing the forms sent to the clients
     * @throws TechnicalException
     */
    List<VradiSendingDTO> getFormsByClients(String dateType, Date beginDate, Date endDate, WikittyExtension extension, Boolean receptionProof, Boolean paragraph, int status) throws TechnicalException;

    /**
     * Computes a map of <code>Clients</code> indexed by <code>Forms</code>.
     *
     * @param dateType       the date field. Must be fully qualified (extension.field)
     * @param beginDate      the begin date filter
     * @param endDate        the end date filter
     * @param extension      the extension filter
     * @return a map containing the forms and their associated list of clients
     */
    Map<Form, List<Client>> getClientsByForms(String dateType, Date beginDate, Date endDate, WikittyExtension extension) throws TechnicalException;

    /**
     * Creates and store forms from an <code>XmlStream</code> by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * @param xmlStream        the xml stream to read
     * @param lastItemRecorded the encrypted value of the last read item
     *                         of the xml stream
     * @param vradiUser        the VradiUser that launched the import
     * @return an arry containing :
     * - the encrypted value of the last read item of the xml stream
     * - the number of created forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error  
     */
    Object[] getFormsFromXmlStream(XmlStream xmlStream, String lastItemRecorded, VradiUser vradiUser) throws TechnicalException;

    /**
     * Changes datas directory.
     *
     * @param newDataDir the new data directory path
     * @param oldDataDir the old data directory path.
     *                   If null, the data in the old directory will not be copied.
     */
    void changeDataDir(String newDataDir, String oldDataDir) throws TechnicalException;

    /**
     * Returns all the <code>Users</code>.
     *
     * @return a list of <code>User</code>
     * @throws TechnicalException
     */
    List<User> getAllUsers() throws TechnicalException;

    /**
     * Computes a list of queries returning the specified <code>Form</code>.
     *
     * @param form the form which match queries
     * @return a list of queries
     */
    List<String> getQueriesReturningForm(Form form) throws TechnicalException;

    /**
     * Returns all the <code>Forms</code> with the specified list of <code>formIds</code>.
     *
     * @param formIds the ids of the forms
     * @return a list of <code>Form</code>
     */
    List<Form> getForms(List<String> formIds) throws TechnicalException;

    /**
     * Updates a <code>VradiUser</code>.
     *
     * @param vradiUser the VradiUser to update
     * @return the vradiUser updated
     * @throws TechnicalException
     */
    VradiUser updateVradiUser(VradiUser vradiUser) throws TechnicalException;

    /**
     * Checks if <code>userName</code> and <code>userPassword</code> matches a valid <code>VradiUser</code>.
     *
     * @param userName     the username of a VradiUser
     * @param userPassword the password of a VradiUser
     * @return a <code>VradiUser</code> or null if either userName or userPassword don't match
     */
    VradiUser logVradiUser(String userName, String userPassword) throws TechnicalException;

    /**
     * Update a list of Sending.
     *
     * @param sendings the list of Sending to update
     */
    void updateSendings(List<Sending> sendings) throws TechnicalException;

    /**
     * Import data from an XML file.
     *
     * @param file the XML file containing datas
     * @throws TechnicalException
     */
    void importData(File file) throws TechnicalException;

    /**
     * Export the datas into an XML format.
     *
     * @return the datas as an XML String
     * @throws TechnicalException
     */
    String exportData() throws TechnicalException;

    /**
     * Create or Update a list of <code>Forms</code>.
     *
     * @param forms the list of forms to insert or update
     * @return an updated list of <code>Form</code>
     */
    List<Form> updateForms(List<Form> forms) throws TechnicalException;

    /**
     * Reindex the datas.
     */
    void reindexData() throws TechnicalException;

    /**
     * Computes the number of forms associated with every thesaurus node.
     *
     * @return a map containing the thesaurus nodes and the number of forms
     *         associated with them
     * @throws TechnicalException
     */
    Map<TreeNode, Integer> getNbFormsByThesaurus() throws TechnicalException;

    /**
     * Computes the number of forms associated with the specified <code>thesaurusId</code>.
     *
     * @return the number of forms associated
     * @throws TechnicalException
     */
    int getNbFormsForThesaurus(String thesaurusId) throws TechnicalException;

    /**
     * Returns a <code>Status</code> with the specified <code>statusId</code>.
     *
     * @param statusId the id of the status
     * @return a <code>Status</code>
     * @throws TechnicalException
     */
    Status getStatus(String statusId) throws TechnicalException;

    /**
     * Returns a list of <code>Status</code> with the specified
     * <code>statusIds</code>.
     *
     * @param statusIds the ids of the statuses
     * @return a list of <code>Status</code>
     * @throws TechnicalException
     */
    List<Status> getStatuses(List<String> statusIds) throws TechnicalException;

    /**
     * Returns all <code>Status</code>.
     *
     * @return a list of <code>Status</code>
     * @throws TechnicalException
     */
    List<Status> getAllStatuses() throws TechnicalException;

    /**
     * Updates a <code>Status</code>.
     *
     * @param status the Status to update
     * @return the <code>Status</code> updated
     * @throws TechnicalException
     */
    Status updateStatus(Status status) throws TechnicalException;

    /**
     * Updates a list of <code>Status</code>.
     *
     * @param statuses the list of <code>Status</code> to update
     * @return the list of the <code>Status</code> updated
     * @throws TechnicalException
     */
    List<Status> updateStatuses(List<Status> statuses)
            throws TechnicalException;

    /**
     * Deletes a <code>Status</code>.
     *
     * @param statusId the id of the Status to delete
     * @throws TechnicalException
     */
    void deleteStatus(String statusId) throws TechnicalException;

    /**
     * Deletes a list of <code>Status</code>.
     *
     * @param statusIds the list of ids of the Status to delete
     * @throws TechnicalException
     */
    void deleteStatuses(List<String> statusIds) throws TechnicalException;

    /**
     * Get the unmodifiable status 'to treat'
     * @return a <code>Status</code>
     */
    Status getToTreatStatus();

    /**
     * Get the unmodifiable status 'validated'
     * @return a <code>Status</code>
     */
    Status getValidatedStatus();

    /**
     * Propose thesaurus nodes that might be in relation with a specified form
     *
     * @param form the <code>Form</code> containing the information needed
     * to search the thesaurus nodes
     * @param thesaurus the list of all the tree nodes of the thesaurus
     * @return a list of <code>TreeNode</code>
     * @throws TechnicalException
     */
    List<TreeNode> proposeThesaurus(Form form, List<TreeNode> thesaurus) throws TechnicalException;
}
