package com.jurismarches.vradi.migration;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.UnhandledException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.Resource;
import org.sharengo.wikitty.AbstractWikittyService;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyExtensionStorage;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.WikittySearchEngine;
import org.sharengo.wikitty.WikittyStorage;
import org.sharengo.wikitty.jdbc.WikittyExtensionStorageJDBC;
import org.sharengo.wikitty.jdbc.WikittyStorageJDBC;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;
import org.sharengo.wikitty.solr.WikittySearchEngineSolr;

import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.managers.FormTypeManager;
import com.jurismarches.vradi.services.managers.ThesaurusManager;
import com.jurismarches.vradi.services.search.VradiQueryParser;

/**
 * Migration004.
 *
 * @author $Author: kmorin $
 * @version $Revision: 491 $ $Date: 2010-02-08 18:37:03 +0100 (lun., 08 févr. 2010) $
 * @since 3 févr. 2010 12:21:20
 */
public class Migration004 {
    private static final Log log = LogFactory.getLog(Migration004.class);

    static final String vradi003 = System.getProperty("user.home") +
            File.separator + ".vradi/vradi-0.0.3";
    static final String vradi004 = System.getProperty("user.home") +
            File.separator + ".vradi/vradi-0.0.4";
    
    /**
     * Performs a datas migration from 0.0.3 to 0.0.4 of vradi databases.
     * 
     * The migration only performs if 0.0.3 database exists and if
     * 0.0.4 database is empty, that is, if no thesaurus can be found.
     * 
     * Note: Completes successfully or throws an exception on both
     * WikittyException or UnhandledException
     */
    public static void performMigration() {
        /*
         *  check if vradi-0.0.3 dataDir exists and is not empty
         */
        File dataDir = new File(vradi003);
        if (!dataDir.exists()) {
            return;
        } else {
            String[] list = dataDir.list();
            
            if (list.length == 0) {
                return;
            }
        }
        
        /*
         * check if root thesaurus exists, there is no many ways to check if
         * database is already in use.
         */
        WikittyProxy proxy = ServiceHelper.getWikittyProxy();
        Search search = Search.query()
                .eq(Element.ELT_EXTENSION, TreeNode.EXT_TREENODE)
                .eq(TreeNode.FQ_FIELD_NAME, ThesaurusManager.ROOT_THESAURUS_NAME);
        Criteria criteria = search.criteria();
        
        TreeNode rootThesaurus = proxy.findByCriteria(TreeNode.class, criteria);
        if (rootThesaurus != null) {
            return;
        }
        
        try {
            log.info("begin of datas migration from 0.0.3 to 0.0.4");

            WikittyServiceJDBC_003 service = new WikittyServiceJDBC_003(vradi003);
            File exportFile = exportObjects_003(service);
            List<WikittyExtension> formTypes = exportFormTypes_003(service);
            service = null;
            
            importObjects_004(exportFile, proxy);
            importFormTypes_004(formTypes, proxy);
            
            log.info("end of datas migration !");

        } catch (Exception e) {
            try {
                FileUtils.deleteDirectory(new File(vradi004));
            } catch (IOException e1) {
            }
            throw new UnhandledException("datas migration failed: ", e);
        }
    }
    
    /**
     * Exports form type from 0.0.3 database.
     * 
     * @param service the wikitty service
     * @return a list of <code>WikittyExtension</code>
     */
    static List<WikittyExtension> exportFormTypes_003(WikittyServiceJDBC_003 service) {
        log.info("export FormType from 0.0.3");
        WikittyProxy proxy = new WikittyProxy();
        proxy.setWikittyService(service);
        
        FormTypeManager formTypeManager = new FormTypeManager(proxy);
        List<WikittyExtension> allFormTypes = formTypeManager.getAllFormTypes();
        return allFormTypes;
    }
    
    /**
     * Import form type to 0.0.4 database.
     * 
     * @param formTypes a list of <code>WikittyExtension</code>
     * @param proxy the wikitty proxy
     */
    static void importFormTypes_004(List<WikittyExtension> formTypes, WikittyProxy proxy) {
        log.info("import FormType to 0.0.4");
        proxy.getWikittyService().storeExtension(formTypes);
    }
    
    /**
     * Exports all objects of all extensions except the extension Form.
     * 
     * @param service the wikitty service
     * @return a <code>File</code> formatted by the wikitty project
     */
    static File exportObjects_003(WikittyServiceJDBC_003 service) {
        log.info("export objects from 0.0.3");
        
        try {
            Search search = VradiQueryParser.search("extensions:(* -Form -Sending)");
            Criteria criteria = search.criteria();
            
            /*
             * deletes objects found in the search engine but not in the wikitty storage
             */
            List<String> deleteList = new ArrayList<String>();
            PagedResult<String> wikittyIds = service.getSearchEngine()
                    .findAllByCriteria(criteria);
            
            for (String wikittyId : wikittyIds.getAll()) {
                boolean exists = service.getWikittyStorage().exists(wikittyId);
                
                if (!exists) {
                    deleteList.add(wikittyId);
                }
            }
            
            log.info("deleting: " + deleteList);
            service.getSearchEngine().delete(deleteList);
            
            // export
            String exportString = service.syncExportAllByCriteria(criteria);
    
            /*
             * transformation, change on bindings due to changes on infogene.
             */
            exportString = StringUtils.replace(exportString,
                    "<XmlFieldBinding.formField>Form.name</XmlFieldBinding.formField>",
                    "<XmlFieldBinding.formField>Infogene.objet</XmlFieldBinding.formField>");
            exportString = StringUtils.replace(exportString,
                    "<XmlFieldBinding.formField>Form.source</XmlFieldBinding.formField>",
                    "<XmlFieldBinding.formField>Infogene.sourceText</XmlFieldBinding.formField>");
            exportString = StringUtils.replace(exportString,
                    "<XmlFieldBinding.formField>Form.description</XmlFieldBinding.formField>",
                    "<XmlFieldBinding.formField>Infogene.description</XmlFieldBinding.formField>");
            exportString = StringUtils.replace(exportString,
                    "<XmlFieldBinding.formField>Form.initiateur</XmlFieldBinding.formField>",
                    "<XmlFieldBinding.formField>Infogene.entity</XmlFieldBinding.formField>");
            exportString = StringUtils.replace(exportString,
                    "<field>WIKITTY xmlFieldBinding[0-*] unique</field>",
                    "<field>STRING formTypeName unique</field>\n"
                            + "<field>WIKITTY xmlFieldBinding[0-*] unique</field>");
            
            // saves exported data to file
            Writer output = null;
            try {
                File exportFile = File.createTempFile("vradi_export_from_003_", null);
                output = new BufferedWriter(new FileWriter(exportFile));
                IOUtils.write(exportString, output);
                
                log.info("export objects from 0.0.3 to file: " + exportFile.getAbsolutePath());
                return exportFile;
                
            } finally {
                IOUtils.closeQuietly(output);
            }
        
        } catch (Exception e1) {
            throw new UnhandledException(e1);
        }
    }
    
    /**
     * Import objects from the specified <code>file</code> to the 0.0.4 database.
     * 
     * @param file the file to import
     * @param proxy the wikitty proxy
     */
    static void importObjects_004(File file, WikittyProxy proxy) {
        log.info("import objects to 0.0.4");
        String toURI = file.toURI().toString();
        proxy.syncImportFromUri(toURI);
    }
    
    /**
     * WikittyServiceJDBC_003.
     */
    private static class WikittyServiceJDBC_003 extends AbstractWikittyService {
        protected WikittySearchEngine searchEngine;
        protected WikittyExtensionStorage extensionStorage;
        protected WikittyStorage wikittyStorage;
        
        public WikittyServiceJDBC_003(String solrDataDir) {
            Properties properties = loadProperties();
            
            extensionStorage = new WikittyExtensionStorageJDBC(properties);
            wikittyStorage = new WikittyStorageJDBC(extensionStorage, properties);
            searchEngine = new WikittySearchEngineSolr(extensionStorage, solrDataDir);
        }
        
        Properties loadProperties() {
            try {
                Properties properties = new Properties();
                
                URL url = Resource.getURL("com/jurismarches/vradi/migration/jdbc003.properties");
                properties.load(url.openStream());

                return properties;
                
            } catch (Exception e) {
                throw new WikittyException(e);
            }
        }

        @Override
        protected WikittySearchEngine getSearchEngine() {
            return searchEngine;
        }
        @Override
        protected WikittyExtensionStorage getExtensionStorage() {
            return extensionStorage;
        }
        @Override
        protected WikittyStorage getWikittyStorage() {
            return wikittyStorage;
        }
    };
    
}
