package com.cybelia.sandra.web.action.tour;

import com.cybelia.sandra.entities.Etape;
import com.cybelia.sandra.entities.LigneProduit;
import com.cybelia.sandra.entities.Tour;
import com.cybelia.sandra.entities.Usine;
import com.cybelia.sandra.entities.trace.SuiviEtape;
import com.cybelia.sandra.entities.trace.SuiviTour;
import com.cybelia.sandra.entities.trace.SuiviUsine;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import org.apache.commons.lang3.tuple.Pair;

/**
 * Parcours la tournée avec les usines et les étapes
 *
 * @author julien
 */
public abstract class StepsHelper {


    public static void getStepsByEtapes(List<Pair<Etape, SuiviEtape>> etapes, DataSteps... listDataSteps) {
        if (etapes == null) {
            return;
        }
        for (Pair<Etape, SuiviEtape> etape : etapes) {
            Etape key = etape.getKey();
            Tour tour = key.getTour();
            for (DataSteps dataSteps : listDataSteps) {
                dataSteps.beginTour(tour);
            }
            for (DataSteps dataSteps : listDataSteps) {
                SuiviEtape suiviEtape = etape.getValue();
                if (suiviEtape != null) {
                    dataSteps.addDataStep(suiviEtape);
                } else {
                    dataSteps.addDataStep(key);
                }
            }
            for (DataSteps dataSteps : listDataSteps) {
                dataSteps.endTour(tour);
            }
        }
    }

    public static void getSteps(List<Tour> tours, DataSteps... listDataSteps) {
        if (tours == null) {
            return;
        }
        for (Tour tour : tours) {
            for (DataSteps dataSteps : listDataSteps) {
                dataSteps.beginTour(tour);
            }

            SuiviTour suiviTour = tour.getSuiviTour();
            if (suiviTour == null) {
                // Tour theorique
                getTheoreticalSteps(tour, listDataSteps);
            } else {
                // Tour reel
                getRealSteps(tour, listDataSteps);
            }

            for (DataSteps dataSteps : listDataSteps) {
                dataSteps.endTour(tour);
            }
        }
    }

    public static boolean getTheoreticalSteps(Tour tour, DataSteps... listDataSteps) {
        // Tour theorique
        List<Etape> steps = tour.getEtapes();
        Collections.sort(steps, new Comparator<Etape>() {
            public int compare(Etape etape, Etape other) {
                int order = etape.getOrdre();
                int otherOrder = other.getOrdre();
                return (order < otherOrder ? -1 : (order == otherOrder ? 0 : 1));
            }
        });

        List<Usine> factories = new ArrayList<Usine>();
        for (Etape step : steps) {
            List<LigneProduit> produits = step.getProduits();
            for (LigneProduit product : produits) {

                Usine factory = product.getUsine();
                if (!factories.contains(factory)) {
                    factories.add(factory);

                    for (DataSteps dataSteps : listDataSteps) {
                        dataSteps.addDataStep(factory);
                    }
                }
            }

            for (DataSteps dataSteps : listDataSteps) {
                dataSteps.addDataStep(step);
            }
        }

        return false;
    }

    public static void getRealSteps(Tour tour, DataSteps... listDataSteps) {

        // FIXME sletellier 20120124 : see if we must do the same for usine ?

        // Creations de la collection des etapes
        List<Etape> etapes = tour.getEtapes();

        // Creations de la collection des suivi etapes
        SuiviTour suiviTour = tour.getSuiviTour();
        List<SuiviEtape> reportingSteps = suiviTour.getSuiviEtapes();

        // merge with etapes
        for (SuiviEtape suiviEtape : reportingSteps) {
            etapes.remove(suiviEtape.getEtape());
        }

        List<SuiviUsine> reportingFactories = suiviTour.getSuiviUsines();

        List<Object> steps = new ArrayList<Object>(reportingSteps.size() + reportingFactories.size());

        steps.addAll(etapes);
        steps.addAll(reportingSteps);
        steps.addAll(reportingFactories);

        // Trie des etapes
        Collections.sort(steps, new Comparator<Object>() {
            @Override
            public int compare(Object o1, Object o2) {
                int order1 = 0;
                if (o1 instanceof Etape) {
                    order1 = ((Etape) o1).getOrdre();
                } else if (o1 instanceof SuiviEtape) {
                    order1 = ((SuiviEtape) o1).getNouvelleOrdre();
                } else if (o1 instanceof SuiviUsine) {
                    order1 = ((SuiviUsine) o1).getNouvelleOrdre();
                }

                int order2 = 0;
                if (o2 instanceof Etape) {
                    order2 = ((Etape) o2).getOrdre();
                } else if (o2 instanceof SuiviEtape) {
                    order2 = ((SuiviEtape) o2).getNouvelleOrdre();
                } else if (o2 instanceof SuiviUsine) {
                    order2 = ((SuiviUsine) o2).getNouvelleOrdre();
                }

                return (order1 < order2 ? -1 : (order1 == order2 ? 0 : 1));
            }
        });

        // Reparcours pour avoir les DataSteps
        for (Object step : steps) {
            if (step != null) {
                for (DataSteps dataSteps : listDataSteps) {
                    if (step instanceof Etape) {
                        dataSteps.addDataStep((Etape) step);
                    } else if (step instanceof SuiviEtape) {
                        dataSteps.addDataStep((SuiviEtape) step);
                    } else if (step instanceof SuiviUsine) {
                        dataSteps.addDataStep((SuiviUsine) step);
                    }
                }
            }
        }
    }

    public static abstract class DataSteps {
        private static final long serialVersionUID = 1L;

        protected Tour tour;

        protected void beginTour(Tour tour) {
            this.tour = tour;
        }

        protected void endTour(Tour tour) {
            this.tour = null;
        }

        protected abstract void addDataStep(Etape step);

        protected abstract void addDataStep(SuiviEtape step);

        protected abstract void addDataStep(SuiviUsine factory);

        protected abstract void addDataStep(Usine factory);
    }

    public static class DefaultDataSteps extends DataSteps {
        private static final long serialVersionUID = 1L;

        protected int number;
        protected List<DataStep> dataSteps = new ArrayList<DataStep>();

        @Override
        protected void beginTour(Tour tour) {
            super.beginTour(tour);
            this.number = 0;
        }

        @Override
        protected void endTour(Tour tour) {
            super.endTour(tour);
            this.number = 0;
        }

        public boolean add(DataStep dataStep) {
            if (dataStep != null) {
                dataStep.setNumber(number++);
                dataStep.setTour(tour);
                return dataSteps.add(dataStep);
            } else {
                return false;
            }
        }

        public List<DataStep> getList() {
            return dataSteps;
        }

        @Override
        protected void addDataStep(Etape step) {
            add(new DataStep<Etape>(step));
        }

        @Override
        protected void addDataStep(SuiviEtape step) {
            add(new DataStep<SuiviEtape>(step));
        }

        @Override
        protected void addDataStep(SuiviUsine factory) {
            add(new DataStep<SuiviUsine>(factory));
        }

        @Override
        protected void addDataStep(Usine factory) {
            add(new DataStep<Usine>(factory));
        }
    }

    public static class DataStep<T> {

        protected Tour tour;
        protected int number = 0;
        protected T step;

        public DataStep(T step) {
            this.step = step;
        }

        public int getNumber() {
            return number;
        }

        public void setNumber(int number) {
            this.number = number;
        }

        public T getStep() {
            return step;
        }

        public Tour getTour() {
            return tour;
        }

        public void setTour(Tour tour) {
            this.tour = tour;
        }

        public T getUsine() {
            if (step instanceof Usine) {
                return step;
            } else {
                return null;
            }
        }

        public T getEtape() {
            if (step instanceof Etape) {
                return step;
            } else {
                return null;
            }
        }

        public T getSuiviUsine() {
            if (step instanceof SuiviUsine) {
                return step;
            } else {
                return null;
            }
        }

        public T getSuiviEtape() {
            if (step instanceof SuiviEtape) {
                return step;
            } else {
                return null;
            }
        }
    }
}
