package com.cybelia.sandra.web.action.loading.model;

import com.cybelia.sandra.entities.Camion;
import com.cybelia.sandra.entities.ChargementUsineConfig;
import com.cybelia.sandra.entities.Etape;
import com.cybelia.sandra.entities.InfoChargement;
import com.cybelia.sandra.entities.LigneProduit;
import com.cybelia.sandra.entities.StatutEnum;
import com.cybelia.sandra.entities.Usine;
import com.cybelia.sandra.entities.trace.SuiviEtape;
import com.cybelia.sandra.entities.trace.SuiviUsine;
import com.cybelia.sandra.web.action.tour.StepsHelper.DataSteps;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Le modele du tableau affiche dans l'ecran de chargement
 *
 * @author chemit
 */
public class LoadingTable extends DataSteps {

    private static final long serialVersionUID = 1L;

    /**
     * log
     */
    protected Log log = LogFactory.getLog(getClass());

    /**
     * Unite entre la quantite et le temps
     */
    protected static float QUANTITY_TIME_UNIT = 10000;

    /**
     * la borne inferieure de la fenetre a afficher
     */
    protected Date startingDate;

    /**
     * la borne superieure de la fenetre a afficher
     */
    protected Date endingDate;

    /**
     * l'heure courante (null si pas dans la fenetre a afficher)
     */
    protected Date currentDate;

    /**
     * Decalage dans l'heure en pixel pour la date courante
     */
    protected int currentDateNbPixelForMinute;

    /**
     * le nombre de pixel d'une heure
     */
    protected int nbPixelPerHour;

    /**
     * les lignes du tables
     */
    protected Map<String, Column[]> rows;

    /**
     * les heures *
     */
    protected Column[] hours;

    /**
     * la config du chargement d'une usine
     */
    protected ChargementUsineConfig config;

    /**
     * les jours
     */
    protected Day[] days;

    public LoadingTable(Date currentDate) {
        this.currentDate = currentDate;
    }

    public void init(Date startingDate, Date endingDate, ChargementUsineConfig config) {
        this.startingDate = startingDate;
        this.endingDate = endingDate;
        //fixme : pas de valeur en dure...
        this.nbPixelPerHour = (int) ((float) 900 / (config.getNombreHeures()));
        this.config = config;

        if (currentDate.before(startingDate) || currentDate.after(endingDate)) {
            // currentDate is out of range
            currentDate = null;
            currentDateNbPixelForMinute = 0;
        } else {
            // Calcul du decalage dans l'heure
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(currentDate);
            currentDateNbPixelForMinute = nbPixelPerHour * calendar.get(Calendar.MINUTE) / 60;
        }
    }

    public void initRows() {
        days = Day.newDays(startingDate, endingDate);
        hours = ModelHelper.createColumnModel(config.getNombreHeures(), days, currentDate);
        rows = new LinkedHashMap<String, Column[]>();
    }

    public Date getStartingDate() {
        return startingDate;
    }

    public Date getEndingDate() {
        return endingDate;
    }

    public Date getCurrentDate() {
        return currentDate;
    }

    public int getNbPixelPerHour() {
        return nbPixelPerHour;
    }

    public Day[] getDays() {
        return days;
    }

    public Map<String, Column[]> getRows() {
        return rows;
    }

    public Column[] getHours() {
        return hours;
    }

    public int getCurrentDateNbPixelForMinute() {
        return currentDateNbPixelForMinute;
    }

    /* variable pour le visiteur */

    /**
     * chargement en cours
     */
    protected Chargement chargement;

    /**
     * camion en cours
     */
    protected Camion camion;

    /**
     * etape termine, statut venant du PDA
     */
    public static int ETAPE_STATUS_TERMINED = -1;

    /**
     * etape charge, statut venant de IBU
     */
    public static int ETAPE_STATUS_CHARGED = 40;

    /**
     * tour termine, statut venant du PDA
     */
    public static int TOUR_STATUS_TERMINED = -1;

    /**
     * tour en cours de livraison, statut venant du PDA
     */
    public static int TOUR_STATUS_DELIVERY = 4;

    /* visiteur */
    @Override
    protected void addDataStep(Etape step) {
        if (chargement != null) {
            boolean oneAvailable = false; //un produit disponible
            boolean allAvailable = true; //tous les produits disponibles
            boolean oneCharged = false; //un produit charges

            //calcul du temps
            chargement.setDelay(chargement.getDelay() + config.getTempFixeParLigneProduit());

            for (LigneProduit produit : step.getProduits()) {
                //calcul du temps
                Float delay = produit.getQuantiteACharger() / QUANTITY_TIME_UNIT;
                delay = delay * config.getTempVariableParQuantite();
                chargement.setDelay(chargement.getDelay() + delay.intValue());

                //date chargement réel
                InfoChargement info = produit.getInfoChargement();
                if (step.getEtat() > ETAPE_STATUS_CHARGED) {
                    Date date = info.getDate();
                    if (date != null) {
                        chargement.setStartingDate(date);
                    }
                }

                //calcul des statuts
                oneAvailable |= produit.getDisponible();
                allAvailable &= produit.getDisponible();
                oneCharged |= produit.getInfoChargement().getQuantite() > 0;
            }

            //calcul du decalage dans l'heure
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(chargement.getStartingDate());
            chargement.setNbPixelForMinute(nbPixelPerHour * calendar.get(Calendar.MINUTE) / 60);
            addChargement(camion.getTopiaId(), chargement);

            //mise a jour des statuts
            if (oneCharged) {
                setStatutChargement(chargement, StatutEnum.CHARGE);
            } else if (allAvailable) {
                setStatutChargement(chargement, StatutEnum.DISPONIBLE);
            } else if (oneAvailable) {
                setStatutChargement(chargement, StatutEnum.SEMI_DISPONIBLE);
            }
        }
    }

    @Override
    protected void addDataStep(Usine factory) {
        if (factory.getTopiaId().equals(config.getUsine().getTopiaId())) {
            camion = tour.getCamion();
            chargement = new Chargement();
            chargement.setCamionId(camion.getTopiaId());
            chargement.setTourId(tour.getTopiaId());
            chargement.setTourNumero(tour.getNumero());
            chargement.setStartingDate(tour.getDateChargement()); //date chargement planifie
            chargement.setStatut(StatutEnum.PLANIFIE); //par defaut
            chargement.setDelay(config.getTempFixeParTournee());
        } else {
            chargement = null;
        }
    }

    @Override
    protected void addDataStep(SuiviEtape step) {
        addDataStep(step.getEtape());

        if (chargement != null) {
            //etape termine
            if (step.getStatus() == ETAPE_STATUS_TERMINED) {
                setStatutChargement(chargement, StatutEnum.SEMI_LIVRE);
            }
        }
    }

    @Override
    protected void addDataStep(SuiviUsine factory) {
        addDataStep(factory.getUsine());

        if (chargement != null) {
            //date chargement PDA
            Date date = factory.getDateChargementPrevue();
            if (date != null && date.getTime() != 0) {
                chargement.setStartingDate(date);
            }

            //tour en livraison
            if (tour.getSuiviTour().getStatus() == TOUR_STATUS_DELIVERY) {
                setStatutChargement(chargement, StatutEnum.EN_COURS_LIVRAISON);
            }

            //tour termine
            if (tour.getSuiviTour().getStatus() == TOUR_STATUS_TERMINED) {
                setStatutChargement(chargement, StatutEnum.TERMINE);
            }
        }

    }

    protected void setStatutChargement(Chargement chargement, StatutEnum statut) {
        if (chargement.getStatut().getCode() < statut.getCode()) {
            chargement.setStatut(statut);
        }
    }

    protected void addChargement(String camionId, Chargement chargement) {
        Column[] columns = rows.get(camionId);

        // build for new camion, his ligne
        if (columns == null) {
            columns = ModelHelper.createColumnModel(config.getNombreHeures(), days, null);
            rows.put(camion.getTopiaId(), columns);
        }

        for (Column column : columns) {
            // find the matching column to receave this chargement
            Date sDate = column.getStartingDate();
            Date eDate = column.getEndingDate();
            Date date = chargement.getStartingDate();
            if ((sDate.equals(date) || sDate.before(date)) && eDate.after(date)) {
                column.setChargement(chargement);
                return;
            }
        }
    }
}
