package com.franciaflex.faxtomail.ui.swing;

/*
 * #%L
 * FaxToMail :: UI
 * %%
 * Copyright (C) 2014 Franciaflex
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.franciaflex.faxtomail.FaxToMailConfiguration;
import com.franciaflex.faxtomail.persistence.entities.Configuration;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailTopiaApplicationContext;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailTopiaPersistenceContext;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUser;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.services.DecoratorService;
import com.franciaflex.faxtomail.services.FaxToMailServiceContext;
import com.franciaflex.faxtomail.persistence.RessourceClassLoader;
import com.franciaflex.faxtomail.services.service.ConfigurationService;
import com.franciaflex.faxtomail.services.service.EmailService;
import com.franciaflex.faxtomail.services.service.InitFaxToMailService;
import com.franciaflex.faxtomail.services.service.MailFolderService;
import com.franciaflex.faxtomail.services.service.ReferentielService;
import com.franciaflex.faxtomail.services.service.UserService;
import com.franciaflex.faxtomail.services.service.ValidationService;
import com.franciaflex.faxtomail.ui.swing.content.MainUI;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeUIModel;
import com.franciaflex.faxtomail.ui.swing.content.search.SearchUIModel;
import com.franciaflex.faxtomail.ui.swing.util.UIMessageNotifier;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.swing.session.BeanDoubleListState;
import jaxx.runtime.swing.session.BeanFilterableComboBoxState;
import jaxx.runtime.swing.session.State;
import jaxx.runtime.swing.session.SwingSession;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractBean;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.ClassPathI18nInitializer;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.i18n.init.UserI18nInitializer;
import org.nuiton.jaxx.application.ApplicationConfiguration;
import org.nuiton.jaxx.application.ApplicationIOUtil;
import org.nuiton.jaxx.application.swing.ApplicationUIContext;
import org.nuiton.jaxx.application.swing.action.ApplicationActionEngine;
import org.nuiton.jaxx.application.swing.action.ApplicationActionFactory;
import org.nuiton.jaxx.application.swing.action.ApplicationActionUI;
import org.nuiton.jaxx.application.swing.util.ApplicationErrorHelper;
import org.nuiton.util.converter.ConverterUtil;

import javax.swing.*;

import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.List;

import static org.nuiton.i18n.I18n.t;

/**
 * UI application context.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class FaxToMailUIContext extends AbstractBean implements Closeable, UIMessageNotifier, ApplicationUIContext {

    /** Logger. */
    private static final Log log = LogFactory.getLog(FaxToMailUIContext.class);

    public static final String PROPERTY_SCREEN = "screen";

    public static final String PROPERTY_LOCALE = "locale";

    public static final Set<String> PROPERTIES_TO_SAVE = Sets.newHashSet(PROPERTY_LOCALE);

    public static final String PROPERTY_BUSY = "busy";

    public static final String PROPERTY_HIDE_BODY = "hideBody";

    /**
     * Application context (only one for all the application).
     *
     * @since 0.1
     */
    private static FaxToMailUIContext applicationContext;

    /**
     * Application global configuration.
     *
     * @since 0.1
     */
    protected final FaxToMailConfiguration config;

    /**
     * ClassLoader ressource.
     *
     * @since 0.3
     */
    protected final RessourceClassLoader resourceLoader;

    /**
     * Service context used by any service.
     *
     * @since 0.1
     */
    protected final FaxToMailServiceContext serviceContext;

    /**
     * Swing session used to save ui states.
     *
     * @since 0.1
     */
    protected final SwingSession swingSession;

    /**
     * Erro helper.
     *
     * @since 1.0
     */
    protected final ApplicationErrorHelper errorHelper;

    /**
     * Shared data context.
     *
     * @since 1.0.2
     */
//    protected TuttiDataContext dataContext;

    /**
     * Current screen displayed in ui.
     *
     * @since 0.1
     */
    protected FaxToMailScreen screen;

    /**
     * Current locale used in application.
     *
     * @since 1.0.3
     */
    protected Locale locale;

    /**
     * Busy state ({@code true} when a blocking action is running).
     *
     * @since 1.0.3
     */
    protected boolean busy;

    /**
     * Flag to hide (or not) the body of application.
     *
     * @since 1.1
     */
    protected boolean hideBody;

    /**
     * Message notifiers.
     *
     * @since 0.3
     */
    protected final Set<UIMessageNotifier> messageNotifiers;

    private MainUI mainUI;

    private ApplicationActionUI actionUI;

    private JFrame secondaryFrame;

    /**
     * Flag to know if there is an exsiting db.
     *
     * @since 1.0
     */
    private boolean dbExist;

    /**
     * Flag to know if there is a loaded db.
     *
     * @since 1.0
     */
    private boolean dbLoaded;

    private final ApplicationActionFactory faxToMailActionFactory;

    private final ApplicationActionEngine faxToMailActionEngine;

    protected Configuration faxToMailConfiguration;

    protected FaxToMailUser currentUser;

    protected MailFolder currentMailFolder;

    protected DemandeUIModel currentEmail;

    protected List<MailFolder> expandedFolders = new ArrayList<MailFolder>();

    protected SearchUIModel search;

    public static FaxToMailUIContext newContext(FaxToMailConfiguration config) {
        Preconditions.checkNotNull(config);
        Preconditions.checkState(applicationContext == null,
                                 "Application context was already opened!");
        applicationContext = new FaxToMailUIContext(config);
        return applicationContext;
    }

    public static FaxToMailUIContext getApplicationContext() {
        return applicationContext;
    }

    public ApplicationErrorHelper getErrorHelper() {
        return applicationContext.errorHelper;
    }

    @Override
    public String getI18nPrefix() {
        return "faxtomail.property.";
    }

    @Override
    public String getDateFormat() {
        return getConfig().getDateFormat();
    }

    protected FaxToMailUIContext(FaxToMailConfiguration config) {
        this.config = config;
        this.resourceLoader = new RessourceClassLoader(Thread.currentThread().getContextClassLoader());
        this.serviceContext = new FaxToMailServiceContext();
        FaxToMailTopiaApplicationContext topiaApplicationContext = new FaxToMailTopiaApplicationContext(config.getTopiaProperties());
        FaxToMailTopiaPersistenceContext persistenceContext = topiaApplicationContext.newPersistenceContext();
        serviceContext.setPersistenceContext(persistenceContext);
        serviceContext.setApplicationConfig(config);

        InitFaxToMailService initExtranetService =
                serviceContext.newService(InitFaxToMailService.class);

        initExtranetService.init();

        Map<Class, State> additionalStates = Maps.newHashMap();
        additionalStates.put(BeanFilterableComboBox.class, new BeanFilterableComboBoxState());
        additionalStates.put(BeanDoubleList.class, new BeanDoubleListState());
        this.swingSession = SwingSession.newSession(getConfig().getUIConfigFile(), false, additionalStates);

        this.errorHelper = new ApplicationErrorHelper(this) {

            @Override
            public void showWarningDialog(String message) {

                JOptionPane.showMessageDialog(context.getActionUI(), "<html><body>" + message + "</body></html>",
                                              t("application.error.ui.business.warning"),
                                              JOptionPane.WARNING_MESSAGE);
            }

            @Override
            public void showErrorDialog(String message, Throwable cause) {
                super.showErrorDialog(message, cause);
            }
        };

        UIMessageNotifier logMessageNotifier = new UIMessageNotifier() {

            @Override
            public void showInformationMessage(String message) {
                if (StringUtils.isNotBlank(message)) {
                    message = message.replaceAll("\\<strong\\>", "");
                    message = message.replaceAll("\\<.strong\\>", "");
                    message = message.replaceAll("\\<li\\>", "");
                    message = message.replaceAll("\\<.li\\>", "");
                    message = message.replaceAll("\\<ul\\>", "");
                    message = message.replaceAll("\\<.ul\\>", "");
                    if (log.isInfoEnabled()) {
                        log.info(message);
                    }
                }
            }
        };

        this.messageNotifiers = Sets.newHashSet();
        addMessageNotifier(logMessageNotifier);

        faxToMailActionFactory = new FaxToMailActionFactory();
        faxToMailActionEngine = new ApplicationActionEngine(faxToMailActionFactory);
    }

    @Override
    public ApplicationConfiguration getConfiguration() {
        return config;
    }

    @Override
    public Component getBodyUI() {
        MainUI mainUI = getMainUI();
        return mainUI == null ? null : mainUI.getBody();
    }

    @Override
    public Component getStatusUI() {
        MainUI mainUI = getMainUI();
        return mainUI == null ? null : mainUI.getStatus();
    }

    //------------------------------------------------------------------------//
    //-- Open / close methods                                               --//
    //------------------------------------------------------------------------//

    public void init() {

        config.prepareDirectories();

        // use our special classLoader (which will read some files from resources from a configuration directory)
        Thread.currentThread().setContextClassLoader(getResourceLoader());

        // converters are stored in current classloader, we need then to rescan them
        // each time we change current classloader
        ConverterUtil.deregister();
        ConverterUtil.initConverters();

        // Use shutdownHook to close context on System.exit
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {

            @Override
            public void run() {
                if (log.isInfoEnabled()) {
                    log.info("Close context on shutdown");
                }
                close();
            }
        }));

        //--------------------------------------------------------------------//
        // init i18n
        //--------------------------------------------------------------------//

        Locale i18nLocale = config.getI18nLocale();

        if (log.isInfoEnabled()) {
            log.info(String.format("Starts i18n with locale [%s]", i18nLocale));
        }
        
        I18n.init(new ClassPathI18nInitializer(), i18nLocale);

//        //--------------------------------------------------------------------//
//        // init help
//        //--------------------------------------------------------------------//
//
//        File helpDirectory = config.getHelpDirectory();
//
//        if (!config.isFullLaunchMode()) {
//
//            if (!helpDirectory.exists()) {
//                helpDirectory = new File(config.getDataDirectory(), "help");
//            }
//        }
//
//        if (log.isDebugEnabled()) {
//            log.debug("Help directory: " + helpDirectory);
//        }
//        TuttiIOUtil.forceMkdir(
//                helpDirectory,
//                _("tutti.help.mkDir.error", helpDirectory));
//
//        // load help mapping
//        String mappingProperties =
//                "/tutti-help-" + i18nLocale.getLanguage() + ".properties";
//        try {
//
//            InputStream resourceAsStream =
//                    getClass().getResourceAsStream(mappingProperties);
//            helpMapping = new Properties();
//            helpMapping.load(resourceAsStream);
//
//        } catch (Exception eee) {
//            log.error("Failed to load help mapping file at '" +
//                      mappingProperties + "'", eee);
//        }
//        if (log.isInfoEnabled()) {
//            log.info(String.format("Starts help with locale at [%s]",
//                                   helpDirectory));
//        }
//

//        Company franciaflex = getCompanyService().getCompanyByName("Franciaflex");
//        setCurrentCompany(franciaflex);

        FaxToMailUser currentUser = getUserService().getUserByLogin("cbaillet");
        setCurrentUser(currentUser);

        Configuration config = getConfigurationService().getConfiguration();
        setFaxToMailConfiguration(config);

//        //--------------------------------------------------------------------//
//        // init action UI
//        //--------------------------------------------------------------------//
        setActionUI(new ApplicationActionUI(null, this));
    }

    public void open() {

        setLocale(config.getI18nLocale());

//        boolean dbExists = config.isDbExists();
//
//        setDbExist(dbExists);

        // save back to config
        saveContextToConfig();

        // list when programId or campaingId change to save the configuration
        addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                if (PROPERTIES_TO_SAVE.contains(evt.getPropertyName())) {
                    saveContextToConfig();
                }
            }
        });
    }

    @Override
    public void close() {

//        // Clear data references
        messageNotifiers.clear();
//        validationContext = null;
//        IOUtils.closeQuietly(dataContext);

        setScreen(null);

//        IOUtils.closeQuietly(serviceContext);

//        // remove listeners
        PropertyChangeListener[] listeners = getPropertyChangeListeners();
        for (PropertyChangeListener listener : listeners) {
            if (log.isDebugEnabled()) {
                log.debug("Remove listener: " + listener);
            }
            removePropertyChangeListener(listener);
        }
        setMainUI(null);
        if (actionUI != null) {

            // close action ui
            actionUI.getModel().clear();
        }
        setActionUI(null);
    }

    //------------------------------------------------------------------------//
    //-- Service methods                                                    --//
    //------------------------------------------------------------------------//

    public DecoratorService getDecoratorService() {
        return serviceContext.newService(DecoratorService.class);
    }

    public ValidationService getValidationService() {
        return serviceContext.newService(ValidationService.class);
    }

    public MailFolderService getMailFolderService() {
        return serviceContext.newService(MailFolderService.class);
    }

    public UserService getUserService() {
        return serviceContext.newService(UserService.class);
    }

    public EmailService getEmailService() {
        return serviceContext.newService(EmailService.class);
    }

    public ReferentielService getReferentielService() {
        return serviceContext.newService(ReferentielService.class);
    }

    public ConfigurationService getConfigurationService() {
        return serviceContext.newService(ConfigurationService.class);
    }

    //------------------------------------------------------------------------//
    //-- DataContext methods                                                --//
    //------------------------------------------------------------------------//

    //------------------------------------------------------------------------//
    //-- Config methods                                                     --//
    //------------------------------------------------------------------------//

    public FaxToMailConfiguration getConfig() {
        return config;
    }

    protected void saveContextToConfig() {
        config.setI18nLocale(getLocale());
        config.save();
    }

    //------------------------------------------------------------------------//
    //-- UI methods                                                         --//
    //------------------------------------------------------------------------//

    public MainUI getMainUI() {
        return mainUI;
    }

    public void setMainUI(MainUI mainUI) {
        this.mainUI = mainUI;
    }

    public JFrame getSecondaryFrame() {
        return secondaryFrame;
    }

    public void setSecondaryFrame(JFrame secondaryFrame) {
        this.secondaryFrame = secondaryFrame;
    }

    @Override
    public ApplicationActionUI getActionUI() {
        return actionUI;
    }

    @Override
    public ApplicationActionUI getExistingActionUI() {
        while (actionUI == null) {

            try {
                Thread.sleep(50);
            } catch (InterruptedException e) {
                // ignore this one
            }
        }
        return actionUI;
    }

    public void setActionUI(ApplicationActionUI actionUI) {
        this.actionUI = actionUI;
    }

    public SwingSession getSwingSession() {
        return swingSession;
    }

    public FaxToMailScreen getScreen() {
        return screen;
    }

    public void setScreen(FaxToMailScreen screen) {
        Object oldValue = getScreen();
        this.screen = screen;
        firePropertyChange(PROPERTY_SCREEN, oldValue, screen);
    }

    @Override
    public boolean isBusy() {
        return busy;
    }

    @Override
    public void setBusy(boolean busy) {
        this.busy = busy;
        firePropertyChange(PROPERTY_BUSY, null, busy);
    }

    @Override
    public boolean isHideBody() {
        return hideBody;
    }

    @Override
    public void setHideBody(boolean hideBody) {
        this.hideBody = hideBody;
        firePropertyChange(PROPERTY_HIDE_BODY, null, hideBody);
    }

    @Override
    public Color getColorBlockingLayer() {
        return getConfig().getColorBlockingLayer();
    }

    @Override
    public ApplicationActionFactory getActionFactory() {
        return faxToMailActionFactory;
    }

    @Override
    public ApplicationActionEngine getActionEngine() {
        return faxToMailActionEngine;
    }
    //------------------------------------------------------------------------//
    //-- UIMessageNotifier methods                                          --//
    //------------------------------------------------------------------------//

    public void addMessageNotifier(UIMessageNotifier messageNotifier) {
        this.messageNotifiers.add(messageNotifier);
    }

    public void removeMessageNotifier(UIMessageNotifier messageNotifier) {
        this.messageNotifiers.remove(messageNotifier);
    }

    @Override
    public void showInformationMessage(String message) {
        for (UIMessageNotifier messageNotifier : messageNotifiers) {
            messageNotifier.showInformationMessage(message);
        }
    }

    public RessourceClassLoader getResourceLoader() {
        return resourceLoader;
    }

    //------------------------------------------------------------------------//
    //-- Other methods                                                      --//
    //------------------------------------------------------------------------//

    public Locale getLocale() {
        return locale;
    }

    public void setLocale(Locale locale) {
        this.locale = locale;
        firePropertyChange(PROPERTY_LOCALE, null, locale);
    }

    public Configuration getFaxToMailConfiguration() {
        return faxToMailConfiguration;
    }

    public void setFaxToMailConfiguration(Configuration faxToMailConfiguration) {
        this.faxToMailConfiguration = faxToMailConfiguration;
    }

    public FaxToMailUser getCurrentUser() {
        return currentUser;
    }

    public void setCurrentUser(FaxToMailUser currentUser) {
        this.currentUser = currentUser;
    }

    public MailFolder getCurrentMailFolder() {
        return currentMailFolder;
    }

    public void setCurrentMailFolder(MailFolder currentMailFolder) {
        this.currentMailFolder = currentMailFolder;
    }

    public DemandeUIModel getCurrentEmail() {
        return currentEmail;
    }

    public void setCurrentEmail(DemandeUIModel currentEmail) {
        this.currentEmail = currentEmail;
    }

    public List<MailFolder> getExpandedFolders() {
        return expandedFolders;
    }

    public void setExpandedFolders(List<MailFolder> expandedFolders) {
        this.expandedFolders = expandedFolders;
    }

    public SearchUIModel getSearch() {
        return search;
    }

    public void setSearch(SearchUIModel search) {
        this.search = search;
    }
}
