/*
 * Decompiled with CFR 0.152.
 */
package com.azure.data.appconfiguration.implementation;

import com.azure.core.util.BinaryData;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.logging.ClientLogger;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import reactor.core.Exceptions;

public class ConfigurationClientCredentials {
    private final ClientLogger logger = new ClientLogger(ConfigurationClientCredentials.class);
    private static final ByteBuffer EMPTY_BYTE_BUFFER = ByteBuffer.allocate(0);
    private static final String HOST_HEADER = "Host";
    private static final String DATE_HEADER = "Date";
    private static final String CONTENT_HASH_HEADER = "x-ms-content-sha256";
    private static final String[] SIGNED_HEADERS = new String[]{"Host", "Date", "x-ms-content-sha256"};
    private static final String AUTHORIZATION_HEADER = "Authorization";
    private final CredentialInformation credentials;
    private final AuthorizationHeaderProvider headerProvider;

    public ConfigurationClientCredentials(String connectionString) throws InvalidKeyException, NoSuchAlgorithmException {
        this.credentials = new CredentialInformation(connectionString);
        this.headerProvider = new AuthorizationHeaderProvider(this.credentials);
    }

    public String getBaseUri() {
        return this.credentials.baseUri().toString();
    }

    Map<String, String> getAuthorizationHeaders(URL url, String httpMethod, BinaryData binaryData) {
        MessageDigest messageDigest;
        ByteBuffer byteBuffer = binaryData == null ? EMPTY_BYTE_BUFFER : binaryData.toByteBuffer();
        try {
            messageDigest = MessageDigest.getInstance("SHA-256");
        }
        catch (NoSuchAlgorithmException e) {
            throw this.logger.logExceptionAsError(Exceptions.propagate((Throwable)e));
        }
        messageDigest.update(byteBuffer);
        return this.headerProvider.getAuthenticationHeaders(url, httpMethod, messageDigest);
    }

    private static class CredentialInformation {
        private static final String ENDPOINT = "endpoint=";
        private static final String ID = "id=";
        private static final String SECRET = "secret=";
        private final URL baseUri;
        private final String id;
        private final byte[] secret;

        URL baseUri() {
            return this.baseUri;
        }

        String id() {
            return this.id;
        }

        byte[] secret() {
            return this.secret;
        }

        CredentialInformation(String connectionString) {
            if (CoreUtils.isNullOrEmpty((CharSequence)connectionString)) {
                throw new IllegalArgumentException("'connectionString' cannot be null or empty.");
            }
            String[] args = connectionString.split(";");
            if (args.length < 3) {
                throw new IllegalArgumentException("invalid connection string segment count");
            }
            URL baseUri = null;
            String id = null;
            byte[] secret = null;
            for (String arg : args) {
                String segment = arg.trim();
                String lowerCase = segment.toLowerCase(Locale.US);
                if (lowerCase.startsWith(ENDPOINT)) {
                    try {
                        baseUri = new URL(segment.substring(ENDPOINT.length()));
                        continue;
                    }
                    catch (MalformedURLException ex) {
                        throw new IllegalArgumentException(ex);
                    }
                }
                if (lowerCase.startsWith(ID)) {
                    id = segment.substring(ID.length());
                    continue;
                }
                if (!lowerCase.startsWith(SECRET)) continue;
                String secretBase64 = segment.substring(SECRET.length());
                secret = Base64.getDecoder().decode(secretBase64);
            }
            this.baseUri = baseUri;
            this.id = id;
            this.secret = secret;
            if (this.baseUri == null || CoreUtils.isNullOrEmpty((CharSequence)this.id) || this.secret == null) {
                throw new IllegalArgumentException("Could not parse 'connectionString'. Expected format: 'endpoint={endpoint};id={id};secret={secret}'. Actual:" + connectionString);
            }
        }
    }

    private static class AuthorizationHeaderProvider {
        private final String signedHeadersValue = String.join((CharSequence)";", ConfigurationClientCredentials.access$100());
        private static final String HMAC_SHA256 = "HMAC-SHA256 Credential=%s&SignedHeaders=%s&Signature=%s";
        private final CredentialInformation credentials;
        private final Mac sha256HMAC;

        AuthorizationHeaderProvider(CredentialInformation credentials) throws NoSuchAlgorithmException, InvalidKeyException {
            this.credentials = credentials;
            this.sha256HMAC = Mac.getInstance("HmacSHA256");
            this.sha256HMAC.init(new SecretKeySpec(credentials.secret(), "HmacSHA256"));
        }

        private Map<String, String> getAuthenticationHeaders(URL url, String httpMethod, MessageDigest messageDigest) {
            HashMap<String, String> headers = new HashMap<String, String>();
            String contentHash = Base64.getEncoder().encodeToString(messageDigest.digest());
            headers.put(ConfigurationClientCredentials.HOST_HEADER, url.getHost());
            headers.put(ConfigurationClientCredentials.CONTENT_HASH_HEADER, contentHash);
            if (headers.get(ConfigurationClientCredentials.DATE_HEADER) == null) {
                String utcNow = OffsetDateTime.now(ZoneOffset.UTC).format(DateTimeFormatter.RFC_1123_DATE_TIME);
                headers.put(ConfigurationClientCredentials.DATE_HEADER, utcNow);
            }
            this.addSignatureHeader(url, httpMethod, headers);
            return headers;
        }

        private void addSignatureHeader(URL url, String httpMethod, Map<String, String> httpHeaders) {
            String pathAndQuery = url.getPath();
            if (url.getQuery() != null) {
                pathAndQuery = pathAndQuery + '?' + url.getQuery();
            }
            String signed = Arrays.stream(SIGNED_HEADERS).map(httpHeaders::get).collect(Collectors.joining(";"));
            String stringToSign = httpMethod.toUpperCase(Locale.US) + "\n" + pathAndQuery + "\n" + signed;
            String signature = Base64.getEncoder().encodeToString(this.sha256HMAC.doFinal(stringToSign.getBytes(StandardCharsets.UTF_8)));
            httpHeaders.put(ConfigurationClientCredentials.AUTHORIZATION_HEADER, String.format(HMAC_SHA256, this.credentials.id(), this.signedHeadersValue, signature));
        }
    }
}

