

/*
 * Class:        OrnsteinUhlenbeckProcess
 * Description:  
 * Environment:  Java
 * Software:     SSJ 
 * Copyright (C) 2001  Pierre L'Ecuyer and Université de Montréal
 * Organization: DIRO, Université de Montréal
 * @author       
 * @since

 * SSJ is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License (GPL) as published by the
 * Free Software Foundation, either version 3 of the License, or
 * any later version.

 * SSJ is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * A copy of the GNU General Public License is available at
   <a href="http://www.gnu.org/licenses">GPL licence site</a>.
 */

package umontreal.iro.lecuyer.stochprocess;
import umontreal.iro.lecuyer.rng.*;
import umontreal.iro.lecuyer.probdist.*;
import umontreal.iro.lecuyer.randvar.*;



/**
 * This class represents an <SPAN  CLASS="textit">Ornstein-Uhlenbeck</SPAN> process
 *  
 * <SPAN CLASS="MATH">{<I>X</I>(<I>t</I>) : <I>t</I>&nbsp;&gt;=&nbsp;0}</SPAN>, sampled at times 
 * <SPAN CLASS="MATH">0 = <I>t</I><SUB>0</SUB> &lt; <I>t</I><SUB>1</SUB> &lt; <SUP> ... </SUP> &lt; <I>t</I><SUB>d</SUB></SPAN>.
 * This process obeys the stochastic differential equation
 * 
 * <P></P>
 * <DIV ALIGN="CENTER" CLASS="mathdisplay"><A NAME="eq:ornstein"></A>
 * <I>dX</I>(<I>t</I>) = <I>&#945;</I>(<I>b</I> - <I>X</I>(<I>t</I>))<I>dt</I> + <I>&#963;</I>&nbsp;<I>dB</I>(<I>t</I>)
 * </DIV><P></P>
 * with initial condition <SPAN CLASS="MATH"><I>X</I>(0) = <I>x</I><SUB>0</SUB></SPAN>, 
 * where <SPAN CLASS="MATH"><I>&#945;</I></SPAN>, <SPAN CLASS="MATH"><I>b</I></SPAN> and <SPAN CLASS="MATH"><I>&#963;</I></SPAN> are positive constants,
 * and 
 * <SPAN CLASS="MATH">{<I>B</I>(<I>t</I>),&nbsp;<I>t</I>&nbsp;&gt;=&nbsp;0}</SPAN> is a standard Brownian motion
 * (with drift 0 and volatility 1).
 * This process is <SPAN  CLASS="textit">mean-reverting</SPAN> in the sense that it always tends to
 * drift toward its general mean <SPAN CLASS="MATH"><I>b</I></SPAN>.
 * The process is generated using the sequential technique
 * 
 * <P></P>
 * <DIV ALIGN="CENTER" CLASS="mathdisplay"><A NAME="eq:ornstein-seq"></A>
 * <I>X</I>(<I>t</I><SUB>j</SUB>) = <I>e</I><SUP>-<I>&#945;</I>(t<SUB>j</SUB>-t<SUB>j-1</SUB>)</SUP><I>X</I>(<I>t</I><SUB>j-1</SUB>) + <I>b</I>(1 - <I>e</I><SUP>-<I>&#945;</I>(t<SUB>j</SUB>-t<SUB>j-1</SUB>)</SUP>) + <I>&#963;</I>(1 - e^-2&alpha;(t_j - t_j-1)2&alpha;)<SUP>1/2</SUP>&nbsp;<I>Z</I><SUB>j</SUB>
 * </DIV><P></P>
 * where 
 * <SPAN CLASS="MATH"><I>Z</I><SUB>j</SUB>&#8764;<I>N</I>(0, 1)</SPAN>. The time intervals 
 * <SPAN CLASS="MATH"><I>t</I><SUB>j</SUB> - <I>t</I><SUB>j-1</SUB></SPAN>
 * can be arbitrarily large.
 * 
 */
public class OrnsteinUhlenbeckProcess extends StochasticProcess  {
    protected NormalGen    gen;
    protected double       alpha,
                           beta,
                           sigma;
    // Precomputed values 
    protected double[]     badt,
                           alphadt,
                           sigmasqrdt;



   /**
    * Constructs a new <TT>OrnsteinUhlenbeckProcess</TT> with parameters
    *  <SPAN CLASS="MATH"><I>&#945;</I> =</SPAN> <TT>alpha</TT>, <SPAN CLASS="MATH"><I>b</I></SPAN>, <SPAN CLASS="MATH"><I>&#963;</I> =</SPAN> <TT>sigma</TT> and initial value
    * 
    * <SPAN CLASS="MATH"><I>X</I>(<I>t</I><SUB>0</SUB>) =</SPAN> <TT>x0</TT>. The normal variates <SPAN CLASS="MATH"><I>Z</I><SUB>j</SUB></SPAN> will be 
    * generated by inversion using the stream <TT>stream</TT>.
    * 
    */
   public OrnsteinUhlenbeckProcess (double x0, double alpha, double b,
                                    double sigma, RandomStream stream) {
        this (x0, alpha, b, sigma, new NormalGen (stream));
    }


   /**
    * Here, the normal variate generator is specified directly
    * instead of specifying the stream.
    * The normal generator <TT>gen</TT> can use another method than inversion.
    * 
    */
   public OrnsteinUhlenbeckProcess (double x0, double alpha, double b,
                                    double sigma, NormalGen gen)  {
      this.alpha = alpha;
      this.beta  = b;
      this.sigma = sigma;
      this.x0    = x0;
      this.gen   = gen;
   }


   public double nextObservation() {
        double xOld = path[observationIndex];
        double x = badt[observationIndex] + xOld * alphadt[observationIndex]
                   + sigmasqrdt[observationIndex] * gen.nextDouble();
        observationIndex++;
        path[observationIndex] = x;
        return x;
    }

   /**
    * Generates and returns the next observation at time <SPAN CLASS="MATH"><I>t</I><SUB>j+1</SUB> =</SPAN>
    *  <TT>nextTime</TT>, using the previous observation time <SPAN CLASS="MATH"><I>t</I><SUB>j</SUB></SPAN> defined earlier 
    * (either by this method or by <TT>setObservationTimes</TT>), 
    * as well as the value of the previous observation <SPAN CLASS="MATH"><I>X</I>(<I>t</I><SUB>j</SUB>)</SPAN>. 
    * <SPAN  CLASS="textit">Warning</SPAN>: This method will reset the observations time <SPAN CLASS="MATH"><I>t</I><SUB>j+1</SUB></SPAN>
    * for this process to <TT>nextTime</TT>. The user must make sure that
    * the <SPAN CLASS="MATH"><I>t</I><SUB>j+1</SUB></SPAN> supplied is 
    * <SPAN CLASS="MATH">&nbsp;&gt;=&nbsp;<I>t</I><SUB>j</SUB></SPAN>.
    * 
    */
   public double nextObservation (double nextTime)  {
        double previousTime = t[observationIndex];
        double xOld = path[observationIndex];
        observationIndex++;
        t[observationIndex] = nextTime;
        double dt = nextTime - previousTime;
        double tem = Math.exp(-alpha * dt);
        double tem1 = -Math.expm1(-alpha * dt);
        double x = tem*xOld + beta*tem1 + sigma *
            Math.sqrt(tem1*(1.0 + tem)/(2.0*alpha)) * gen.nextDouble();
        path[observationIndex] = x;
        return x;
    }


   /**
    * Generates an observation of the process in <TT>dt</TT> time units,
    * assuming that the process has value <SPAN CLASS="MATH"><I>x</I></SPAN> at the current time.
    * Uses the process parameters specified in the constructor.
    * Note that this method does not affect the sample path of the process 
    * stored internally (if any).
    * 
    * 
    */
   public double nextObservation (double x, double dt)  {
        double tem = Math.exp(-alpha * dt);
        double tem1 = -Math.expm1(-alpha * dt);
        x = tem*x + beta*tem1 + sigma *
            Math.sqrt(tem1*(1.0 + tem)/(2.0*alpha)) * gen.nextDouble();
        return x;
    }


   public double[] generatePath() {
        double x;
        double xOld = x0;
        for (int j = 0; j < d; j++) {
            x = badt[j] + xOld * alphadt[j] + sigmasqrdt[j] * gen.nextDouble();
            path[j + 1] = x;
            xOld = x;
        }
        observationIndex = d;
        return path;
    }

   public double[] generatePath (RandomStream stream) {
        gen.setStream (stream);
        return generatePath();
    }


   /**
    * Resets the parameters 
    * <SPAN CLASS="MATH"><I>X</I>(<I>t</I><SUB>0</SUB>) =</SPAN> <TT>x0</TT>, <SPAN CLASS="MATH"><I>&#945;</I> =</SPAN> <TT>alpha</TT>,
    *  <SPAN CLASS="MATH"><I>b</I> =</SPAN> <TT>b</TT> and <SPAN CLASS="MATH"><I>&#963;</I> =</SPAN> <TT>sigma</TT> of the process. 
    * <SPAN  CLASS="textit">Warning</SPAN>: This method will recompute some quantities stored internally, 
    * which may be slow if called too frequently.
    * 
    */
   public void setParams (double x0, double alpha, double b, double sigma)  { 
        this.alpha = alpha;
        this.beta  = b;
        this.sigma = sigma;
        this.x0    = x0;
        if (observationTimesSet) init(); // Otherwise not needed.
    }


   /**
    * Resets the random stream of the normal generator to <TT>stream</TT>.
    * 
    */
   public void setStream (RandomStream stream)  { gen.setStream (stream); }


   /**
    * Returns the random stream of the normal generator.
    * 
    */
   public RandomStream getStream ()  { return gen.getStream (); }


   /**
    * Returns the value of <SPAN CLASS="MATH"><I>&#945;</I></SPAN>.
    * 
    */
   public double getAlpha()  { return alpha; }


   /**
    * Returns the value of <SPAN CLASS="MATH"><I>b</I></SPAN>.
    * 
    */
   public double getB()  { return beta; }


   /**
    * Returns the value of <SPAN CLASS="MATH"><I>&#963;</I></SPAN>.
    * 
    */
   public double getSigma()  { return sigma; }


   /**
    * Returns the normal random variate generator used.
    * The <TT>RandomStream</TT> used for that generator can be changed via 
    * <TT>getGen().setStream(stream)</TT>, for example.
    * 
    */
   public NormalGen getGen()  { return gen; }
 

   protected void initArrays(int d) {
      double dt, tem, tem1;
      for (int j = 0; j < d; j++) {
         dt = t[j+1] - t[j];
         tem = Math.exp(-alpha * dt);
         tem1 = -Math.expm1(-alpha * dt);
         badt[j] = beta*tem1;
         alphadt[j] = tem;
         sigmasqrdt[j] = sigma * Math.sqrt (tem1*(1.0 + tem)/(2.0*alpha));
      }
   }

    // This is called by setObservationTimes to precompute constants
    // in order to speed up the path generation.
    protected void init() {
        super.init();
        badt = new double[d];
        alphadt = new double[d];
        sigmasqrdt = new double[d];
        initArrays(d);
     }

} 
