

/*
 * Class:        GeometricGen
 * Description:  random variate generator for the geometric distribution
 * Environment:  Java
 * Software:     SSJ 
 * Copyright (C) 2001  Pierre L'Ecuyer and Université de Montréal
 * Organization: DIRO, Université de Montréal
 * @author       
 * @since

 * SSJ is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License (GPL) as published by the
 * Free Software Foundation, either version 3 of the License, or
 * any later version.

 * SSJ is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * A copy of the GNU General Public License is available at
   <a href="http://www.gnu.org/licenses">GPL licence site</a>.
 */

package umontreal.iro.lecuyer.randvar;
import umontreal.iro.lecuyer.rng.*;
import umontreal.iro.lecuyer.probdist.*;


/**
 * This class implements a random variate generator for the 
 * <EM>geometric</EM> distribution. Its has parameter <SPAN CLASS="MATH"><I>p</I></SPAN> and mass function
 * 
 * <P></P>
 * <DIV ALIGN="CENTER" CLASS="mathdisplay">
 * <I>p</I>(<I>x</I>) = <I>p</I>(1 - <I>p</I>)<SUP>x</SUP> for <I>x</I> = 0, 1, 2,...,
 * </DIV><P></P>
 * where 
 * <SPAN CLASS="MATH">0&nbsp;&lt;=&nbsp;<I>p</I>&nbsp;&lt;=&nbsp;1</SPAN>.
 * Random variates are generated by calling inversion on the distribution object.
 * 
 */
public class GeometricGen extends RandomVariateGenInt  {
   private double p;



   /**
    * Creates a geometric random variate generator with
    *   parameter <SPAN CLASS="MATH"><I>p</I></SPAN>, using stream <TT>s</TT>.
    * 
    */
   public GeometricGen (RandomStream s, double p)  {
      super (s, new GeometricDist(p));
      setParams (p);
   }


   /**
    * Creates a new generator for the distribution <TT>dist</TT>, using 
    *     stream <TT>s</TT>.
    * 
    */
   public GeometricGen (RandomStream s, GeometricDist dist)  {
      super (s, dist);
      if (dist != null)
         setParams (dist.getP());
   }
 
    
   public int nextInt () {
      return GeometricDist.inverseF (p, stream.nextDouble());
   }

   /**
    * Generates a <EM>geometric</EM> random variate with parameter
    *   <SPAN CLASS="MATH"><I>p</I> =</SPAN>&nbsp;<TT>p</TT>, using stream <TT>s</TT>, by inversion.
    * 
    */
   public static int nextInt (RandomStream s, double p)  {
      return GeometricDist.inverseF (p, s.nextDouble());
   }


   /**
    * Returns the parameter <SPAN CLASS="MATH"><I>p</I></SPAN> of this object.
    * 
    * 
    */
   public double getP() {
      return p;
   }


   /**
    * Sets the parameter <SPAN CLASS="MATH"><I>n</I></SPAN> and <SPAN CLASS="MATH"><I>p</I></SPAN> of this object.
    * 
    */
   protected void setParams (double p) {
      if (p < 0.0 || p > 1.0)
         throw new IllegalArgumentException ("p not in range [0, 1]");
      this.p = p;
   }

}
