package org.planx.xpath.expr.axis;

import org.planx.xpath.Navigator;
import org.planx.xpath.XPathException;


/**
 * Descendant axis and descendant-or-self axis.
 **/
public class DescendantAxis extends Axis {
    private final boolean includeSelf;

    /**
     * Create a descendant axis.
     **/
    public DescendantAxis() {
        this(false);
    }

    /**
     * If <code>includeSelf</code> is <code>true</code> create
     * a descendant-or-self axis.
     * If <code>includeSelf</code> is <code>false</code> create
     * a descendant axis.
     **/
    public DescendantAxis(boolean includeSelf) {
        this.includeSelf = includeSelf;
    }

    /**
     * The descendant nodes are visited in document order.
     **/
    public AxisIterator iterator(Object contextNode, Navigator navigator)
                                                  throws XPathException {
        return new DescendantAxisIterator(contextNode, navigator, includeSelf);
    }

    /**
     * The principal node type of this axis is
     * {@link Navigator#ELEMENT}.
     **/
    public int getPrincipalNodeType() {
        return Navigator.ELEMENT;
    }

    public boolean isReverse() {
        return false;
    }

    public String toString() {
        if (includeSelf) return "descendant-or-self";
        else return "descendant";
    }

    /*
     * Works by recursively creating DescendantAxisIterators.
     */
    class DescendantAxisIterator implements AxisIterator {
        private Object contextNode;
        private Navigator navigator;
        private int count; // num of children of contextNode
        private int current; // current child of contextNode
                             // -1 indicates contextNode itself
        private AxisIterator currentIterator = null;

        DescendantAxisIterator(Object contextNode, Navigator navigator,
                            boolean includeSelf) throws XPathException {
            this.contextNode = contextNode;
            this.navigator = navigator;
            count = navigator.childCount(contextNode);
            current = includeSelf ? -1 : 0;
        }

        public boolean hasNext() {
            return (current < count) ||
                ((currentIterator != null) && currentIterator.hasNext());
        }

        public Object next() throws XPathException {
            if ((currentIterator == null) || !currentIterator.hasNext()) {
                if (current >= count) throw new XPathException(
                                "No more elements in iterator");

                // Return self if current index -1

                if (current == -1) {
                    current++;
                    return contextNode;
                }

                // Return next child and create iterator for that child

                Object child = navigator.getChild(contextNode, current++);
                currentIterator = new DescendantAxisIterator(
                                      child, navigator, true);
            }
            return currentIterator.next();
        }





/*
        DescendantAxisIterator(Object contextNode, Navigator navigator)
                                                 throws XPathException {
            this.contextNode = contextNode;
            this.navigator = navigator;
            count = navigator.childCount(contextNode);
        }

        public boolean hasNext() {
            return (current < count) ||
                ((currentIterator != null) && currentIterator.hasNext());
        }

        public Object next() throws XPathException {
            if ((currentIterator == null) || !currentIterator.hasNext()) {
                if (current >= count) throw new XPathException(
                                "No more elements in iterator");

                // Return next child and create iterator for that child

                Object child = navigator.getChild(contextNode, current++);
                currentIterator = new DescendantAxisIterator(child, navigator);
                return child;
            } else {
                return currentIterator.next();
            }
        }
*/
    }

}
