package org.planx.xpath;

import org.planx.xpath.expr.Expression;
import org.planx.xpath.expr.XPathParser;
import org.planx.xpath.object.*;

/**
 * Represents an XPath expression. See package description for
 * information about what is supported and what is not, as well
 * as a brief overview of the design.
 * <br><br>
 * A simple usage example:
 * <br><br>
 * <blockquote><pre>
 *     Navigator nav = new XMLStoreNavigator();
 *     DocNode node = new DocNode(<i>...get node from XML Store...</i>);
 *     XPath xpath = new XPath("child::node()", nav);
 *     XObject result = xpath.evaluate(node);
 * </pre></blockquote>
 **/
public class XPath {
    private final Expression expr;
    private final Navigator nav;

    /**
     * Construct a new XPath from the given string expression and
     * set it to use the specified navigator.
     **/
    public XPath(String expr, Navigator nav) throws XPathException {
        try {
            XPathParser parser = new XPathParser(expr);
            this.expr = (Expression) parser.parse().value;
            this.nav = nav;
        } catch (Exception e) {
            throw new XPathException("Syntax error: "+e.toString());
        }
    }

    /**
     * Evaluate this XPath expression using the specified context
     * and environment.
     **/
    public XObject evaluate(Context ctxt, Environment env)
                                    throws XPathException {
        return expr.evaluate(ctxt, env, nav);
    }

    /**
     * Evaluate this XPath expression using the specified context node
     * and environment.
     **/
    public XObject evaluate(Object ctxtNode, Environment env)
                                    throws XPathException {
        Context ctxt = new Context(ctxtNode, 1, 1);
        return evaluate(ctxt, env);
    }

    /**
     * Evaluate this XPath expression using the specified context node.
     * The environment will be initialized to default
     * values (i.e. using core library functions only and
     * no variables bound).
     **/
    public XObject evaluate(Object ctxtNode) throws XPathException {
        Environment env = new Environment();
        return evaluate(ctxtNode, env);
    }

    /**
     * Returns the expression as a string. The string returned is not the
     * original string used to construct the expression, but rather
     * a representation of the expression after parsing.
     * Useful for debugging purposes.
     **/
    public String toString() {
        return expr.toString();
    }
}
