package org.planx.xmlstore.regions;

import java.io.*;
import java.lang.ref.PhantomReference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.*;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.*;
import org.planx.xmlstore.nodes.*;
import org.planx.xmlstore.references.*;
import org.planx.util.*;

public class RegionManager {
    private static final Object PRESENT = new Object();

    // I/O fields
    private XMLStore xmlstore;
    private FileSystem fs;
    private NodeConverter dnav;

    // Region fields
    private RegionConfiguration conf;
    private Map<FileSystemIdentifier,Region> regions;
    private Region head;
    private Map<Region,Object> cache;
    private int idSeq = 0;      // assigns unique local ids to regions

    // Root set fields
    private Collection<ReferenceListener> rootListeners;
    private Map<LocalLocator,Integer> roots; // permanent roots with reference counts
    private WeakHashMap<LocalLocator,WeakLocator> liveClones;
    private WeakHashSet<LocalLocator> liveRoots; // transient roots
    private ReferenceQueue<LocalLocator> queue;

    // Sharer fields
    private Sharer sharer;
    private long originalSize = 0;           // for statistics

    // CONSTRUCTORS

    public RegionManager(XMLStore xmlstore) throws IOException {
        this(xmlstore, new RegionConfiguration());
    }

    public RegionManager(XMLStore xmlstore, RegionConfiguration conf)
                                                  throws IOException {
        this.xmlstore = xmlstore;
        this.conf = conf;

        // Init file system
        fs = new LocalFileSystem(xmlstore.toString());
        dnav = new NodeConverter(xmlstore);

        // Init cache
        regions = new HashMap<FileSystemIdentifier,Region>();
        cache = new LinkedHashMap<Region,Object>() {
            protected boolean removeEldestEntry(
                        Map.Entry<Region,Object> eldest) {
                if (size() > RegionManager.this.conf.CACHE_SIZE) {
                    Region r = eldest.getKey();
                    try {
                        r.flush();
                    } catch (IOException e) {
                        e.printStackTrace();
                    } catch (UnknownReferenceException e) {
                        e.printStackTrace();
                    }
                    return true;
                }
                return false;
            }
        };

        // Init root sets
        roots = new HashMap<LocalLocator,Integer>();
        liveClones = new WeakHashMap<LocalLocator,WeakLocator>();
        liveRoots = new WeakHashSet<LocalLocator>();
        queue = new ReferenceQueue<LocalLocator>();
        rootListeners = new ArrayList<ReferenceListener>();
        readRootSet();   // load permanent roots from disk

        // Init sharer
        sharer = (conf.USE_HASH_SHARER) ?
                new HashSharer(this):new MSDSharer(this);
        if (conf.ENABLE_SHARER) sharer.start();
    }

    // PUBLIC READ/WRITE/FLUSH

    public synchronized SystemNode load(LocalLocator loc)
                                      throws IOException,
                               UnknownReferenceException {
        checkClosed();
        expungeStaleRoots();
        Region r = lookup(loc);
        if (r == null) throw new UnknownReferenceException(loc);
        synchronized (r) {
            // TODO: Second argument has been deprecated, please remove
            SystemNode node = r.load(loc, loc);
            return node;
        }
    }

    public synchronized LocalLocator save(SystemNode node)
                                       throws IOException {
        checkClosed();
        expungeStaleRoots();
        try {
            // Check if already saved
            if (node.getLocator() != null) {
                return node.getLocator();
            }
            // Save in write region
            Region r = head();
            synchronized (r) {
                LocalLocator loc = r.save(node);
                addWeakRoot(loc, r);  // hang on to this as long
                                      // as referenced in-memory
                return loc;
            }
        } catch (UnknownReferenceException e) {
            throw new IOException(e.toString());
        }
    }

    /**
     * Persists the current writing <code>Region</code> and begins a new.
     */
    public synchronized void flush() throws IOException {
        checkClosed();
        try {
            if (head != null) {
                synchronized (head) {
                    head.close();
                    originalSize += head.originalSize();
                    if (conf.DO_SHARE_NEW) {
                        // Removes internal redundancy and persists data
                        sharer.addRegion(head);
                    } else {
                        head.flush();
                    }
                    head = null;
                }
            }
        } catch (UnknownReferenceException e) {
            throw new IOException(e.toString());
        }
    }

    /**
     * Returns the current <code>Region</code> to write new data to.
     */
    private synchronized Region head() throws IOException,
                                UnknownReferenceException {
        // The head Region could have been closed since last time
        // cause it became full
        if (head != null) {
            synchronized (head) {
                if (head.isClosed()) flush();
            }
        }
        if (head == null) {
            head = new Region(this, ++idSeq);
            regions.put(head.getIdentifier(), head);
        }
        return head;
    }

    /**
     * Attempts to release all cached ressources held. This means
     * flushing the write region, clearing the cache, and flushing
     * all <code>Regions</code>.
     */
    public synchronized void release() throws IOException {
        try {
            checkClosed();
            flush();
            cache.clear();
            for (Region r : regions.values()) {
                r.flush();
            }
            expungeStaleRoots();
        } catch (UnknownReferenceException e) {
            throw new IOException(e.toString());
        }
    }


    // AGGRESSIVE DISCRIMINATION

    /**
     * Forces brutal compaction of the store by running the
     * <code>Sharer</code> continously until the store is fully
     * discriminated. This will, of course,
     * not do well with concurrently executing applications.
     */
    public synchronized void discriminate() throws IOException {
        compact(true);
    }

    public synchronized void compact(boolean doComplete) throws IOException {
        try {
            while ((!doComplete && !sharer.isIterated()) ||
                   (doComplete && !sharer.isDiscriminated())) {
                sharer.share();
            }
        } catch (UnknownReferenceException e) {
            IOException ie = new IOException(e.toString());
            ie.setStackTrace(e.getStackTrace());
            throw ie;
        }
    }

    // RESOURCES

    public RegionConfiguration configuration() {
        checkClosed();
        return conf;
    }

    FileSystem getFileSystem() {
        checkClosed();
        return fs;
    }

    NodeConverter getConverter() {
        checkClosed();
        return dnav;
    }

    public Sharer getSharer() {
        checkClosed();
        return sharer;
    }

    XMLStore getXMLStore() {
        return xmlstore;
    }

    // PACKAGE PRIVATE UTILITIES

    /**
     * Called by a system application to request a <code>Region</code>
     * be put in the cache.
     */
    synchronized void cache(Region r) throws IOException,
                                UnknownReferenceException {
        checkClosed();
        r.cache();
    }

    /**
     * Called by a <code>Region</code> to inform the manager that it
     * has been cached.
     */
    synchronized void informCached(Region r) {
        checkClosed();
        cache.put(r, PRESENT);
    }

    /**
     * Returns the <code>Region</code> containing the specified
     * <code>LocalLocator/code>.
     */
    synchronized Region lookup(LocalLocator loc) {
        checkClosed();
        if (loc == null) return null;
        return regions.get(loc.getFileSystemId());
    }

    /**
     * Changes the ID of the specified <code>Region</code> in the
     * lookup table.
     * Warning: Currently it is done by removing the old entry and
     * re-inserting the region. This means that the method cannot be
     * called from within an iterator of the regions (results in concurrent
     * modification).
     */
    synchronized void change(FileSystemIdentifier oldId, Region region) {
        checkClosed();
        regions.remove(oldId);
        regions.put(region.getIdentifier(), region);
    }

    synchronized void remove(Region region) {
        checkClosed();
        synchronized (region) {
            assert region.incomingSize() == 0;
            assert region.outgoingSize() == 0;
            regions.remove(region.getIdentifier());
            cache.remove(region);
            if (head == region) head = null;
            sharer.removeRegion(region);
        }
    }

    synchronized Collection<Region> getRegions() {
        checkClosed();
        return regions.values();
    }

    // ROOT SET SERVICE METHODS

    /**
     * Adds a root permanently that must be explicitly removed using
     * {@link #release(LocalLocator)}.
     */
    public synchronized void retain(LocalLocator loc)
                    throws UnknownReferenceException {
        checkClosed();
        expungeStaleRoots();
        Integer count = roots.get(loc);
        if (count == null) {
            // new root
            addToIncoming(loc, lookup(loc));
            count = new Integer(1);
        } else {
            count = count + 1;
        }
        roots.put(loc, count);
    }

    /**
     * Removes a root previously added with {@link #retain(LocalLocator)}.
     */
    public synchronized void release(LocalLocator loc)
                     throws UnknownReferenceException {
        checkClosed();
        expungeStaleRoots();
        Integer count = roots.get(loc);
        if (count != null) {
            count--;
            if (count <= 0) {
                // dead root
                roots.remove(loc);
                removeFromIncoming(loc, lookup(loc));
            } else {
                roots.put(loc, count);
            }
        } else {
            throw new UnknownReferenceException(
                    "LocalLocator not in root set "+loc);
        }
    }

    /**
     * Clears all cached soft references to data to conserve memory.
     */
    public synchronized void expunge() {
        for (Region r : regions.values()) {
            r.clearRefMap();
        }
        expungeStaleRoots();
    }

    /**
     * Adds a transient root that will disappear when no outside (in-memory)
     * references to it exist anymore.
     */
    private void addWeakRoot(LocalLocator loc, Region r)
                        throws UnknownReferenceException {
        expungeStaleRoots();
        LocalLocator memLoc = liveRoots.get(loc);
        // Ensure that we keep track of the locator pointed to by others
        // so that it will be garbage collected last
        if (memLoc != null) {
            // Matches existing in-memory locator
            loc.locatorMoved(memLoc);
        } else {
            // New in-memory locator
            WeakLocator wc = new WeakLocator(loc, queue);
            liveClones.put(loc, wc);
            liveRoots.add(loc);
            addToIncoming(wc.clone, r);
        }
    }

    private class WeakLocator extends PhantomReference<LocalLocator> {
        final LocalLocator clone;

        WeakLocator(LocalLocator loc, ReferenceQueue<LocalLocator> queue) {
            super(loc, queue);
            this.clone = loc.clone();
        }

        public String toString() {
            return "WeakLocator{"+clone+"}";
        }
    }

    /**
     * For any in-memory locators that went out of scope, remove from
     * incoming sets only if no other equal permanent or weak root exist.
     */
    private void expungeStaleRoots() {
        WeakLocator wc;
        while ((wc = (WeakLocator) (PhantomReference<LocalLocator>) queue.poll()) != null) {
            LocalLocator deadRoot = wc.clone;
            wc.clear();
            if (deadRoot != null && !liveRoots.contains(deadRoot)
                                 && !roots.containsKey(deadRoot)) {
                Region r = regions.get(deadRoot.getFileSystemId());
                if (r != null) {
                    synchronized (r) {
                        InterRegionEdge edge =
                            new InterRegionEdge(deadRoot, null, 0, r, null);
                        r.removeIncoming(edge);
                    }
                }
            }
        }
    }

    /**
     * Permanent roots are merged by summing their reference counts.
     * Weak roots are merged by creating a clone of the new locator and
     * letting all in-memory references point to this, and inserting the
     * pointed-to locator (clone) in weakRoots so that the entry here-in
     * will live the longest (i.e. other live locators point to this).
     *
     * @return <code>false</code> if <code>oldLoc</code> was not known,
     *         that is, the root has been removed since last
     */
    synchronized boolean rootMoved(LocalLocator oldLoc, LocalLocator newLoc) {
        checkClosed();
        expungeStaleRoots();

        // Migrate reference count, summing counts for merged nodes
        Integer oldCount = roots.remove(oldLoc);
        if (oldCount != null) {
            Integer newCount = roots.get(newLoc);
            if (newCount == null) {
                roots.put(newLoc, oldCount);
            } else {
                roots.put(newLoc, newCount + oldCount);
            }
            // Inform listeners
            callRootListeners(oldLoc, newLoc);
        }

        // Union-find to merge in-memory locators
        LocalLocator memLoc = liveRoots.get(oldLoc);
        if (memLoc != null) {
            // No need to expunge or keep track of deprecated locators
            WeakLocator liveClone = liveClones.remove(oldLoc);
            if (liveClone != null) liveClone.clear();
            liveRoots.remove(oldLoc);

            // Ensure that we keep track of the locator pointed to by others
            // so that it will be garbage collected last
            WeakLocator wc = new WeakLocator(newLoc, queue);
            liveClones.put(newLoc, wc);
            liveRoots.add(newLoc);

            // Get in-memory locators to point to the new locator
            memLoc.locatorMoved(newLoc);
        }

        // Is this a root we know at all? If not, remove it from incoming set
        if (oldCount == null && memLoc == null) {
            try {
                removeFromIncoming(newLoc, lookup(newLoc));
                return false;
            } catch (UnknownReferenceException e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    private void addToIncoming(LocalLocator loc, Region r)
                            throws UnknownReferenceException {
        if (r == null) throw new UnknownReferenceException(loc);
        synchronized (r) {
            InterRegionEdge edge = new InterRegionEdge(loc, null, 0, r, null);
            r.addIncoming(edge);
        }
    }

    private void removeFromIncoming(LocalLocator loc, Region r)
                                throws UnknownReferenceException {
        if (!liveRoots.contains(loc) && !roots.containsKey(loc)) {
            if (r == null) return;
            synchronized (r) {
                InterRegionEdge edge =
                    new InterRegionEdge(loc, null, 0, r, null);
                r.removeIncoming(edge);
            }
        }
    }

    /**
     * Registers the specified <code>ReferenceListener</code> for receiving
     * <code>LocalLocator</code> events for roots.
     */
    public synchronized void addRootListener(ReferenceListener l) {
        checkClosed();
        rootListeners.add(l);
    }

    /**
     * Removes a <code>ReferenceListener</code> previously registered with
     * {@link #addRootListener(ReferenceListener)}.
     */
    public synchronized void removeRootListener(ReferenceListener l) {
        checkClosed();
        rootListeners.remove(l);
    }

    private void callRootListeners(LocalLocator oldLoc, LocalLocator newLoc) {
        // Handle listeners listening for root events
        for (ReferenceListener l : rootListeners) {
            l.referenceMoved(oldLoc, newLoc);
        }
    }

    // STATISTICS

    /**
     * Returns the on-disk file size of the store. This includes any free space.
     * Refer to {@link #statistics()} for more detailed size information.
     */
    public synchronized long size() throws IOException {
        checkClosed();
        return fs.size();
    }

    public synchronized int numOfRegions() {
        checkClosed();
        return regions.size();
    }

    public synchronized boolean isDiscriminated() {
        checkClosed();
        return sharer.isDiscriminated();
    }

    public synchronized String toString() {
        if (xmlstore == null) return "RegionManager is closed";
        return "RM["+regions.size()+"]";
    }

    public synchronized Statistics statistics() {
        return statistics(false);
    }

    public synchronized Statistics statistics(boolean isVerbose) {
        checkClosed();
        expungeStaleRoots();
        return new Statistics(this, isVerbose);
    }

    public static class Statistics {
        public boolean isVerbose;
        public long size = 0, fileSize, originalSize;
        public int currentNodes = 0, originalNodes = 0;
        public int permanentRoots, transientRoots, transientClones;
        public int numOfRegions, cachedRegions;
        public int incomingSize = 0, outgoingSize = 0;
        public int discriminations = 0;
        public boolean isDiscriminated, isIterated;
        public String roots, liveRoots, liveClones;
        public Region.Statistics[] regions = null;
        public long freeMemory, totalMemory, maxMemory;

        Statistics(RegionManager m, boolean isVerbose) {
            try {
                this.isVerbose = isVerbose;
                this.isDiscriminated = m.isDiscriminated();
                this.fileSize = m.fs.size();
                this.originalSize = m.originalSize;
                this.permanentRoots = m.roots.size();
                this.transientRoots = m.liveRoots.size();
                this.transientClones = m.liveClones.size();
                this.numOfRegions = m.regions.size();
                this.cachedRegions = m.cache.size();
                this.roots = m.roots.toString();
                this.liveRoots = m.liveRoots.toString();
                this.liveClones = m.liveClones.toString();
                this.discriminations = m.sharer.invocations();
                this.isIterated = m.sharer.isIterated();

                if (isVerbose) regions = new Region.Statistics[m.regions.size()];
                int i = 0;
                for (Region r : m.regions.values()) {
                    if (isVerbose) regions[i++] = r.statistics();
                    currentNodes += r.currentNodes;
                    originalNodes += r.originalNodes;
                    incomingSize += r.incomingSize();
                    outgoingSize += r.outgoingSize();
                    size += r.size();
                }

                Runtime rt = Runtime.getRuntime();
                freeMemory = rt.freeMemory();
                totalMemory = rt.totalMemory();
                maxMemory = rt.maxMemory();
            } catch (IOException e) {}
        }

        public String toString() {
            return brief();
        }

        public String brief() {
            return "RM["+numOfRegions+"]{size="+size+",nodes="+currentNodes+
                   ",inset="+incomingSize+",outset="+outgoingSize+",freeMem="+freeMemory+"}";
        }

        public String full() {
            String s = "RegionManager\n"+
                       "  Size:             "+size+"\n"+
                       "  Original size:    "+originalSize+"\n"+
                       "  File size:        "+fileSize+"\n"+
                       "  Nodes:            "+currentNodes+"\n"+
                       "  Original nodes:   "+originalNodes+"\n"+
                       "  Is discriminated: "+isDiscriminated+"\n"+
                       "  Is iterated:      "+isIterated+"\n"+
                       "  Discriminations:  "+discriminations+"\n"+
                       "  Regions:          "+numOfRegions+"\n"+
                       "  Regions in cache: "+cachedRegions+"\n"+
                       "  Permanent roots:  "+permanentRoots+"\n"+
                       "  Transient roots:  "+transientRoots+"\n"+
                       "  Incoming size:    "+incomingSize+"\n"+
                       "  Outgoing size:    "+outgoingSize+"\n"+
                       "  Free memory:      "+freeMemory+"\n"+
                       "  Total memory:     "+totalMemory+"\n"+
                       "  Max memory:       "+maxMemory+"\n";
            if (isVerbose) {
                StringBuilder sb = new StringBuilder();
                sb.append(s);
                for (int i=0; i<regions.length; i++) {
                    sb.append(regions[i].full());
                }
                return sb.toString();
            }
            return s;
        }
    }

    // CLOSING

    /**
     * Closes the <code>RegionManager</code> and releases all ressources.
     * Any subsequent calls to methods will throw an <code>IllegalStateException</code>.
     */
    public synchronized void close() throws IOException {
        checkClosed();
        expungeStaleRoots();
        sharer.stop();
        try {
            // Impl note: Copies regions list to avoid concurrent modification.
            for (Region r : new ArrayList<Region>(regions.values())) {
                synchronized (r) {
                    r.flush();
                    // TODO: Write all regions' incoming and outgoing sets to disk
                }
            }
        } catch (UnknownReferenceException e) {
            throw new IOException(e.toString());
        }
        sharer = null;
        regions = null;
        xmlstore = null;
        fs = null;
        dnav = null;
        head = null;
        cache = null;
        rootListeners = null;
        roots = null;
        liveClones = null;
        queue = null;
    }

    private void checkClosed() {
        if (xmlstore == null) throw new IllegalStateException(
                "RegionManager closed");
    }

    // READ/WRITE FIELDS TO PERSISTENT STORAGE

    private synchronized void writeRootSet() throws IOException {
        expungeStaleRoots();
        Streamer<LocalLocator> locStreamer = LocalLocator.getStreamer(false);
        DataOutputStream out = new DataOutputStream(
            new FileOutputStream(xmlstore.toString()+".roots"));
        out.writeInt(roots.size());
        for (Map.Entry<LocalLocator,Integer> entry : roots.entrySet()) {
            locStreamer.toStream(out, entry.getKey());
            Streamers.writeShortInt(out, entry.getValue());
        }
        out.flush();
        out.close();
    }

    private synchronized void readRootSet() throws IOException {
        expungeStaleRoots();
        try {
            Streamer<LocalLocator> locStreamer = LocalLocator.getStreamer(false);
            DataInputStream in = new DataInputStream(
                new FileInputStream(xmlstore.toString()+".roots"));
            int size = in.readInt();
            for (int i=0; i<size; i++) {
                LocalLocator loc = locStreamer.fromStream(in);
                int count = Streamers.readShortInt(in);
                roots.put(loc, count);
            }
            in.close();
        } catch (FileNotFoundException e) {}
    }
}
