package org.planx.xmlstore.stores;

import java.io.*;
import java.net.*;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.*;
import org.planx.xmlstore.nodes.*;
import org.planx.xmlstore.references.*;
import static org.planx.xmlstore.stores.NetworkProtocol.*;

/**
 * @author Henning Niss
 * @author Thomas Ambus
 */
class NetworkRequestHandler implements Runnable {
    private static Streamer<Node> inlineNodeStreamer = new InlineNodeStreamer();
    private Streamer<Reference> vrefStreamer;
    private Socket socket;
    private XMLStore xmlstore;
    private DataInputStream in;
    private DataOutputStream out;
    private boolean isRunning = true;

    /**
     * Create a handler for requests coming in on the specified
     * socket, and dispatch all synthesized XML Store commands
     * to the specified XML Store.
     *
     * @param xmlstore   the underlying XML Store
     * @param socket     the socket on which the request arrived
     **/
    public NetworkRequestHandler(XMLStore xmlstore, Socket socket) throws IOException {
        this.xmlstore = xmlstore;
        this.socket = socket;
        in = new DataInputStream(new BufferedInputStream(socket.getInputStream()));
        out = new DataOutputStream(new BufferedOutputStream(socket.getOutputStream()));
        socket.setSoTimeout(5000); // milliseconds
        vrefStreamer = Streamers.getPolymorphicStreamer(Reference.class);
    }

    public void run() {
        try {
            // Protocol hallo handshake

            byte b = in.readByte();
            if (b != REQUEST_VERSION)
                throw new NetworkProtocolException("Expected REQUEST_VERSION");
            out.writeByte(PROTOCOL_VERSION);
            out.flush();

            // Handle client commands

            while (isRunning) {
                byte command = in.readByte();
                switch (command) {
                case REQUEST_LOAD:
                    handleLoad();
                    break;
                case REQUEST_SAVE:
                    handleSave();
                    break;
                default:
                    throw new NetworkProtocolException
                        ("Unknown client command "+command);
                }
            }
        } catch (InterruptedIOException e) {
            /* client didn't send anything within the timeout;
               stop this "session" (that is return) */
        } catch (EOFException e) {
            /* client closed down the socket;
               stop this "session" (that is return) */
        } catch (SocketException e) {
            // Do nothing - socket closed by client or close method
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {in.close();} catch (IOException e) {}
            try {out.close();} catch (IOException e) {}
            try {socket.close();} catch (IOException e) {}
        }
    }

    private void handleLoad() throws IOException {
        Reference vref = vrefStreamer.fromStream(in);
        try {
            Node node = xmlstore.load(vref);
            out.writeByte(RESPONSE_OK);
            inlineNodeStreamer.toStream(out, node);
        } catch (IOException e) {
            out.writeByte(RESPONSE_IO_EXCEPTION);
            e.printStackTrace();
        } catch (UnknownReferenceException e) {
            out.writeByte(RESPONSE_UNKNOWN_VREF);
            e.printStackTrace();
        }
        out.flush();
    }

    private void handleSave() throws IOException {
        Node node = inlineNodeStreamer.fromStream(in);
        try {
            Reference vref = xmlstore.save(node);
            out.writeByte(RESPONSE_OK);
            vrefStreamer.toStream(out, vref);
        } catch (IOException e) {
            out.writeByte(RESPONSE_IO_EXCEPTION);
            e.printStackTrace();
        }
        out.flush();
    }
}
