package org.planx.xmlstore.nameserver;

import java.io.IOException;
import java.io.Serializable;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import org.planx.xmlstore.NameAlreadyBoundException;
import org.planx.xmlstore.NameServer;
import org.planx.xmlstore.StaleReferenceException;
import org.planx.xmlstore.Reference;
import org.planx.xmlstore.routing.DistributedMap;
import org.planx.xmlstore.routing.Identifier;

/**
 * A distributed <code>NameServer</code> that relies on a <code>DistributedMap</code>.
 * A typical <code>DistributedMap</code> implementation such as <code>Kademlia</code>
 * will not use any global consensus algorithms or locks to ensure that all peers
 * agree on the contents of the global name server at any given time. In a stabile
 * network with reliable communication that uses Kademlia, updates will be globally
 * available with a delay of the time it takes to send an UDP packet. With unreliable
 * communication, no guarantees are given other than that the peers will eventually agree
 * on the newest version of a name server entry.
 * <p>
 * All-in-all, the semantics of the operations are somewhat more loose than required,
 * so this implementation should be regarded as experimental.
 *
 * @author Thomas Ambus
 */
public class GlobalNameServer implements NameServer {
    private DistributedMap map;

    public GlobalNameServer(DistributedMap map) {
        this.map = map;
    }

    public Reference lookup(String name) throws IOException {
        return (Reference) map.get(getIdentifier(name));
    }

    public void bind(String name, Reference vref) throws IOException,
                                                NameAlreadyBoundException {
        Reference tmp = lookup(name);
        if (tmp != null && tmp.equals(vref)) return;
        if (tmp != null) throw new NameAlreadyBoundException(name,vref);
        map.put(getIdentifier(name), (Serializable) vref);
    }

    public void rebind(String name, Reference oldr, Reference newr)
                                 throws IOException, StaleReferenceException {
        Reference tmp = lookup(name);

        if (tmp==null || tmp.equals(oldr)) {
            // mimick bind, but without the checks
            map.put(getIdentifier(name), (Serializable) newr);
        } else {
            throw new StaleReferenceException(name,oldr,tmp);
        }
    }

    private static Identifier getIdentifier(String s) {
        MessageDigest md;
        try {
            md = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
        md.update(s.getBytes());
        return new Identifier(md.digest());
    }
}
