package org.planx.msd.number;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.planx.msd.*;
import org.planx.msd.util.*;

/**
 * This class provides a skeletal implementation of a <code>Discriminator</code>
 * of numbers. The numbers are chopped into chunks of size 16 bits.
 *
 * @author Thomas Ambus
 */
public abstract class NumberDiscriminator<T> extends AbstractDiscriminator<T> {
    protected static final int CHUNK_BITSIZE = 16;
    protected static final int CHUNK_MASK    = 0xFFFF;
    protected Memory memory;

    protected NumberDiscriminator(Memory memory) {
        this.memory = memory;
    }

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                              Extractor<U,? extends T,S> e) {
        ChunkExtractor<U> chunkExtractor = chunkExtractor(e);

        Collection<List<? extends U>> unfin, work;
        Collection<List<U>> partition;
        Collection<List<S>> result = new ArrayList<List<S>>();
        unfin = (Collection) Collections.singletonList(values);

        // Step through chunks
        for (int offset=0,size=bitsize(); offset<size && !unfin.isEmpty();
                                offset+=CHUNK_BITSIZE) {
            work = unfin;
            unfin = new ArrayList<List<? extends U>>();

            for (List<? extends U> block : work) {
                chunkExtractor.setOffset(offset);
                partition = memory.discriminate(block, chunkExtractor);

                for (List<U> subBlock : partition) {
                    if (subBlock.size() > 1) unfin.add(subBlock);
                    else result.add(Discriminators.valueList(subBlock, e));
                }
            }
        }

        for (List<? extends U> block : unfin) {
            result.add(Discriminators.valueList(block, e));
        }
        return result;
    }

    /**
     * Returns the total bitsize of the specific type of number.
     */
    protected abstract int bitsize();

    /**
     * returns a <code>ChunkExtractor</code> for the specific type of number.
     */
    protected abstract <U> ChunkExtractor<U> chunkExtractor(Extractor<U,? extends T,?> e);

    /**
     * A <code>ChunkExtractor</code> is capable of extracting an integer chunk of
     * bitsize <code>CHUNK_BITSIZE</code> beginning from the offset set by
     * <code>setOffset</code> from the type <code>U</code>.
     */
    protected abstract class ChunkExtractor<U> implements IntExtractor<U,U> {
        protected int offset = -1;
        protected Extractor<U,? extends T,?> e;

        protected ChunkExtractor(Extractor<U,? extends T,?> e) {
            this.e = e;
        }

        public void setOffset(int offset) {
            this.offset = offset;
        }

        public abstract int getLabel(U elm);

        public U getValue(U elm) {
            return elm;
        }
    }
}
