package org.planx.msd;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.planx.msd.number.IntExtractor;

/**
 * Encapsulates the memory needed for an atomic discriminator. A
 * <code>Memory</code> can only be used by one thread at a time, and it
 * should be reused as much as possible, as it allocates a significant
 * amount of memory. A good strategy, is to create one <code>Memory</code>
 * per thread, and let all the discriminators of a particular thread use
 * the <code>Memory</code> designated for that thread. This is done by
 * passing the <code>Memory</code> object to the constructor of the
 * discriminator. If only one thread is needed, use the
 * <code>getMemory</code> method of {@link DiscriminatorFactory}.
 *
 * @author Thomas Ambus
 */
public final class Memory {
    /**
     * The size of the <code>dictionary</code> and <code>used</code> arrays.
     */
    public static final int CAPACITY = 1<<16;

    /**
     * A dictionary array of size <code>CAPACITY</code> for use during discrimination.
     * After each discrimination all entries in this array must be <code>null</code>.
     */
    public final List[] dictionary;

    /**
     * An array of size <code>CAPACITY</code> to keep track of which indexes in
     * <code>dictionary</code> is used.
     */
    public final int[] used;

    /**
     * Creates a new <code>Memory</code> that allocates a large amount of memory.
     */
    public Memory() {
        dictionary = new List[CAPACITY];
        used = new int[CAPACITY];
    }

    /**
     * Convenience method to easily use the <code>Memory</code>'s dictionary.
     */
    public <U,S> Collection<List<S>> discriminate(List<? extends U> values, IntExtractor<U,S> e) {
        List[] dictionary = this.dictionary;
        int[] used = this.used;
        int used_size = 0;

        for (U elm : values) {
            int index = e.getLabel(elm);

            List list = dictionary[index];
            if (list == null) {
                used[used_size++] = index;    // remember this index is used
                list = new ArrayList<S>();
                dictionary[index] = list;
            }
            list.add(e.getValue(elm));
        }

        Collection<List<S>> result = new ArrayList<List<S>>(used_size);
        for (int i=0; i<used_size; i++) {
            int index = used[i];
            result.add(dictionary[index]);
            dictionary[index] = null;         // allow gc
        }
        return result;
    }
}
