/*
 * #%L
 * ToPIA :: Service Replication
 * 
 * $Id: ReplicationModel.java 1894 2010-04-15 15:44:51Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.4/topia-service-replication/src/main/java/org/nuiton/topia/replication/model/ReplicationModel.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.topia.replication.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityEnum;
import org.nuiton.topia.persistence.util.EntityOperator;
import org.nuiton.topia.persistence.util.TopiaEntityHelper;
import org.nuiton.topia.replication.TopiaReplicationOperation;
import org.nuiton.topia.replication.operation.AttachLink;
import org.nuiton.topia.replication.operation.DettachAssociation;
import org.nuiton.topia.replication.operation.Duplicate;
import org.nuiton.topia.replication.operation.LoadLink;

public class ReplicationModel {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(ReplicationModel.class);
    /**
     * l'ensemble des contrats d'entites a repliquer
     */
    final protected TopiaEntityEnum[] contracts;
    /**
     * les ids des entites a repliquer (non utilise en mode replicateAll)
     */
    final protected String[] topiaIds;
    /**
     * le dictionnaire des noeuds a repliquer associes a leur type
     */
    protected Map<Class<? extends TopiaEntity>, ReplicationNode> nodes;
    /**
     * la liste des noeuds a repliquer (dans l'ordre de replication)
     */
    protected List<ReplicationNode> order;
    /**
     * un drapeau pour savoir si on effectue une replication de toutes les donnees
     * des contrats.
     */
    final protected boolean replicateAll;

    public ReplicationModel(TopiaEntityEnum[] contracts, Set<Class<? extends TopiaEntity>> types, String... topiaIds) {
        this.contracts = contracts;
        this.topiaIds = topiaIds;
        replicateAll = false;
        nodes = new LinkedHashMap<Class<? extends TopiaEntity>, ReplicationNode>();
        order = new ArrayList<ReplicationNode>();
        Map<Class<? extends TopiaEntity>, ReplicationNode> tmpNodes = new HashMap<Class<? extends TopiaEntity>, ReplicationNode>();
        for (Class<? extends TopiaEntity> k : types) {
            TopiaEntityEnum e = TopiaEntityHelper.getEntityEnum(k, contracts);
            ReplicationNode replicationNode = new ReplicationNode(e);
            tmpNodes.put(k, replicationNode);
        }
        nodes = Collections.unmodifiableMap(tmpNodes);
    }

    public ReplicationModel(TopiaEntityEnum[] contracts, boolean replicateAll, String... topiaIds) {
        this.contracts = contracts;
        this.topiaIds = topiaIds;
        this.replicateAll = replicateAll;
        nodes = new LinkedHashMap<Class<? extends TopiaEntity>, ReplicationNode>();
        order = new ArrayList<ReplicationNode>();
        Map<Class<? extends TopiaEntity>, ReplicationNode> tmpNodes = new HashMap<Class<? extends TopiaEntity>, ReplicationNode>();
        for (TopiaEntityEnum e : contracts) {
            ReplicationNode replicationNode = new ReplicationNode(e);
            tmpNodes.put(e.getContract(), replicationNode);
            if (!replicateAll) {
                // mode restreint : l'ordre est induit par l'ordre sur les contrats passes
                order.add(replicationNode);
            }
        }
        nodes = Collections.unmodifiableMap(tmpNodes);

    }

    public Collection<ReplicationNode> getNodes() {
        return nodes.values();
    }

    public Set<Class<? extends TopiaEntity>> getTypes() {
        return nodes.keySet();
    }

    public ReplicationNode getNode(Class<? extends TopiaEntity> clazz) {
        return nodes.get(clazz);
    }

    public ReplicationNode getNode(TopiaEntityEnum contract) {
        return nodes.get(contract.getContract());
    }

    public void addDependency(List<ReplicationNode> nodes) {
        order.addAll(nodes);
    }

    public TopiaEntityEnum[] getContracts() {
        return contracts;
    }

    public String[] getTopiaIds() {
        return topiaIds;
    }

    public List<ReplicationNode> getOrder() {
        return order;
    }

    public boolean isReplicateAll() {
        return replicateAll;
    }

    @SuppressWarnings("unchecked")
    public ReplicationNode getNode(String propertyName, Class<?> propertyType) {
        if (TopiaEntity.class.isAssignableFrom(propertyType)) {
            //TODO on devrait repasser par le contracts pour etre sur d'etre
            //TODO sur d'etre sur une interface d'entite ?
            Class<? extends TopiaEntity> t = (Class<? extends TopiaEntity>) propertyType;
            if (nodes.containsKey(t)) {
                ReplicationNode dep = getNode(t);
                return dep;
            }
        }
        return null;
    }

    public void detectAssociations(TopiaEntityEnum... filter) throws TopiaException {
        for (Class<? extends TopiaEntity> type : nodes.keySet()) {
            ReplicationNode node = getNode(type);
            EntityOperator<? extends TopiaEntity> operator = node.getOperator();
            List<String> associationProperties = operator.getAssociationProperties();
            if (!associationProperties.isEmpty()) {

                for (String p : associationProperties) {
                    ReplicationNode dep = getNode(p, operator.getAssociationPropertyType(p));
                    if (dep != null) {
                        if (log.isDebugEnabled()) {
                            log.debug("from type - " + type.getSimpleName() + " [" + p + ":" + dep + "]");
                        }
                        node.addAssociation(p, dep);
                    }
                }
            }
        }
    }

    public void detectDirectDependencies() throws TopiaException {
        for (Class<? extends TopiaEntity> type : nodes.keySet()) {
            ReplicationNode node = getNode(type);
            EntityOperator<? extends TopiaEntity> operator = node.getOperator();
            List<String> properties = operator.getProperties();
            if (!properties.isEmpty()) {
                for (String p : properties) {
                    ReplicationNode dep = getNode(p, operator.getPropertyType(p));
                    if (dep != null) {
                        if (log.isDebugEnabled()) {
                            log.debug("from type - " + type.getSimpleName() + " [" + p + ":" + dep + "]");
                        }
                        node.addDependency(p, dep);
                    }
                }
            }
        }
    }

    public void detectDependencies() throws TopiaException {
        Set<ReplicationNode> toResolved = new HashSet<ReplicationNode>(nodes.values());
        Set<ReplicationNode> resolved = new HashSet<ReplicationNode>();
        List<Set<ReplicationNode>> levels = new ArrayList<Set<ReplicationNode>>();
        // premiere passe pour detecter les niveaux de replications
        // on ne regarde que les dependences directe et pas les associations
        // si A -> B alors B doit etre dans un niveau inferieure (i.e replique
        // avant).
        //TODO Il faut pouvoir gerer les cycles (pour cela on doit avoir
        // un dictionnaire pour les compositions qui peuvent etre nulle
        // ainsi on doit etre capable de ne pas tenir compte d'une composition
        // si c'est nullable, sinon cela veut dire obligatoirement
        // que B doit etre replique avant A...
        while (!toResolved.isEmpty()) {
            Set<ReplicationNode> level = new HashSet<ReplicationNode>();
            for (ReplicationNode node : toResolved) {
                if (node.hasDependency()) {
                    for (ReplicationNode n : node.getDependencies().values()) {
                        if (!resolved.contains(n)) {
                            level.add(n);
                        }
                    }
                }
            }
            Set<ReplicationNode> safeLevel = new HashSet<ReplicationNode>();
            if (level.isEmpty()) {
                safeLevel.addAll(toResolved);
            } else {
                // des depedences trouvees
                for (ReplicationNode n : level) {
                    //TODO il faut verifier que le type n'est pas une dependence de level
                    safeLevel.add(n);
                }
                if (safeLevel.isEmpty()) {
                    // on a detecte un cycle sur les dependences, on ne peut rien faire pour le moment
                    throw new IllegalStateException("un cycle dans les dependences a été détecté, l\'algorithme necessite plus de donnes... \n niveau courant : " + level + "\n resolus : " + getOrder());
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("level [" + levels.size() + "] resolved : " + safeLevel);
            }
            toResolved.removeAll(safeLevel);
            resolved.addAll(safeLevel);
            levels.add(safeLevel);
            level.clear();
        }
        // seconde passe : on recherche le meilleur ordre possible pour
        // les types de chaque niveau
        // on ne base desormais que sur les associations
        // si A -*> B alors on va essayer de repliquer B avant A
        // si on detecte un cycle alors on ne peut pas imposer un ordre optimal
        // et on devra dettacher B de A pendant la replication puis reattacher B
        // a A apres replication de A et B
        // l'ordre optimal permet de ne pas effectuer
        // calcul des ordres de replications pour les ensembles de chaque niveau
        // pour trouver le bon ordre, on travaille sur les couvertures
        // des noeuds.;
        HashSet<ReplicationNode> done = new HashSet<ReplicationNode>();
        for (Set<ReplicationNode> level : levels) {
            detectDependenciesOrder(level, done);
        }
    }

    public void detectDependenciesOrder(Set<ReplicationNode> safeLevel, Set<ReplicationNode> doned) {
        if (log.isDebugEnabled()) {
            log.debug("will detect " + safeLevel);
        }
        Map<ReplicationNode, Set<ReplicationNode>> dico = new HashMap<ReplicationNode, Set<ReplicationNode>>();
        for (ReplicationNode n : safeLevel) {
            Set<ReplicationNode> shell = new HashSet<ReplicationNode>(n.getShell());
            shell.retainAll(safeLevel);
            if (log.isDebugEnabled()) {
                log.debug("shell to use for " + n + " : " + shell);
            }
            dico.put(n, shell);
        }
        List<Set<ReplicationNode>> levels = new ArrayList<Set<ReplicationNode>>();
        while (!dico.isEmpty()) {
            if (log.isDebugEnabled()) {
                log.debug("level [" + levels.size() + "] on  " + safeLevel);
                for (Entry<ReplicationNode, Set<ReplicationNode>> entry : dico.entrySet()) {
                    log.debug("node " + entry.getKey() + " : " + entry.getValue());
                }
            }


            // detection des noeud libres
            Set<ReplicationNode> free = new HashSet<ReplicationNode>();
            for (Entry<ReplicationNode, Set<ReplicationNode>> e : dico.entrySet()) {
                if (e.getValue().isEmpty()) {
                    free.add(e.getKey());
                }
            }

            if (free.isEmpty()) {
                // un cycle a ete detectee
                // on ne peut plus rien predire pour cet ensemble
                if (log.isWarnEnabled()) {
                    log.warn("level [" + levels.size() + "] cycle detecte : " + dico.keySet());
                }
                throw new IllegalStateException("un cycle n'a pas pu etre resoud entre l'ensemble " + dico.keySet());
            }

            log.info("there is some free node(s) to resolve : " + free);
            for (Entry<ReplicationNode, Set<ReplicationNode>> e : dico.entrySet()) {
                Set<ReplicationNode> list = e.getValue();
                list.removeAll(free);
            }
            for (ReplicationNode n : free) {
                dico.remove(n);
            }
            if (log.isDebugEnabled()) {
                log.debug("level [" + levels.size() + "] resolved : " + free);
            }
            levels.add(free);
            doned.addAll(free);

            if (dico.isEmpty()) {
                // ordre optimal trouve
                break;
            }
        }
        for (Set<ReplicationNode> nodesForLevel : levels) {
            addDependency(new ArrayList<ReplicationNode>(nodesForLevel));
        }
        dico.clear();
        levels.clear();
    }

    public void detectObjectsToDettach() {
        Set<ReplicationNode> universe = new HashSet<ReplicationNode>();
        for (ReplicationNode node : getOrder()) {
            // on detecte si le node a des associations
            // sortants de l'universe deja replique
            // si oui, alors on marque l'association pour un dettachement
            if (node.hasAssociation()) {
                for (Entry<String, ReplicationNode> e : node.getAssociations().entrySet()) {
                    ReplicationNode nodeDst = e.getValue();
                    if (!universe.contains(nodeDst)) {
                        if (log.isDebugEnabled()) {
                            log.debug("association to dettach " + e.getKey() + " for " + node);
                        }
                        // association sortant
                        node.addAssociationToDettach(e.getKey());
                    }
                }
            }
            // on fait de meme pour les dependences directes
            //TODO la resolution des conflits sur dependences n'est pas encore
            //TODO en place
            if (node.hasDependency()) {
                for (Entry<String, ReplicationNode> e : node.getDependencies().entrySet()) {
                    ReplicationNode nodeDst = e.getValue();
                    if (!universe.contains(nodeDst)) {
                        if (log.isDebugEnabled()) {
                            log.debug("dependency to dettach " + e.getKey() + " for " + node);
                        }
                        // association sortant
                        node.addDependencyToDettach(e.getKey());
                    }
                }
            }
            universe.add(node);
        }
    }

    public void detectOperations() {
        Set<ReplicationNode> universe = new HashSet<ReplicationNode>();
        Set<Link> links = new HashSet<Link>();
        Set<Link> linksToLoad = new HashSet<Link>();

        // premiere passe pour recuperer toutes les associations

        for (ReplicationNode node : order) {
            if (node.hasAssociation()) {
                for (Entry<String, ReplicationNode> entry : node.getAssociations().entrySet()) {
                    String name = entry.getKey();
                    // dans tous les cas, on ajoute un link d'association a reattacher
                    ReplicationNode target = node.getAssociations().get(name);
                    Link link = new Link(node, target, name, true);
                    if (nodes.containsValue(target)) {
                        // on a trouve une association que l'on doit gerer
                        links.add(link);
                        if (log.isDebugEnabled()) {
                            log.debug("link to treate : " + link);
                        }
                    } else {
                    }
                }
                List<String> associationProperties = node.getOperator().getAssociationProperties();
                for (String name : associationProperties) {
                    Class<?> associationPropertyType = node.getOperator().getAssociationPropertyType(name);
                    if (!TopiaEntity.class.isAssignableFrom(associationPropertyType) ||
                            !nodes.containsKey(associationPropertyType)) {

                        Link link = new Link(node, null, name, true);

                        linksToLoad.add(link);
                        if (log.isDebugEnabled()) {
                            log.debug("link to load before replication : " + link);
                        }
                    }
                }
            }
        }

        // deuxieme passe pour detecter les operations a realiser
        for (ReplicationNode node : order) {
            log.debug("------------------------------- for node " + node);
            // on detecte si le node a des associations
            // sortants de l'universe deja replique
            // si oui, alors on marque l'association pour un dettachement
            if (node.hasAssociationsToDettach()) {
                Set<String> names = node.getAssociationsToDettach();
                // operations de dettachement d'association
                for (String name : names) {
                    addOperation(node, node, ReplicationOperationPhase.before, DettachAssociation.class, name);
                }
            }
            Set<Link> tmpLinks = new HashSet<Link>();

            // recherche des associations a charger avant replication

            for (Link link : linksToLoad) {
                if (node.equals(link.getSource())) {
                    tmpLinks.add(link);
                }
            }
            if (!tmpLinks.isEmpty()) {
                // on a des associations a charger avant replication
                for (Link link : tmpLinks) {
                    ReplicationOperationDef op = new ReplicationOperationDef(ReplicationOperationPhase.before, LoadLink.class, node, link);
                    node.addOperation(op);
                }
                linksToLoad.removeAll(links);
                tmpLinks.clear();
            }

            // on fait de meme pour les dependences directes
            //TODO la resolution des conflits sur dependences n'est pas encore
            //TODO en place (mais cela ne devrait pas etre necessaire... pour le moment)

//                if (node.hasDependenciesToDettach()) {
//                    Set<String> names = node.getDependenciesToDettach();
//                    // operations de dettachement d'association
//                    for (String name : names) {
//                        addOperation(node, node, ReplicationOperationPhase.before, DettachDependency.class, name);
//                    }
//                }
            // operation de duplication
            addOperation(node, node, ReplicationOperationPhase.duplicate, Duplicate.class);

            universe.add(node);

            // operations de reattachement
            for (Link link : links) {
                if (link.canReattach(universe, node)) {
                    // lien reattachable
                    tmpLinks.add(link);
                }
            }
            if (!tmpLinks.isEmpty()) {
                // on a trouve des liens a reattacher
                for (Link link : tmpLinks) {
                    ReplicationOperationDef op = new ReplicationOperationDef(ReplicationOperationPhase.after, AttachLink.class, node, link);
                    node.addOperation(op);
                }
                // ces liens ne sont plus a traiter
                links.removeAll(tmpLinks);
            }
        }
    }

    public void adjustOperations(Map<Class<? extends TopiaEntity>, List<String>> data) {
        for (TopiaEntityEnum e : getContracts()) {
            Class<? extends TopiaEntity> contract = e.getContract();
            List<String> ids = data.get(contract);
            ReplicationNode node = getNode(contract);
            if (node == null) {
                // le noeud n'est pas connu (ce n'est pas normal!)
                continue;
            }
            if (ids == null || ids.isEmpty()) {
                List<ReplicationOperationDef> operations = node.getOperations();
                log.info("skip operations on node " + node + " (no data associated)");
                for (ReplicationOperationDef op : operations) {
                    log.info("  skip " + op);
                }
                operations.clear();
            }

        }
    }

    public void detectShell() {
        for (ReplicationNode n : nodes.values()) {
            Set<ReplicationNode> shell = new HashSet<ReplicationNode>();
            getShell(n, shell);
            shell.remove(n);
            n.setShell(shell);
        }
    }

    protected void getShell(ReplicationNode node, Set<ReplicationNode> explored) {
        if (!explored.contains(node)) {
            explored.add(node);
        }
        if (node.hasAssociation()) {
            for (ReplicationNode n : node.getAssociations().values()) {
                if (!explored.contains(n)) {
                    getShell(n, explored);
                }
            }
        }
        if (node.hasDependency()) {
            for (ReplicationNode n : node.getDependencies().values()) {
                if (!explored.contains(n)) {
                    getShell(n, explored);
                }
            }
        }
    }

    protected void addOperation(ReplicationNode ownerNode, ReplicationNode node, ReplicationOperationPhase phase, Class<? extends TopiaReplicationOperation> operationClass, Object... params) {
        ReplicationOperationDef op = new ReplicationOperationDef(phase, operationClass, node, params);
        ownerNode.addOperation(op);
    }
}
