package org.nuiton.topia.migration.mappings;

/*
 * #%L
 * ToPIA :: Service Migration
 * $Id: TMSVersionDAO.java 2984 2014-01-18 20:43:10Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-11/topia-service-migration/src/main/java/org/nuiton/topia/migration/mappings/TMSVersionDAO.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.cfg.Configuration;
import org.hibernate.jdbc.Work;
import org.hibernate.tool.hbm2ddl.SchemaExport;
import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.topia.persistence.support.TopiaHibernateSupport;
import org.nuiton.topia.persistence.util.TopiaUtil;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

/**
 * TMSVersion DAO helper.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.3.4
 */
public class TMSVersionDAO {

    /** logger */
    private final static Log log = LogFactory.getLog(TMSVersionDAO.class);

    public static final String LEGACY_TABLE_NAME = "tmsVersion";

    public static final String TABLE_NAME = "tms_version";

    public static TMSVersion get(TopiaHibernateSupport tx) throws TopiaException {

        try {
            Session session = tx.getHibernateSession();
            Criteria criteria = session.createCriteria(TMSVersion.class);
            List<?> list = criteria.list();
            TMSVersion result = list.isEmpty() ? null : (TMSVersion) list.get(0);
            return result;
        } catch (HibernateException e) {
            throw new TopiaException("Could not obtain version", e);
        }
    }

    public static void createTable(Configuration configuration) {
        // creer le schema en base
        // dans la configuration il n'y a que la table version
        SchemaExport schemaExport = new SchemaExport(configuration);
        schemaExport.create(log.isDebugEnabled(), true);
    }

    public static void dropTable(Configuration configuration) {
        // supprimer le schema en base
        // dans la configuration il n'y a que la table version
        SchemaExport schemaExport = new SchemaExport(configuration);
        schemaExport.drop(log.isDebugEnabled(), true);
    }

    public static TMSVersion create(TopiaHibernateSupport tx, String version) throws TopiaException {

        try {
            Session session = tx.getHibernateSession();

            TMSVersion result = TMSVersion.valueOf(version);
            // save entity
            session.save(result);
            return result;
        } catch (HibernateException e) {
            throw new TopiaException("Could not create version " + version, e);
        }
    }

    public static void update(TopiaHibernateSupport tx, TMSVersion version) throws TopiaException {
        try {
            Session session = tx.getHibernateSession();
            session.saveOrUpdate(version);
//            tx.commit(); // XXX AThimel 25/11/2013 I guess, this has nothing to do here
        } catch (HibernateException e) {
            throw new TopiaException("Could not update version " + version, e);
        }
    }

    public static void deleteAll(TopiaHibernateSupport tx) throws TopiaException {
        try {
            Session session = tx.getHibernateSession();
            Criteria criteria = session.createCriteria(TMSVersion.class);
            List<?> list = criteria.list();
            for (Object o : list) {
                session.delete(o);
            }
        } catch (HibernateException e) {
            throw new TopiaException("Could not delete all versions", e);
        }
    }

    public static final String LEGACY_MAPPING =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
            "<!DOCTYPE hibernate-mapping PUBLIC \"-//Hibernate/Hibernate Mapping DTD 3.0//EN\" \"classpath://org/hibernate/hibernate-mapping-3.0.dtd\">\n" +
            "<hibernate-mapping>\n" +
            "    <class name=\"" + TMSVersion.class.getName() + "\" table=\"" + LEGACY_TABLE_NAME + "\">\n" +
            "    <id column=\"" + TMSVersion.PROPERTY_VERSION + "\" name=\"" + TMSVersion.PROPERTY_VERSION + "\"/>\n" +
            "  </class>\n" +
            "</hibernate-mapping>";

    public static Version getVersion(TopiaHibernateSupport tx, String tableName) {
        try {
            Configuration hibernateConfiguration = tx.getHibernateConfiguration();

            // Get schema name
            String schemaName = TopiaUtil.getSchemaName(hibernateConfiguration);

            GetVersionWork work = new GetVersionWork(schemaName, tableName);
            tx.getHibernateSession().doWork(work);
            Version v = work.getVersion();
            return v;
        } catch (TopiaException e) {
            throw new TopiaException("Can't obtain dbVersion for reason " + e.getMessage(), e);
        }
    }

    public static class GetVersionWork implements Work {

        protected Version version;

        private final String tableName;

        private final String schemaName;

        public GetVersionWork(String schemaName, String tableName) {
            this.tableName = tableName;
            this.schemaName = schemaName;
        }

        @Override
        public void execute(Connection connection) throws SQLException {

            String fullTableName = schemaName == null ?
                                   tableName : schemaName + "." + tableName;


            PreparedStatement st = connection.prepareStatement("select " + TMSVersion.PROPERTY_VERSION + " from " + fullTableName + ";");
            try {
                ResultSet set = st.executeQuery();

                if (set.next()) {
                    version = VersionUtil.valueOf(set.getString(1));
                }
            } catch (SQLException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Could not find version", eee);
                }
                version = null;
            } finally {
                st.close();
            }
        }

        public Version getVersion() {
            return version;
        }
    }
}
