package org.nuiton.topia.persistence;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: TopiaQueryBuilderRunQueryStep.java 3131 2014-05-23 08:28:43Z athimel $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-5/topia-persistence/src/main/java/org/nuiton/topia/persistence/TopiaQueryBuilderRunQueryStep.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.util.List;

import org.nuiton.util.pagination.PaginationParameter;
import org.nuiton.util.pagination.PaginationResult;

import com.google.common.base.Optional;

/**
 * This interface represents different common operations that a user may do after a query is defined (using the
 * {@link org.nuiton.topia.persistence.TopiaQueryBuilderAddCriteriaStep})
 * <p/>
 * There are different methods according to the supposed existence or uniqueness of the result. Also some methods may be
 * used only if order is defined in query.
 * <p/>
 * Some methods return an {@link Optional}, but since it's not yet available in JDK, we use Guava's. If you don't want
 * your project to require Guava dependency, we provide equivalent method named XXXOrNull() for the same purpose.
 *
 * @since 3.0
 */
public interface TopiaQueryBuilderRunQueryStep<E extends TopiaEntity> {

    /**
     * @return true if the query returns at least one element
     */
    boolean exists();

    /**
     * This method is equivalent as calling {@link java.util.Collection#size()}
     * after doing a {@link #findAll()} but it may be faster.
     *
     * @return the number of that the query would have returned, if executed
     */
    long count();

    E findUnique() throws TopiaNoResultException, TopiaNonUniqueResultException;

    E findUniqueOrNull() throws TopiaNonUniqueResultException;

    Optional<E> tryFindUnique() throws TopiaNonUniqueResultException;

    /**
     * Get the first element of the non-empty result set.
     *
     * @return the first value from the set of result,
     * according to given order. Returned value
     * can't be null
     * @throws QueryMissingOrderException if you the query
     *                                    misses an order clause
     */
    E findFirst() throws QueryMissingOrderException, TopiaNoResultException;

    /**
     * Get the first element of the result set or null if
     * query result was empty.
     * <p/>
     * This method duplicates {@link #tryFindFirst()}
     * but allows you to prevent using Guava in you project.
     *
     * @return the first value from the set of result,
     * according to given order, or null of result
     * set for given query was empty
     * @throws QueryMissingOrderException if you the query
     *                                    misses an order clause
     */
    E findFirstOrNull() throws QueryMissingOrderException;

    /**
     * Get the first element of the result set.
     * <p/>
     * This method duplicates {@link #tryFindFirst()}
     * but allows you to prevent using Guava in you project.
     * <p/>
     * If the call must return a result, prefer {@link #findFirst()}
     *
     * @return the first value from the set of result,
     * according to given order. It's an optional
     * because the query may return no result.
     * @throws QueryMissingOrderException if you the query
     *                                    misses an order clause
     */
    Optional<E> tryFindFirst() throws QueryMissingOrderException;

    E findAny() throws TopiaNoResultException;

    E findAnyOrNull();

    Optional<E> tryFindAny();

    List<E> findAll();

    Iterable<E> findAllLazy();

    Iterable<E> findAllLazy(int batchSize);

    List<E> find(int startIndex, int endIndex);

    List<E> find(PaginationParameter page);

    PaginationResult<E> findPage(PaginationParameter page);

    List<String> findAllIds();

    List<String> findIds(int startIndex, int endIndex);

    List<String> findIds(PaginationParameter page);

    PaginationResult<String> findIdsPage(PaginationParameter page);

}
