/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: TopiaContextImplTest.java 2269 2011-04-16 22:15:02Z fdesbois $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.5.4/topia-persistence/src/test/java/org/nuiton/topia/framework/TopiaContextImplTest.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.topia.framework;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.h2.Driver;
import org.hibernate.cfg.Configuration;
import org.hibernate.mapping.PersistentClass;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.nuiton.i18n.I18n;
import org.nuiton.topia.TopiaContextFactory;
import org.nuiton.topia.TopiaNotFoundException;
import org.nuiton.topiatest.persistence.Entity1;
import org.nuiton.topiatest.persistence.Entity1Impl;
import org.nuiton.topiatest.service.FakeService;
import org.nuiton.topiatest.service.TestService;

import java.io.File;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;

/**
 * Created: 10 mai 2010
 *
 * @author fdesbois <fdesbois@codelutin.com>
 * @version $Id: TopiaContextImplTest.java 2269 2011-04-16 22:15:02Z fdesbois $
 */
public class TopiaContextImplTest {

    private static final Log log =
            LogFactory.getLog(TopiaContextImplTest.class);

    protected Properties properties = new Properties();

    @BeforeClass
    public static void setUpClass() throws Exception {
        I18n.init(null, Locale.FRENCH);
    }

    @Before
    public void setUp() throws Exception {
        properties.clear();
    }

    @After
    public void tearDown() throws Exception {
    }

    @Test
    public void testLoadServices() throws Exception {
        log.info("## testLoadServices");

        /** PREPARE DATA **/
        properties.setProperty("topia.service.test",
                TestService.class.getName());

        TopiaContextImpl context = new TopiaContextImpl();

        /** EXEC METHOD **/
        log.info("test 1 : load a simple TestService from properties");
        Map<String, TopiaService> results = context.loadServices(properties);
        Assert.assertEquals(1, results.size());
        Assert.assertTrue(results.containsKey("test"));
        TopiaService service = results.get("test");
        Assert.assertEquals(TestService.class, service.getClass());

        log.info("test 2 : load with wrong key : will display a WARN");
        properties.clear();

        properties.setProperty("topia.service.fake",
                TestService.class.getName());

        results = context.loadServices(properties);
        Assert.assertEquals(0, results.size());
        Assert.assertFalse(results.containsKey("fake"));

        log.info("test 3 : load with fake service name : will display an ERROR");
        properties.clear();

        properties.setProperty("topia.service.test", "FAKE");

        results = context.loadServices(properties);
        Assert.assertEquals(0, results.size());
        Assert.assertFalse(results.containsKey("test"));

        // TODO-fdesbois-20100510 : need integration tests for all existing topia services
    }

    @Test
    public void testGetServices() throws Exception {
        log.info("## testGetServices");

        /** PREPARE DATA **/
        properties.setProperty("topia.service.test",
                TestService.class.getName());

        // Calling the constructor with properties will load the services
        TopiaContextImpl context = new TopiaContextImpl(properties);

        // Instantiate a child context and set its parent
        TopiaContextImpl child = new TopiaContextImpl(context);

        /** EXEC METHOD **/
        log.info("test 1 : with child context");
        Map<String, TopiaService> test1 = child.getServices();
        Assert.assertEquals(1, test1.size());
        Assert.assertTrue(test1.containsKey("test"));

        log.info("test 2 : test serviceEnabled method");
        boolean test2 = child.serviceEnabled("test");
        Assert.assertTrue(test2);

        log.info("test 3 : test getService method");
        TopiaService test3 = child.getService("test");
        Assert.assertEquals(TestService.class, test3.getClass());

        log.info("test 4 : test serviceEnabled from class TestService");
        boolean test4 = child.serviceEnabled(TestService.class);
        Assert.assertTrue(test4);

        log.info("test 5 : test getService from class TestService");
        TestService test5 = child.getService(TestService.class);
        Assert.assertNotNull(test5);

        log.info("test 6 : test serviceEnabled error with class FakeService");
        // FakeService doesn't contains property SERVICE_NAME used by
        // serviceEnabled method
        // Even it's properly loaded the serviceEnabled method will return false
        properties.clear();
        properties.setProperty("topia.service.fake",
                FakeService.class.getName());
        TopiaContextImpl otherContext = new TopiaContextImpl(properties);

        boolean test6 = otherContext.serviceEnabled(FakeService.class);
        Assert.assertFalse(test6);

        log.info("test 7 : test getService with error TopiaNotFoundException" +
                " : service not loaded");
        // TestService is not loaded in otherContext
        try {
            TestService test7 = otherContext.getService(TestService.class);
        } catch (Exception eee) {
            log.error(eee.getClass().getSimpleName() + " : " + eee.getMessage());
            Assert.assertEquals(TopiaNotFoundException.class, eee.getClass());
        }
    }

    @Test
    public void testContextHierarchy() throws Exception {
        log.info("## testContextHierarchy");

        /** PREPARE DATA **/
        TopiaContextImpl context = new TopiaContextImpl(properties);

        /** EXEC METHODS **/
        log.info("test 1 : constructor with parent context");
        TopiaContextImpl test1 = new TopiaContextImpl(context);
        Assert.assertEquals(context, test1.parentContext);

        log.info("test 2 : addChildContext");
        TopiaContextImpl test2 = new TopiaContextImpl(properties);
        TopiaContextImpl child2 = new TopiaContextImpl();
        test2.addChildContext(child2);
        Assert.assertEquals(1, test2.childContext.size());

        log.info("test 3 : removeChildContext");
        TopiaContextImpl test3 = new TopiaContextImpl(properties);
        TopiaContextImpl child3 = new TopiaContextImpl(test3);
        test3.childContext.add(child3);
        test3.removeChildContext(child3);
        Assert.assertEquals(0, test3.childContext.size());

        // No remove if context is closed
        test3.childContext.add(child3);
        test3.closed = true;
        test3.removeChildContext(child3);
        Assert.assertEquals(1, test3.childContext.size());

        log.info("test 4 : getRootContext");
        TopiaContextImplementor test4 = child3.getRootContext();
        Assert.assertEquals(test3, test4);

        // Note : existing test is already done for concurrency problem on
        // getChildContext(). Go to : http://www.nuiton.org/repositories/browse/sandbox/testTopiaPostgresError/trunk
    }
//
//    @Test
//    public void testCreateSchema() throws Exception {
//    }
//
//    @Test
//    public void testShowCreateSchema() throws Exception {
//    }
//
//    @Test
//    public void testUpdateSchema() throws Exception {
//    }
//
//    @Test
//    public void testGetHibernate() throws Exception {
//    }
//

    @Test
    public void testGetHibernateFactory() throws Exception {
        log.info("## testGetHibernateFactory");

        /** PREPARE DATA **/
        TopiaContextImpl context = new TopiaContextImpl();
        context.services = new HashMap<String, TopiaService>();

        String basedir = System.getenv("basedir");
        if (basedir == null) {

            // says basedir is where we start tests.
            basedir = new File("").getAbsolutePath();
        }

        log.debug("baseDir : " + basedir);
        File persistenceDir = new File(basedir,
                                     "target" + File.separator +
                                     "test-classes" +File.separator +
                                     "org" + File.separator +
                                     "nuiton" + File.separator +
                                     "topiatest" + File.separator +
                                     "persistence");
        log.debug("persistenceDir : " + persistenceDir);
        File resourcesDir = new File(basedir,
                                     "target" + File.separator +
                                     "test-classes");

        /** EXEC METHOD **/
        log.info("test 1 : load mappings from directory");
     
        properties.setProperty(TopiaContextFactory.CONFIG_PERSISTENCE_DIRECTORIES,
                persistenceDir.getAbsolutePath());
        context.config = properties;

        Configuration test1 = context.getHibernateConfiguration();
        PersistentClass persistentClass =
                test1.getClassMapping(Entity1Impl.class.getName());
        Assert.assertNotNull(persistentClass);
        Assert.assertEquals(Entity1.class, persistentClass.getProxyInterface());

//        for (Iterator<RootClass> it = test1.getClassMappings(); it.hasNext();) {
//            RootClass o = it.next();
//            log.debug("entity : " + o.getEntityName());
//        }

        log.info("test 2 : load mappings for all entities");
        //reset from previous test
        context = new TopiaContextImpl();
        context.services = new HashMap<String, TopiaService>();
        properties.clear();

        // use property TOPIA_PERSISTENCE_CLASSES
        properties.setProperty(TopiaContextFactory.CONFIG_PERSISTENCE_CLASSES,
                Entity1Impl.class.getName());
        context.config = properties;

        Configuration test2 = context.getHibernateConfiguration();
        persistentClass = test2.getClassMapping(Entity1Impl.class.getName());
        Assert.assertNotNull(persistentClass);
        Assert.assertEquals(Entity1.class, persistentClass.getProxyInterface());

        log.info("test 3 : add properties from file");
        //reset from previous test
        context = new TopiaContextImpl();
        context.services = new HashMap<String, TopiaService>();
        properties.clear();

        // use property TOPIA_PERSISTENCE_PROPERTIES_FILE to add default
        // properties from file
        properties.setProperty(TopiaContextFactory.CONFIG_PERSISTENCE_PROPERTIES_FILE,
                resourcesDir + File.separator + "TopiaContextImpl.properties");
        context.config = properties;

        Configuration test3 = context.getHibernateConfiguration();
        Assert.assertEquals(
                test3.getProperty("hibernate.connection.driver_class"),
                Driver.class.getName());

        // Note : maybe add a test to load classes from services
    }
//
//    @Test
//    public void testGetHibernateConfiguration() throws Exception {
//    }
//
//    @Test
//    public void testGetDAO() throws Exception {
//    }
//
//    @Test
//    public void testBeginTransaction() throws Exception {
//    }
//
//    @Test
//    public void testCommitTransaction() throws Exception {
//    }
//
//    @Test
//    public void testRollbackTransaction() throws Exception {
//    }
//
//    @Test
//    public void testCloseContext() throws Exception {
//    }
//
//    @Test
//    public void testIsClosed() throws Exception {
//    }
//
//    @Test
//    public void testFindByTopiaId() throws Exception {
//    }
//
//    @Test
//    public void testFind() throws Exception {
//    }
//
//    @Test
//    public void testFind2() throws Exception {
//    }
//
//    @Test
//    public void testExecute() throws Exception {
//    }
//
//    @Test
//    public void testAdd() throws Exception {
//    }
//
//    @Test
//    public void testImportXML() throws Exception {
//    }
//
//    @Test
//    public void testExportXML() throws Exception {
//    }
//
//    @Test
//    public void testReplicate() throws Exception {
//    }
//
//    @Test
//    public void testReplicateEntity() throws Exception {
//    }
//
//    @Test
//    public void testReplicateEntities() throws Exception {
//    }
//
//    @Test
//    public void testGetFiresSupport() throws Exception {
//    }
//
//    @Test
//    public void testBackup() throws Exception {
//    }
//
//    @Test
//    public void testRestore() throws Exception {
//    }
//
//    @Test
//    public void testClear() throws Exception {
//    }
//
//    @Test
//    public void testGetPersistenceClasses() throws Exception {
//    }
//
//    @Test
//    public void testIsSchemaExist() throws Exception {
//    }
}
