/*
 * *##% 
 * Nuiton Processor :: Api
 * Copyright (C) 2002 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */

/* *
 * I18nExtractor.java
 *
 * Created: Aug 16, 2004
 *
 * @author Cédric Pineau <pineau@codelutin.com>
 * @version $Revision: 317 $
 *
 * Last update : $Date: 2010-03-09 19:22:11 +0100 (mar., 09 mars 2010) $
 * by : $Author: tchemit $
 */

package org.nuiton.processor;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.nuiton.processor.filters.I18nFilter;

/**
 * 
 */
public class I18nExtractor {

    protected I18nFilter filter = new I18nFilter();
    
    protected Map<String,List<String>> i18nStringsMap =
            new HashMap<String,List<String>>();
    
    public void extract(File[] srcFiles, File outputFile) throws IOException {
        PrintWriter writer = new PrintWriter(
                new BufferedWriter(
                        new OutputStreamWriter(
                                new FileOutputStream(outputFile), "UTF-8")));
        try {
            for (File srcFile : srcFiles) {
                processFile(srcFile);
            }

            for (Object o : i18nStringsMap.keySet()) {
                String i18nString = (String) o;
                List<?> locations = i18nStringsMap.get(i18nString);
                for (Object location1 : locations) {
                    String location = (String) location1;
                    writer.println("# " + location);
                }
                writer.println(i18nString + "=");
            }
        } finally {
            writer.close();
        }
    }
    
    protected void processFile(File srcFile) throws IOException {
        LineNumberReader lnr = new LineNumberReader(
                new InputStreamReader(new FileInputStream(srcFile), "UTF-8"));
        try {
            while (lnr.ready()) {
                String line = lnr.readLine();
                int lineNumber = lnr.getLineNumber();
                processLine(line, srcFile, lineNumber);
            }
        } finally {
            lnr.close();
        }
    }

    protected void processLine(String line, File srcFile, int lineNumber) {
        String i18nStringsSet = filter.parse(line);
        if (I18nFilter.EMPTY_STRING.equals(i18nStringsSet)) {
            // Found a set of i18n Strings, split it.
            String[] i18nStrings = i18nStringsSet.split("=");
            for (String i18nString : i18nStrings) {
                List<String> locations = i18nStringsMap.get(i18nString);
                if (locations == null) {
                    locations = new ArrayList<String>();
                    i18nStringsMap.put(i18nString, locations);
                }
                locations.add(srcFile.getPath() + "(" + lineNumber + ")");
            }
        }
    }

    public static void main(String[] args) {
        if(args.length < 2) {
            System.out.println(
                    "Please give at least sources and destination file");
            System.exit(0);
        } else if (args.length > 2) {
            File[] sources = new File[args.length-1];
            for (int i = 0; i < args.length-1; i++) {
                sources[i] =new File(args[i]);
            }
            File destination = new File(args[args.length - 1]);
            try {
                new I18nExtractor().extract(sources, destination);
            } catch (IOException eee) {
                Logger logger =
                        Logger.getLogger("org.nuiton.processor.I18nExtractor.");
                logger.log(Level.SEVERE,
                           "Error during i18n extraction : " + eee, eee);
            }
        }
        
    }
}
