/*
 * #%L
 * Nuiton Utils :: Nuiton Validator
 * 
 * $Id: BeanValidatorFactory.java 2068 2011-01-27 08:10:53Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.5/nuiton-validator/src/main/java/org/nuiton/validator/bean/BeanValidatorFactory.java $
 * %%
 * Copyright (C) 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.validator.bean;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.validator.NuitonValidator;
import org.nuiton.validator.NuitonValidatorFactory;
import org.nuiton.validator.NuitonValidatorProvider;
import org.nuiton.validator.NuitonValidatorScope;

/**
 * Factory of {@link BeanValidator}.
 * <p/>
 * To obtain a new {@link BeanValidator}, use one of the method
 * <p/>
 * <pre>
 * BeanValidatorFactory.newBeanValidator(XXX)
 * </pre>
 *
 * @author tchemit <chemit@codelutin.com>
 * @see BeanValidator
 * @since 2.0
 */
public class BeanValidatorFactory {


    /** Logger. */
    static private final Log log =
            LogFactory.getLog(BeanValidatorFactory.class);


    /**
     * Object to create new validators.
     * <p/>
     * If none is given, it will use the default one, says the
     * {@link DefaultBeanValidatorCreator}.
     */
    protected static BeanValidatorCreator creator;

    /**
     * Obtain the bean validator creator.
     * <p/>
     * If none is defined, then will use a default one, says the
     * {@link DefaultBeanValidatorCreator}.
     *
     * @return the bean validator creator
     */
    public static BeanValidatorCreator getCreator() {
        if (creator == null) {
            creator = new DefaultBeanValidatorCreator();
            if (log.isInfoEnabled()) {
                log.info("Will use default beanValidator creator : " + creator);
            }

        }
        return creator;
    }

    /**
     * Sets an explicit bean validator creator.
     *
     * @param creator the creator to use
     */
    public static void setCreator(BeanValidatorCreator creator) {
        BeanValidatorFactory.creator = creator;
        if (log.isInfoEnabled()) {
            log.info("Will use injected beanValidator creator : " + creator);
        }
    }

    public static boolean isDefaultCreator() {
        return creator == null ||
               creator.getClass().equals(DefaultBeanValidatorCreator.class);
    }

    /**
     * Obtain a new {@link BeanValidator} for the given parameters.
     * <p/>
     * <b>Note:</b> It will use the default provider of {@link NuitonValidator}
     *
     * @param type    type of bean to validate
     * @param context context of validation
     * @param scopes  authorized scopes (if {@code null}, will use all scopes)
     * @param <O>     type of bean to validate
     * @return the new instanciated {@link BeanValidator}.
     * @throws NullPointerException if type is {@code null}
     * @see NuitonValidatorFactory#getDefaultProviderName()
     */
    public static <O> BeanValidator<O> newBeanValidator(Class<O> type,
                                                        String context,
                                                        NuitonValidatorScope... scopes) throws NullPointerException {


        // get the provider default name
        String providerName = NuitonValidatorFactory.getDefaultProviderName();

        // get the bean validator with this provider
        BeanValidator<O> beanValidator = newBeanValidator(providerName,
                                                          type,
                                                          context,
                                                          scopes
        );
        return beanValidator;
    }

    /**
     * Obtain a new {@link BeanValidator} for the given parameters.
     * <p/>
     * <b>Note:</b> It will use the provider of {@link NuitonValidator}
     * defined by the {@code providerName}.
     *
     * @param providerName name of {@link NuitonValidator} to use
     * @param type         type of bean to validate
     * @param context      context of validation
     * @param scopes       authorized scopes (if {@code null}, will use all scopes)
     * @param <O>          type of bean to validate
     * @return the new instanciated {@link BeanValidator}.
     * @throws NullPointerException if type is {@code null}
     * @see NuitonValidatorFactory#getProvider(String)
     */
    public static <O> BeanValidator<O> newBeanValidator(String providerName,
                                                        Class<O> type,
                                                        String context,
                                                        NuitonValidatorScope... scopes) throws NullPointerException {

        if (type == null) {
            throw new NullPointerException(
                    "type parameter can not be null.");
        }

        // get delegate validator provider
        NuitonValidatorProvider provider =
                NuitonValidatorFactory.getProvider(providerName);

        // get bean validator creator
        BeanValidatorCreator creator = getCreator();

        // create the new instance of bean validator
        BeanValidator<O> beanValidator =
                creator.newBeanValidator(provider, type, context, scopes);

        return beanValidator;
    }

    /**
     * Contract to create a {@link BeanValidator}.
     * <p/>
     * A such object is registred in the {@link BeanValidatorFactory} to create
     * new instances of a {@link BeanValidator}.
     *
     * @see BeanValidatorFactory
     * @see BeanValidatorFactory#getCreator()
     * @see BeanValidatorFactory#setCreator(BeanValidatorCreator)
     * @see BeanValidatorFactory#isDefaultCreator()
     * @since 2.0
     */
    public interface BeanValidatorCreator {

        /**
         * Given the parameters, instanciate a new {@link BeanValidator}.
         *
         * @param provider the delegate validator provider
         * @param type     the type of object to validate
         * @param context  the context of validation (can be {@code null})
         * @param scopes   scopes to use (if none given, will use all available scopes)
         * @param <O>      type of object to validate
         * @return the new instance of bean validator
         */
        <O> BeanValidator<O> newBeanValidator(
                NuitonValidatorProvider provider,
                Class<O> type,
                String context,
                NuitonValidatorScope... scopes

        );
    }

    /**
     * Default implementation of {@link BeanValidatorCreator} which just
     * instanciate a {@link BeanValidator} from the given parameters.
     * <p/>
     * If no bean validator creator is given to the {@link BeanValidatorFactory}
     * it will instanciate a such creator and use it unless you change it.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 2.0
     */
    public static class DefaultBeanValidatorCreator implements BeanValidatorCreator {
        @Override
        public <O> BeanValidator<O> newBeanValidator(
                NuitonValidatorProvider provider,
                Class<O> type,
                String context,
                NuitonValidatorScope... scopes) {

            BeanValidator<O> beanValidator = new BeanValidator<O>(provider,
                                                                  type,
                                                                  context,
                                                                  scopes
            );
            return beanValidator;
        }
    }
}
