/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: ConverterUtil.java 2360 2012-06-11 10:24:36Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.6/nuiton-utils/src/main/java/org/nuiton/util/converter/ConverterUtil.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.util.converter;

import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.lang.reflect.Method;
import java.util.ServiceLoader;

/**
 * Une classe contenant des méthodes utiles sur les converters et les conversions
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3 (replace {@code org.nuiton.util.ConverterUtil}).
 */
public class ConverterUtil {

    /** Logger. */
    static private Log log = LogFactory.getLog(ConverterUtil.class);

    /**
     * le paquetage où chercher les implentations de Converter, si non présents
     * dans le système
     */
    protected static final String CONVERTER_PACKAGE = "org.nuiton.util.converter";

    /** un drapeau pour savoir si on doit charger les converters specifiques */
    protected static Boolean WAS_INIT = Boolean.FALSE;

    /**
     * Cherche un converter pour un <code>type</code> donné.
     * <p/>
     * Recherche dans un premier temps dans les converteurs déjà connus.
     * <p/>
     * Si le type est une énum et qu'aucun converter, n'a été trouvé, on
     * enregistre un nouveau convert d'enum.
     * <p/>
     * Sinon on tente d'instancier un converteur dans le paquetage dédié aux
     * converteurs {@link #CONVERTER_PACKAGE}.
     *
     * @param <T>  le type a convertir
     * @param type le type a convertir
     * @return le converter trouvé, ou null si non trouvé
     */
    public static <T> Converter getConverter(Class<T> type) {
        if (!WAS_INIT) {
            initConverters();
        }
        Converter converter = ConvertUtils.lookup(type);
        if (converter != null) {
            return converter;
        }
        if (type.isEnum()) {
            registerEnumConverter(type);
            return ConvertUtils.lookup(type);
        }
        // on essaye de trouver un converter dans le paquetage des converters
        try {
            registerConverter0(type);
            converter = ConvertUtils.lookup(type);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return converter;
    }

    /**
     * Convertir une valeur!
     *
     * @param <T>       le type de donnee recherchee
     * @param type      le type de donnee recherchee
     * @param toConvert l'object a convertir
     * @return la nouvelle instance de l'objet converti type ou null
     */
    @SuppressWarnings({"unchecked"})
    public static <T> T convert(Class<T> type, Object toConvert) {
        if (!WAS_INIT) {
            initConverters();
        }
        T result = null;
        Converter converter = getConverter(type);
        if (converter != null) {
            return (T) converter.convert(type, toConvert);
        }
        return result;
    }

    public static void registerConverter(Class<?> type)
            throws IllegalAccessException,
            InstantiationException,
            ClassNotFoundException {
        if (ConvertUtils.lookup(type) == null) {
            registerConverter0(type);
        }
    }

    protected static void registerConverter0(Class<?> type)
            throws IllegalAccessException,
            InstantiationException,
            ClassNotFoundException {
        Class<?> aClass = Class.forName(
                CONVERTER_PACKAGE + "." + type.getSimpleName() + "Converter");
        Converter converter = (Converter) aClass.newInstance();
        log.info("for type : " + type + " : " + converter);
        ConvertUtils.register(converter, type);
    }

    /**
     * Enregistre un nouveau converter pour un type d'enum donné, avec  une
     * valeur par defaut.
     *
     * @param type         le type d'enum à convertir
     * @param defaultValue la valeur par defaut.
     */
    public static void registerEnumConverter(Class<?> type,
                                             Object defaultValue) {
        if (EnumConverter.isEnabled(type, type) &&
            ConvertUtils.lookup(type) == null) {
            Converter converter = new EnumConverter(type, defaultValue);
            log.info("for type : " + type + " : " + converter);
            ConvertUtils.register(converter, type);
        }
    }

    /**
     * Enregistre un nouveau converter pour un type d'enum donné, sans utiliser
     * de valeur par defaut.
     *
     * @param type le type d'enum à convertir
     */
    public static void registerEnumConverter(Class<?> type) {
        registerEnumConverter(type, null);
    }

    public static byte[] convert(char[] chars) {
        byte[] bytes = new byte[chars.length];
        for (int i = 0; i < chars.length; i++) {
            bytes[i] = (byte) (chars[i] & 0xff);
        }
        return bytes;
    }

    public static synchronized void deregister() {
        ConvertUtils.deregister();
        WAS_INIT = false;
    }

    public static synchronized void initConverters() {
        if (WAS_INIT != null && WAS_INIT) {
            return;
        }
        ServiceLoader<Converter> converters =
                ServiceLoader.load(Converter.class);
        for (Converter converter : converters) {
            if (log.isDebugEnabled()) {
                log.debug("discovered converter " + converter);
            }
            try {
                Method m = converter.getClass().getDeclaredMethod("getType");
                m.setAccessible(true);
                try {
                    Class<?> returnType = (Class<?>) m.invoke(converter);
                    log.info("register converter " + converter);
                    ConvertUtils.register(converter, returnType);
                } catch (Exception ex) {
                    log.warn("could not obtain type of converter " +
                             converter + " for reason : " + ex.getMessage(),
                             ex);
                }
            } catch (NoSuchMethodException ex) {
                log.warn("could not find method getType on converter " +
                         converter + ", will not be registred...");
            } catch (SecurityException ex) {
                log.warn("could not find method getType on converter " +
                         converter + ", will not be registred...");
            }

        }
        WAS_INIT = true;
    }
}
