/*
 * #%L
 * Nuiton Utils :: Nuiton Utils
 * 
 * $Id: PagerUtil.java 2279 2012-01-16 14:39:04Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.6/nuiton-utils/src/main/java/org/nuiton/util/PagerUtil.java $
 * %%
 * Copyright (C) 2004 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.util;

import org.apache.commons.lang3.tuple.Pair;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * Common methods to deal with a pager.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.4
 */
public class PagerUtil {

    /**
     * Given a total count of elements and a page size, compute the number of
     * pages available.
     *
     * @param totalCount total count of elements in the pager
     * @param pageSize   size of a page
     * @return the number of available pages
     */
    public static int getTotalPage(int totalCount, int pageSize) {
        int result = (int) Math.ceil((double) totalCount / (double) pageSize);
        return result;
    }

    /**
     * Compute the pager bound given his datas:
     * <ul>
     * <li>{@code totalCount}: count of all elements of the pager</li>
     * <li>{@code pageSize}: number of elements in a page</li>
     * <li>{@code page}: the requested page number starting at {@code 1}</li>
     * </ul>
     *
     * @param totalCount total count of elements in the pager
     * @param page       pager page number (starting at {@code 1})
     * @param pageSize   number of elements in a page
     * @return the bound of start and end index of the requested elements of the pager
     */
    public static Pair<Integer, Integer> getPageBound(int totalCount,
                                                      int page,
                                                      int pageSize) {

        page = Math.max(page, 1);
        pageSize = Math.max(pageSize, 1);

        int start = (page - 1) * pageSize;
        int end = page * pageSize; // End in subList is exclusive

        end = Math.min(end, totalCount); //To be sure that end will not be out of bound in subList

        Pair<Integer, Integer> result = Pair.of(start, end);
        return result;
    }

    /**
     * Get the elements of the lists using the pager datas:
     * <p/>
     * <ul>
     * <li>{@code elements}:all elements of the pager</li>
     * <li>{@code pageSize}: number of elements in a page</li>
     * <li>{@code page}: the requested page number starting at {@code 1}</li>
     * </ul>
     *
     * @param elements all pager elements
     * @param page     pager page number (starting at {@code 1})
     * @param pageSize number of elements in a page
     * @param <E>      n'importe quel type
     * @return la liste une fois filtrée
     */
    public static <E> List<E> getPage(List<E> elements, int page, int pageSize) {

        Pair<Integer, Integer> bound = getPageBound(elements.size(), page, pageSize);

        int start = bound.getLeft();
        int end = bound.getRight();

        List<E> subList = elements.subList(start, end);
        List<E> result = new ArrayList<E>(subList);

        return result;
    }

    /**
     * Given a pagerBean, compute his {@link PagerBean#recordStartIndex},
     * {@link PagerBean#recordEndIndex} and {@link PagerBean#pagesNumber}.
     *
     * @param bean the pagerBean to fill
     * @since 2.4.3
     */
    public static void computeRecordIndexesAndPagesNumber(PagerBean bean) {
        int records = bean.getRecords();
        int pageSize = bean.getPageSize();

        Pair<Integer, Integer> pageBound =
                getPageBound(records, bean.getPageIndex(), pageSize);
        bean.setRecordStartIndex(pageBound.getLeft());
        bean.setRecordEndIndex(pageBound.getRight());
        bean.setPagesNumber(getTotalPage(records, pageSize));
    }

    public static PagerBean newPagerBean() {
        return new PagerBean();
    }

    /**
     * Pager bean containing all internal states of a pager.
     *
     * @since 2.4.3
     */
    public static class PagerBean implements Serializable {

        private static final long serialVersionUID = 1L;

        /** Number of records (says numbers of rows displayable. */
        protected int records;

        /** Index of the first record in this page. */
        protected int recordStartIndex;

        /** Index of the last record in this page. */
        protected int recordEndIndex;

        /** Index of the page. */
        protected int pageIndex;

        /** Page size, says number of records in a page. */
        protected int pageSize;

        /** Number of pages. */
        protected int pagesNumber;

        public int getRecords() {
            return records;
        }

        public int getRecordStartIndex() {
            return recordStartIndex;
        }

        public int getRecordEndIndex() {
            return recordEndIndex;
        }

        public int getPageIndex() {
            return pageIndex;
        }

        public int getPageSize() {
            return pageSize;
        }

        public int getPagesNumber() {
            return pagesNumber;
        }

        public void setRecords(int records) {
            this.records = records;
        }

        public void setRecordStartIndex(int recordStartIndex) {
            this.recordStartIndex = recordStartIndex;
        }

        public void setRecordEndIndex(int recordEndIndex) {
            this.recordEndIndex = recordEndIndex;
        }

        public void setPageIndex(int pageIndex) {
            this.pageIndex = pageIndex;
        }

        public void setPageSize(int pageSize) {
            this.pageSize = pageSize;
        }

        public void setPagesNumber(int pagesNumber) {
            this.pagesNumber = pagesNumber;
        }

    }
}
