/*
 * #%L
 * Nuiton Utils :: Nuiton Utils
 * 
 * $Id: ApplicationConfigHelper.java 2464 2013-01-13 11:16:02Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.6.6/nuiton-utils/src/main/java/org/nuiton/util/ApplicationConfigHelper.java $
 * %%
 * Copyright (C) 2004 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.HashSet;
import java.util.ServiceLoader;
import java.util.Set;

/**
 * Helper about {@link ApplicationConfig}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.4.8
 */
public class ApplicationConfigHelper {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(ApplicationConfigHelper.class);

    protected ApplicationConfigHelper() {
        // helper with no instance
    }

    /**
     * Obtain all providers on class-path.
     *
     * @param classLoader optional classLoader used to seek for providers
     * @param includes    optional includes providers to use (if none then accept all providers)
     * @param excludes    optional excludes providers (if none the no reject)
     * @param verbose     verbose flag
     * @return sets of providers
     */
    public static Set<ApplicationConfigProvider> getProviders(ClassLoader classLoader,
                                                              Set<String> includes,
                                                              Set<String> excludes,
                                                              boolean verbose) {
        ServiceLoader<ApplicationConfigProvider> loader;
        if (classLoader == null) {
            loader = ServiceLoader.load(ApplicationConfigProvider.class);

        } else {
            loader = ServiceLoader.load(ApplicationConfigProvider.class,
                                        classLoader);
        }

        Set<ApplicationConfigProvider> result =
                new HashSet<ApplicationConfigProvider>();

        for (ApplicationConfigProvider configProvider : loader) {
            String name = configProvider.getName();
            if (includes != null && !includes.contains(name)) {

                // reject by include
                if (verbose) {
                    log.info("configuration named '" + name +
                             "' is rejected by includes.");
                }
                continue;
            }
            if (excludes != null && excludes.contains(name)) {

                // reject by exclude
                if (verbose) {
                    log.info("configuration named '" + name +
                             "' is rejected by excludes.");
                }
                continue;
            }
            if (verbose) {
                log.info("configuration named '" + name +
                         "' will be generated.");
            }
            result.add(configProvider);
        }
        return result;
    }

    public static ApplicationConfigProvider getProvider(ClassLoader classLoader,
                                                        String name) {
        Set<ApplicationConfigProvider> providers = getProviders(
                classLoader, null, null, false);
        ApplicationConfigProvider result = null;
        for (ApplicationConfigProvider provider : providers) {
            if (name.equals(provider.getName())) {
                result = provider;
                break;
            }
        }
        return result;
    }

    /**
     * Load default options from all config provider found in classpath.
     *
     * @param config      config where to add default options.
     * @param classLoader optional classLoader
     * @param includes    config to include (if none given all is include)
     * @param excludes    config to exclude (if none given, no exclude)
     * @param verbose     verbose flag
     * @since 2.6.6
     */
    public static void loadAllDefaultOption(ApplicationConfig config,
                                            ClassLoader classLoader,
                                            Set<String> includes,
                                            Set<String> excludes,
                                            boolean verbose) {

        Set<ApplicationConfigProvider> providers = getProviders(classLoader,
                                                                includes,
                                                                excludes,
                                                                verbose);
        for (ApplicationConfigProvider provider : providers) {
            if (log.isInfoEnabled()) {
                log.info("Load default options from configuration: " +
                         provider.getName());
            }
            if (verbose && log.isInfoEnabled()) {
                for (ApplicationConfig.OptionDef optionDef : provider.getOptions()) {
                    log.info(" " + optionDef.getKey() +
                             " (" + optionDef.getDefaultValue() + ')');
                }
            }
            config.loadDefaultOptions(provider.getOptions());
        }

    }
}
