/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: FormatMap.java 2360 2012-06-11 10:24:36Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.6.4/nuiton-utils/src/main/java/org/nuiton/util/converter/FormatMap.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.util.converter;

import java.util.HashMap;
import java.util.Map;

/**
 * Classe servant de conteneur pour les différentes representation d'un objet
 * Les representations doivents être des instances de {@link Format}. Le mieux
 * est lors de l'ecriture des convertisseurs pour un nouveau format est de
 * créer une instance final static de {@link Format} Format pour representer
 * ce format
 * <p/>
 * <h2>Utilisation</h2>
 * <pre>
 * FormatMap values = new FormatMap(MonObject.class);
 * values.put(FormatConverter.FORMAT_JAVA, monInstance);
 * Element xml = (Element)values.convert(XMLConverter.FORMAT_XML);
 * Object sql = values.convert(SQLConverter.FORMAT_SQL);
 * </pre>
 * <pre>
 * FormatMap values = new FormatMap(MonObject.class);
 * values.put(FormatConverter.FORMAT_XML, monInstanceEnXML);
 * Object sql = values.convert(SQLConverter.FORMAT_SQL);
 * </pre>
 * Dans ce second cas, la demande de la version SQL, transforme automatiquement
 * la representation XML qui est la seul presente en Java, puis a partir de
 * cette representation Java, on recupere la representation SQL. Bien sur
 * Si le convertisseur SQL, peut directement convertir le XML en SQL, alors
 * la conversion Java ne sera pas faite.
 * <p/>
 * Il est souvent plus simple de faire une petite classe avec les methodes
 * getSQL() et getXML(), qui retourne les valeurs directement dans le bon type
 * et qui n'ont pas besoin d'argument.
 * <p/>
 * <p/>
 * Created: 16 septembre 2005 10:41:58 CEST
 *
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 * @version $Id: FormatMap.java 2360 2012-06-11 10:24:36Z tchemit $
 * @since 1.3 (replace {@code org.nuiton.util.FormatMap} class).
 */
public class FormatMap extends HashMap<FormatMap.Format, Object> { // FormatMap

    private static final long serialVersionUID = 1L;

    public static class Format {

        protected String name;

        public Format(String name) {
            this.name = name;
        }

        @Override
        public String toString() {
            return name;
        }
    }

    protected Class<?> clazz;

    public FormatMap(Class<?> clazz) {
        this.clazz = clazz;
    }

    public Class<?> getType() {
        return clazz;
    }

    /**
     * Met a jour la valeur de l'objet. Toutes les autres valeurs calculées
     * sont oubliées et seront recalculé en fonction de cette nouvelle valeur
     *
     * @param format le format a utiliser
     * @param value
     */
    public void setValue(Format format, Object value) {
        clear();
        put(format, value);
    }

    /**
     * Utilise le FormatConverterFactory par defaut pour la conversion
     *
     * @param format le format a utiliser
     * @param args   les arguments
     * @return l'objet converti
     */
    public Object convert(Format format, Object... args) {
        return convert(FormatConverterFactory.getInstance(), format, args);
    }

    /**
     * Recupere la valeur dans le format demandé
     *
     * @param factory la FormatConverterFactory a utiliser
     * @param format  le format souhaité
     * @param args    des arguments facultatifs en fonction du context d'utilisation
     * @return l'objet converti
     */
    public Object convert(FormatConverterFactory factory, Format format,
                          Object... args) {
        Object result;
        Map<Format, Object> values = this;
        //if (!values.containsKey(format) ||
        //        !format.equals(FormatConverter.FORMAT_JAVA)) {
        //    throw new IllegalArgumentException("Aucun valeur disponible");
        //}

        if (values.containsKey(format)) {
            result = values.get(format);
        } else if (format.equals(FormatConverter.FORMAT_JAVA)) {
            // on recherche une representation, que l'on convertie en Java
            if (values.isEmpty()) {
                throw new IllegalArgumentException("Aucun valeur disponible");
            }
            Format otherFormat = values.keySet().iterator().next();
            result = unconvert(factory, otherFormat, args);
            values.put(format, result);
        } else {
            result = factory.convert(format, this, args);
            // on stocke la valeur trouver pour les prochaines fois
            values.put(format, result);
        }

        return result;
    }

    /**
     * Utilise le FormatConverterFactory par defaut pour la conversion
     *
     * @param format le format utilise
     * @param args
     * @return l'objet java
     */
    public Object unconvert(Format format, Object... args) {
        return unconvert(FormatConverterFactory.getInstance(), format, args);
    }

    /**
     * Donne la representation Java de l'objet en essayant de partir de la
     * representation passé en parametre.
     *
     * @param factory la factory a utiliser
     * @param format  le format de départ souhaité
     * @param args    des arguments facultatifs en fonction du context d'utilisation
     * @return l'objet java
     */
    public Object unconvert(FormatConverterFactory factory, Format format,
                            Object... args) {
        Object result = null;
        Map<Format, Object> values = this;
        if (!values.containsKey(format) &&
            !values.containsKey(FormatConverter.FORMAT_JAVA)) {
            throw new IllegalArgumentException("Aucun valeur disponible");
        }

        // si on a deja la representation Java on la retourne tout de suite
        if (values.containsKey(FormatConverter.FORMAT_JAVA)) {
            result = values.get(FormatConverter.FORMAT_JAVA);
        } else if (values.containsKey(format)) {
            result = factory.unconvert(format, this, args);
            values.put(FormatConverter.FORMAT_JAVA, result);
        }

        return result;
    }
} // FormatMap

