/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: FormatConverterFactory.java 1851 2010-05-11 11:24:21Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-1.5/src/main/java/org/nuiton/util/converter/FormatConverterFactory.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.util.converter;

import org.apache.commons.collections.map.MultiKeyMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.converter.FormatMap.Format;

import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;

/**
 * Factory permet d'enregistrer des objets de changement de format, et de
 * les recupérer pour les utiliser.
 * Les objets converter doivent au moins savoir convertir les objets depuis
 * une representation Java. Pour des raisons d'optimisation, il est possible
 * qu'il sache aussi convertir a partir d'autre representation, qui si elle
 * existe sont moins couteuse a convertir.
 * Il faut aussi que les converter sache convertir de leur representation vers
 * un objet Java.
 * par exemple si on enregistre les converiseurs suivant:
 * <pre>
 * addConverter(new MatrixToXMLFormatConverter());
 * addConverter(new MatrixToSQLFormatConverter());
 * FormatConverterFactory.convert(Matrix.class, MatrixToXMLFormatConverter.TYPE,
 *    values, AppContext);
 * </pre>
 * Dans ce cas pour des raisons d'optimisation
 * <p/>
 * Created: 14 septembre 2005 00:19:51 CEST
 *
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 * @version $Id: FormatConverterFactory.java 1851 2010-05-11 11:24:21Z tchemit $
 * @since 1.3 replace the class {@code org.nuiton.util.FormatConverterFactory}.
 */
public class FormatConverterFactory { // FormatConverterFactory

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(FormatConverterFactory.class);

    static protected FormatConverterFactory instance;

    /** <Class, from, to -> FormatConverter> */
    protected MultiKeyMap converters = new MultiKeyMap();

    synchronized static public FormatConverterFactory getInstance() {
        if (instance == null) {
            instance = new FormatConverterFactory();
        }
        return instance;
    }

    /**
     * Permet d'enregitrer un converter pour permettre la convertion d'une
     * certain type Java d'une representation vers une autre.
     * par exemple le type String d'un objet Java vers une chaine XML
     *
     * @param clazz  la class de la representation Java de l'objet
     * @param format le format géré par le FormatConverter
     * @param c      le converter a enregistrer
     */
    public void addConverter(Class<?> clazz, Format format, FormatConverter<?> c) {
        converters.put(clazz, format, c);
    }

    /**
     * permet de recupere le converter pour la classe souhaitée.
     *
     * @param clazz            la classe de l'objet dont on souhaite le converter
     * @param format           qui doit être géré par le converter
     * @param defaultConverter si aucun converter trouvé, ce converter est
     *                         retourné
     * @return le converter souhaité ou defaultConverter
     */
    public FormatConverter<?> getConverter(Class<?> clazz, Format format,
                                           FormatConverter<?> defaultConverter) {
        FormatConverter<?> result = (FormatConverter<?>) converters.get(clazz, format);
        if (result == null) {
            result = defaultConverter;
        }
        return result;
    }

    /**
     * @param clazz
     * @param format
     * @return retourne null si aucun converter trouvé
     * @see #getConverter(Class, Format, FormatConverter)
     */
    public FormatConverter<?> getConverter(Class<?> clazz, Format format) {
        return getConverter(clazz, format, null);
    }

    /**
     * Permet de retrouver le meilleur converter disponible pour l'argument
     * clazz
     *
     * @param clazz            la classe de l'objet dont on souhaite le converter
     * @param format           qui doit être géré par le converter
     * @param defaultConverter si aucun converter trouvé, ce converter est
     *                         retourné
     * @return le converter souhaité ou defaultConverter
     */
    public FormatConverter<?> findConverter(Class<?> clazz, Format format,
                                            FormatConverter<?> defaultConverter) {
        FormatConverter<?> result = null;

        LinkedList<Class<?>> interfaces = new LinkedList<Class<?>>();
        // On recherche la le transformer le plus spécifique sur les Class
        Class<?> parent = clazz;
        while (result == null && parent != null) {
            Collections.addAll(interfaces, parent.getInterfaces());
            result = getConverter(parent, format);
            parent = parent.getSuperclass();
        }

        // Si on a pas encore trouve de transformer on recherche
        // un encodeur/decodeur pour les interfaces
        for (Iterator<Class<?>> i = interfaces.iterator(); result == null && i.hasNext();) {
            result = getConverter(i.next(), format);
        }

        if (result == null) {
            log.warn("Aucun converter trouvé pour le type: " + clazz);
            result = defaultConverter;
        }
        log.debug("converter " + result + " utilisé pour le type: " + clazz);

        return result;
    }

    /**
     * @param clazz
     * @param format
     * @return retourne null si aucun converter trouvé
     * @see #findConverter(Class, Format, FormatConverter)
     */
    public FormatConverter<?> findConverter(Class<?> clazz, Format format) {
        return findConverter(clazz, format, null);
    }

    public Object convert(Format format, FormatMap values, Object... args) {
        FormatConverter<?> c = findConverter(values.getType(), format);
        if (c == null) {
            throw new IllegalArgumentException("Aucun converter utilisable pour les arguments donnés class: " + values.getType().getName() + " format: " + format);
        }
        return c.convert(this, format, values, args);
    }

    public Object unconvert(Format format, FormatMap values, Object... args) {
        FormatConverter<?> c = findConverter(values.getType(), format);
        if (c == null) {
            throw new IllegalArgumentException("Aucun converter utilisable pour les arguments donnés");
        }
        return c.unconvert(this, format, values, args);
    }

    public Object convert(FormatConverter<?> defaultConverter,
                          Format format, FormatMap values, Object... args) {
        FormatConverter<?> c = findConverter(values.getType(), format, defaultConverter);
        return c.convert(this, format, values, args);
    }

    public Object unconvert(FormatConverter<?> defaultConverter,
                            Format format, FormatMap values, Object... args) {
        FormatConverter<?> c = findConverter(values.getType(), format, defaultConverter);
        return c.unconvert(this, format, values, args);
    }

} // FormatConverterFactory

