/*
 * #%L
 * Nuiton Utils
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.util;

import java.util.HashMap;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Cette classe permet de facilement trace le temps d'execution entre deux points
 *
 * usage
 * <pre>
 * TimeTrace timeTrace = new TimeTrace(5000, 1000);
 *
 *
 * long start = timeTrace.getTime();
 * ...
 * // do some work
 * ...
 * timeTrace.add(start, "do some work");
 *
 *
 * System.out.println ("time: " + timeTrace.getCallCount());
 * </pre>
 *
 * @see CallAnalyse
 * @author poussin
 * @version $Revision: 1980 $
 *
 * Last update: $Date: 2010-12-23 10:25:30 +0100 (jeu., 23 déc. 2010) $
 * by : $Author: sletellier $
 */
public class TimeTrace {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private Log log = LogFactory.getLog(TimeTrace.class);

    static public class CallStat {
        long callNumber = 0;
        long callTime = 0;

        @Override
        public String toString() {
            String callTimeString =
                    StringUtil.convertTime(callTime);
            String avgTimeString =
                    StringUtil.convertTime((callTime/ callNumber));
            return String.format("total call %s, total time %s, avg time %s",
                    callNumber, callTimeString, avgTimeString);
        }

    }

    /** time to trigger log time in info level (ns) (default: 1s) */
    protected long timeToLogInfo = 1000l * 1000000l;
    /** time to trigger log time in warn level (ns) (default: 3s) */
    protected long timeToLogWarn = 3000l * 1000000l;

    /** for each method of all proxies, keep number of call */
    protected Map<String, CallStat> callCount = new HashMap<String, CallStat>();

    public TimeTrace() {
    }

    /**
     *
     * @param timeToLogInfo time in milliseconde after that we log info
     * @param timeToLogWarn time in milliseconde after that we log warn
     */
    public TimeTrace(long timeToLogInfo, long timeToLogWarn) {
        setTimeToLogInfo(timeToLogInfo);
        setTimeToLogWarn(timeToLogWarn);
    }

    /**
     * 
     * @param timeToLogInfoMs time in milliseconde after that we log info
     */
    public void setTimeToLogInfo(long timeToLogInfoMs) {
        this.timeToLogInfo = timeToLogInfoMs * 1000000l; // convert ms -> ns
    }

     /**
     *
     * @param timeToWarnInfoMs time in milliseconde after that we log warn
     */
    public void setTimeToLogWarn(long timeToLogWarnMs) {
        this.timeToLogWarn = timeToLogWarnMs * 1000000l; // convert ms -> ns
    }

    public Map<String, CallStat> getCallCount() {
        return callCount;
    }

    /**
     * return time in format acceptable for {@link #add(timeToLogInfo, timeToLogInfo, null)}
     * @return
     */
    static public long getTime() {
        return System.nanoTime();
    }

    /**
     * add new trace, stop time is automaticaly computed
     *
     * @param startNs time returned by {@link #getTime()} method
     * @param methodName key name to store this time
     */
    public void add(long startNs, String methodName) {
        add(startNs, getTime(), methodName);
    }

    /**
     * add new trace
     *
     * @param startNs time returned by {@link #getTime()} method
     * @param stopNs time returned by {@link #getTime()} method
     * @param methodName key name to store this time
     */
    public void add(long startNs, long stopNs, String methodName) {
        long time = stopNs - startNs;

        // incremente le nombre d'appel pour cette methode
        CallStat calls = callCount.get(methodName);
        if (calls == null) {
            // is not thread safe, but if we lose one or two call, is not importante
            calls = new CallStat();
            callCount.put(methodName, calls);
        }
        calls.callNumber++;
        calls.callTime += time;

        // affiche le temps de l'appel si necessaire
        String timeString = StringUtil.convertTime(time);
        String msg = String.format("[%s] for method %s (%s)",
                timeString, methodName, calls);

        if (time > timeToLogWarn && log.isWarnEnabled()) {
            log.warn(msg);
        } else if (time > timeToLogInfo && log.isInfoEnabled()) {
            log.info(msg);
        } else if (log.isDebugEnabled()) {
            log.debug(msg);
        }

    }

}
