/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: ObjectUtil.java 1830 2010-04-15 14:29:20Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-1.3.1/src/main/java/org/nuiton/util/ObjectUtil.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

/* *
 * ObjectUtil.java
 *
 * Created: 2 nov. 2004
 *
 * @author Benjamin Poussin <poussin@codelutin.com>
 * @version $Revision: 1830 $
 *
 * Mise a jour: $Date: 2010-04-15 16:29:20 +0200 (jeu., 15 avril 2010) $
 * par : */

package org.nuiton.util;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import org.apache.commons.beanutils.MethodUtils;
import static org.nuiton.i18n.I18n._;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.ConvertUtils;

public class ObjectUtil { // ObjectUtil

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(ObjectUtil.class);

 protected static final Integer ZERO = 0;
 protected static final Character ZEROC = (char)0;
    protected static final Float ZEROF = 0f;
    protected static final Long ZEROL = 0l;
    protected static final Double ZEROD = 0.;
    protected static final Byte ZEROB = 0;

    /**
    * ObjectUtil constructor
    * private because of this class is a static class : nobody
    * can make an instance of this class
    */
    private ObjectUtil() {}

    /**
     * Create new object from string like org.nuiton.Toto(name=machine, int=10)
     * where machine and int is properties on org.nuiton.Toto object.
     * Conversion between 10 in string and 10 as integer as automaticaly done 
     * 
     * For String property you can use ex:
     * <li> name="my string with , in string"
     * <li> name='my string with , in string'
     * 
     * @param classnameAndProperties
     * @return the instanciated object
     * @throws ClassNotFoundException 
     * @throws IllegalAccessException 
     * @throws InstantiationException 
     * @throws NoSuchMethodException 
     * @throws InvocationTargetException 
     */
    public static Object create(String classnameAndProperties) throws ClassNotFoundException, InstantiationException, IllegalAccessException, InvocationTargetException, NoSuchMethodException {
        int p = classnameAndProperties.indexOf('(');
        int l = classnameAndProperties.lastIndexOf(')');
        String [] properties = null;
        String classname = null;
        if (p != -1) {
            String tmp = classnameAndProperties.substring(p + 1, l);
            properties = StringUtil.split(tmp, ",");
            classname = classnameAndProperties.substring(0, p);
        } else {
            classname = classnameAndProperties;
        }
        Class<?> clazz = Thread.currentThread().getContextClassLoader().loadClass(classname);
        Object o = clazz.newInstance();
        if (properties != null) {
            for (String prop : properties) {
                int e = prop.indexOf('=');
                String propName = prop.substring(0, e).trim();
                String propValue = prop.substring(e+1).trim();
                if (propValue.charAt(0) == '"' && propValue.charAt(propValue.length()-1) == '"') {
                    propValue = propValue.substring(1, propValue.length()-1);
                } else if (propValue.charAt(0) == '\'' && propValue.charAt(propValue.length()-1) == '\'') {
                    propValue = propValue.substring(1, propValue.length()-1);
                }
                BeanUtils.setProperty(o, propName, propValue);
            }
        }
        return o;
    }

    static protected Object convert(String v, Class<?> clazz) {
        Object t = ConvertUtils.convert(v, clazz);

        if (t != null &&
        !String.class.getName().equals(clazz.getName()) &&
                String.class.getName().equals(t.getClass().getName())) {
            throw new IllegalArgumentException(String.format(
                    "Can convert argument to correct type. %s can't be" +
                    " converted from String to %s conversion is done to %s",
                    v, clazz.getName(), t.getClass().getName()));
        }
        return t;
    }
    
    /**
     * Call method m with params as String. Each param is converted to required type for
     * method with beanutils converter
     * @param o object where method must be call
     * @param m method to call
     * @param params parameters for method call
     * @return returned method's value
     * @throws IllegalAccessException
     * @throws IllegalArgumentException
     * @throws InvocationTargetException
     * @throws InstantiationException
     */
    public static Object call(Object o, Method m, String ... params)
            throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, InstantiationException {
        Class<?>[] types = m.getParameterTypes();
        if (!m.isVarArgs() && params.length != types.length) {
            throw new IllegalArgumentException(String.format(
                    "Bad number params we have %1$s parameters and waiting %2$s.",
                    params.length, types.length));
        }
        
        int last = types.length;
        if (m.isVarArgs()) {
            // on traite le dernier differement
            last--;
        }
        
        Object[] parameters = new Object[types.length];
        for (int i=0; i<last; i++) {
            String v = params[i];
            Class<?> clazz = types[i];
            Object t = convert(v, clazz);
            parameters[i] = t;
        }
        
        if (m.isVarArgs()) {
            Class<?> clazz = types[last]; // get var args type
            clazz = clazz.getComponentType(); // get array component type
            List<Object> tmp = new ArrayList<Object>();
            for (int i=last; i<params.length; i++) {
                String v = params[i];
                Object t = convert(v, clazz);
                tmp.add(t);
            }
            parameters[last] = tmp.toArray((Object[])Array.newInstance(clazz, tmp.size()));
        }

        if (log.isDebugEnabled()) {
            log.debug(_("nuitonutil.debug.objectutil.invoke", m, Arrays.toString(parameters)));
        }
        Object result = m.invoke(o, parameters);
        return result;
    }
    
    /**
     * Get all methods with name given in argument without check parameters
     * @param clazz 
     * @param methodName method name to search
     * @param ignoreCase if true, ignore difference in method name case
     * @return list of detected methods
     */
    public static List<Method> getMethod(Class<?> clazz, String methodName, boolean ignoreCase) {
        List<Method> result = new ArrayList<Method>();
        
        Method[] methods = clazz.getMethods();
        for (Method m : methods) {
            if(ignoreCase && methodName.equalsIgnoreCase(m.getName()) ||
                    methodName.equals(m.getName())) {
                result.add(m);
            }
        }
        
        return result;
    }
    
    public static Object newInstance(String constructorWithParams) throws ClassNotFoundException {
        int p = constructorWithParams.indexOf('(');
        int l = constructorWithParams.lastIndexOf(')');
        String [] params = null;
        String classname = null;
        if (p != -1) {
            String tmp = constructorWithParams.substring(p + 1, l);
            params = StringUtil.split(tmp, ",");
            classname = constructorWithParams.substring(0, p);
        } else {
            classname = constructorWithParams;
        }
        Class<?> clazz = Thread.currentThread().getContextClassLoader().loadClass(classname);
        Object result = newInstance(clazz, params);
        return result;
    }
    /**
     * Create new instance of clazz, call constructor with params as String.
     * Each param is converted to required type for
     * constructor with beanutils converter, first constructor that permit
     * instanciation is used
     * 
     * @param <T> type to instanciate
     * @param clazz class to instanciate
     * @param params parameters for constructor call
     * @return new instance of clazz
     * @throws IllegalArgumentException 
     */
    public static <T> T newInstance(Class<T> clazz, String ... params)
            throws IllegalArgumentException {
        if (params == null) {
            params = new String[0];
        }
        List<Constructor<T>> constructors = getConstructor(clazz, params.length);
        
        for (Constructor<T> c : constructors) {
            try {
            Class<?>[] types = c.getParameterTypes();

                int last = types.length;
                if (c.isVarArgs()) {
                    // on traite le dernier differement
                    last--;
                }

                Object[] parameters = new Object[types.length];
                for (int i = 0; i < last; i++) {
                    String v = params[i];
                    Class<?> argClazz = types[i];
                    Object t = convert(v, argClazz);
                    parameters[i] = t;
                }

                if (c.isVarArgs()) {
                    Class<?> argClazz = types[last]; // get var args type
                    argClazz = argClazz.getComponentType(); // get array component type
                    List<Object> tmp = new ArrayList<Object>();
                    for (int i = last; i < params.length; i++) {
                        String v = params[i];
                        Object t = convert(v, argClazz);
                        tmp.add(t);
                    }
                    parameters[last] = tmp.toArray((Object[]) Array.newInstance(argClazz, tmp.size()));
                }

                if (log.isDebugEnabled()) {
                    log.debug(_("nuitonutil.debug.objectutil.create", clazz, Arrays.toString(parameters)));
                }
                T result = c.newInstance(parameters);
                
                return result;
            } catch(Exception eee) {
                // this constructors don't work, try next
                if (log.isDebugEnabled()) {
                    log.debug("Creation failed try with next constructor");
                }
            }
        }
        throw new IllegalArgumentException(_("nuitonutil.debug.objectutil.instantiate",
                clazz, Arrays.toString(params)));
    }
    
    /**
     * Get all constructors that support paramNumber as parameters numbers.
     * Varargs is supported
     * 
     * @param <T> le type de la classe a inspecter
     * @param clazz la classe sur lequel rechercher le constructeur
     * @param paramNumber le nombre de parametre souhaite pour le constructeur,
     * -1 indique que tous les constructeur sont souhaite.
     * @return list of constructors
     */
    @SuppressWarnings("unchecked")
    public static <T> List<Constructor<T>> getConstructor(Class<T> clazz, int paramNumber) {
        List<Constructor<T>> result = new ArrayList<Constructor<T>>();
        Constructor<T>[] constructors = (Constructor<T>[])clazz.getConstructors();
        for (Constructor<T> c : constructors) {
            if (paramNumber < 0 ||
                (c.isVarArgs() && c.getParameterTypes().length <= paramNumber - 1) ||
                    c.getParameterTypes().length == paramNumber) {
                result.add(c);
            }
        }
        
        return result;
    }
    
    /**
    * Method toObject
    *
    * @param o Object to transform
    * @return the same object
    */
    public static Object toObject(Object o){
        return o;
    }

    /**
    * Method toObject
    *
    * transform a char to a Character Object
    * @param c the char to transform
    * @return the Charactere object corresponding
    */
    public static Object toObject(char c){
        return new Character(c);
    }

    /**
    * Method toObject
    *
    * transform a byte to a Byte Object
    * @param b the byte to transform
    * @return the byte object corresponding
    */
    public static Object toObject(byte b){
        return new Byte(b);
    }

    /**
    * Method toObject
    *
    * transform a short to a Short object
    * @param s the short to transform
    * @return the Short object corresponding
    */
    public static Object toObject(short s){
        return new Short(s);
    }

    /**
    * Method toObject
    *
    * transform an int to an Integer object
    * @param i the int to transform
    * @return the Integer Object corresponding
    */
    public static Object toObject(int i){
        return new Integer(i);
    }

    /**
    * Method toObject
    *
    * transform a long to a Long object
    * @param l the long to transform
    * @return the Long Object corresponding
    */
    public static Object toObject(long l){
        return new Long(l);
    }

    /**
    * Method toObject
    *
    * transform a float to a Float Object
    * @param f the float to transform
    * @return the Float Object corresponding
    */
    public static Object toObject(float f){
        return new Float(f);
    }

    /**
    * Method toObject
    *
    * transform a double to a Double object
    * @param d the double to transform
    * @return the Double object corresponding
    */
    public static Object toObject(double d){
        return new Double(d);
    }

    /**
    * Method toObject
    *
    * transform a boolean to a Boolean object
    * @param b the boolean to transform
    * @return the Boolean object corresponding
    */
    public static Object toObject(boolean b){
        return b?Boolean.TRUE:Boolean.FALSE;
    }

    /**
     * Obtains the null value for the given type (works too with primitive
     * types).
     *
     * @param type the type to test
     * @return the {@code null} value or default value for primitive types
     * @since 1.1.5
     */
    public static Object getNullValue(Class<?> type) {
        if (type == null) {
            throw new NullPointerException("parameter 'type' can not be null");
        }
        if (type.isPrimitive()) {
            type = MethodUtils.getPrimitiveWrapper(type);
            if (Boolean.class.isAssignableFrom(type)) {
                return Boolean.FALSE;
            }
            if (Integer.class.isAssignableFrom(type)) {
                return ZERO;
            }
            if (Character.class.isAssignableFrom(type)) {
                return ZEROC;
            }
            if (Float.class.isAssignableFrom(type)) {
                return ZEROF;
            }
            if (Long.class.isAssignableFrom(type)) {
                return ZEROL;
            }
            if (Double.class.isAssignableFrom(type)) {
                return ZEROD;
            }
            if (Byte.class.isAssignableFrom(type)) {
                return ZEROB;
            }
        }
        return null;
    }

    /**
     * Tests if the given value is null according to default value for
     * primitive types if nedded.
     *
     * @param value the value to test
     * @return {@code true} if value is null or default value on a primitive
     * @since 1.1.5
     */
    public static boolean isNullValue(Object value) {
        if (value == null) {
            return true;
        }
        Class<?> type = value.getClass();

        //FIXME-TC20100212 : this case can not be, due to auto-boxing mecanism
        if (type.isPrimitive()) {
            type = MethodUtils.getPrimitiveWrapper(type);

            if (Boolean.class.isAssignableFrom(type)) {
                return Boolean.FALSE.equals(value);
            }
            if (Integer.class.isAssignableFrom(type)) {
                return ZERO.equals(value);
            }
            if (Character.class.isAssignableFrom(type)) {
                return ZEROC.equals(value);
            }
            if (Float.class.isAssignableFrom(type)) {
                return ZEROF.equals(value);
            }
            if (Long.class.isAssignableFrom(type)) {
                return ZEROL.equals(value);
            }
            if (Double.class.isAssignableFrom(type)) {
                return ZEROD.equals(value);
            }
            if (Byte.class.isAssignableFrom(type)) {
                return ZEROB.equals(value);
            }
        }
        return false;
    }

    public static boolean isNullValue(boolean value) {
        return Boolean.FALSE.equals(value);
    }
    
    public static boolean isNullValue(byte value) {
        return value == ZEROB;
    }

    public static boolean isNullValue(int value) {
        return value == ZEROB;
    }

    public static boolean isNullValue(char value) {
        return value == ZEROC;
    }

    public static boolean isNullValue(float value) {
        return value == ZEROF;
    }

    public static boolean isNullValue(double value) {
        return value == ZEROD;
    }

} // ObjectUtil

