/* *##% Nuiton utilities library
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */

/* *
 * I18n.java
 *
 * Created: 2 d?c. 2003
 *
 * @author Benjamin Poussin <poussin@codelutin.com>
 * Copyright Code Lutin
 * @version $Revision: 1762 $
 *
 * Mise a jour: $Date: 2010-02-09 18:00:40 +0100 (mar., 09 févr. 2010) $
 * par : $Author: echatellier $
 */
package org.nuiton.i18n;

import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.logging.Logger;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ConverterUtil;

/**
 * This class is a facility for internationalization. To use it in your soft,
 * you can either :
 * <ul>
 * <li> import the org.nuiton.i18n.I18n class,
 * <li> init the translation support with the init(String language) or
 * init(String language, String country), init(Localelocale) static methods in your main, ( eg:
 * I18n.init("fr","FR") )
 * <li> call the translate static method for each sentence, ( eg:
 * I18n._("hello you !") )
 * <li> create a resource file for each language following the naming
 * convention given in the java.util.ResourceBundle javadoc and translate all
 * the sentence.
 * </ul>
 *
 * @author poussin
 * @author chemit
 * created 2 decembre 2003
 */
public class I18n {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(I18n.class);
    public static final String ISO_8859_1_ENCONDING = "ISO-8859-1";
    public static final String UTF_8_ENCONDING = "UTF-8";
    public static final String DEFAULT_ENCODING = ISO_8859_1_ENCONDING;
    public static final Locale DEFAULT_LOCALE = Locale.UK;
    /** la classe responsable du chargement des ressources */
    static I18nLoader loader;
    /** Filtre a appliquer avant de retourner les chaines */
    protected static I18nFilter filter;
    /** Indique le chemin du fichier dans lequel ecrire les entrees non trouvees */
    protected static String recordFilePath;
    /**
     * some extra urls to given to bundle manager.
     * <p/>
     * Note: use this before call <code>init(...)</code> methods
     */
    static URL[] extraURL;
    /**
     * le nom d'un unique bundle à utiliser.
     * 
     * Si l'on positionne cette propriété, on n'utilisera pas la recherche
     * classique des bunldes à charger mais on se contentera de rechercher
     * le bundle (et ses entrées) à cet emplacement 
     *   <code>META-INF/uniqueBundleName-XXX.properties</code>.
     * 
     * Cela permet de chargement I18n en une seule fois et de ne scruter qu'un
     * seul jar (ou repertoire)...
     *
     * Un goal dans le plugin i18n (i18n:bundle) permet de construire le bundle
     * englobant toutes les autres traductions avec gestion des dépendances.
     *
     * @since 1.0.6
     */
    protected static String uniqueBundleName;

    /** Initialise la librairie avec encoding par defaut et locale par defaut */
    public static void init() {
        init(null);
    }

    /**
     * Initialize the library for given <code>locale</code with {@link #DEFAULT_ENCODING}.
     *
     * @param locale language to use
     */
    public static void init(Locale locale) {
        if (locale == null) {
            locale = newLocale(null, null);
        }
        getLoader().setLanguage(locale);
    }

    /**
     * Initialise la librairie
     *
     * @param language une chaine representant la langue à utiliser fr, en, ...
     * @param country  une chaine representant le pays à utiliser FR, GB, ...
     */
    public static void init(String language, String country) {
        init(newLocale(language, country));
    }

    /**
     * Retourne la chaine traduite si possible.
     *
     * @param message la chaine a traduire
     * @return la traduction si possible ou la chaine passee en parametre
     *         sinon.
     */
    public static String _(String message) {

        // if the key to translate is null, just return null
        if (message == null) {
            return null;
        }

        if (loader == null || loader.getLanguage() == null) {
            return applyFilter(message);
        }
        return applyFilter(loader.getLanguage().translate(message));
    }

    /**
     * Retourne la chaine traduite si possible.
     *
     * @param message message formate avec la meme syntaxe que {@link String#format}
     * @param args    les parametres pour le message.
     * @return la traduction si possible ou la chaine passee en parametre
     *         sinon.
     */
    public static String _(String message, Object... args) {

        // if the key to translate is null, just return null
        if (message == null) {
            return null;
        }

        String result = message;
        Language language = loader == null ? null : loader.getLanguage();
        if (language != null) {
            result = language.translate(message);
        }
        try {
            return applyFilter(String.format(result, args));
        } catch (Exception eee) {
            try {
                return applyFilter(String.format(message, args));
            } catch (Exception zzz) {
                log.warn(I18n._("nuitonutil.error.i18n.untranslated.message", message), zzz);
                return applyFilter(message);
            }
        }
    }

    /**
     * Retourne la chaine passée en argument.
     *
     * Utile surtout pour collecter les chaines et ne pas les traduires à leur
     * apparition.
     * 
     * Par exemple :
     * <pre>String key = "nuitonutils.key";
     * String result = _(key)</pre>
     * fonctionnera, mais la chaine n'aura pas été marquée comme devant être
     * internationalisé.
     * 
     * Tres utile par exemple, pour crée des objets non internationnalisé, et
     * devant être traduit seulement à leur lecture suivant la locale du
     * lecteur et non du créateur.
     * 
     * @param message message formate avec la meme syntaxe que {@link String#format(String, Object...)}
     * @param args les parametres pour le message.
     * @return le message passe en argument mais formatté avec les parametres
     */
    public static String n_(String message, Object... args) {
        try {
            return String.format(message, args);
        } catch (Exception eee) {
            log.warn(I18n._("nuitonutil.error.i18n.unformated.message", message, Arrays.toString(args)), eee);
            return message;
        }
    }

    /**
     * Retourne la chaine passé en argument.
     * 
     * Utile surtout pour collecter les chaines et ne pas les traduires à leur
     * apparition.
     * 
     * Par exemple :
     * <pre>String key = "nuitonutils.key";
     * String result = _(key)</pre>
     * fonctionnera, mais la chaine n'aura pas été marquée comme devant être
     * internationalisé.
     * 
     * Tres utile par exemple, pour crée des objets non internationnalisé, et
     * devant être traduit seulement à leur lecture suivant la locale du
     * lecteur et non du créateur.
     *
     * @param message la chaine à traduire
     * @return la chaine passée en argument.
     */
    public static String n_(String message) {
        return message;
    }

    public static String getRecordFilePath() {
        return recordFilePath;
    }

    /** @return the array of extra url to include in bundle search */
    public static URL[] getExtraURL() {
        return extraURL == null ? new URL[0] : extraURL;
    }

    /**
     *@return the unique bundle name to use for loading of system
     * 
     * @since 1.0.6
     */
    public static String getUniqueBundleName() {
        return uniqueBundleName;
    }

    /**
     * Change le filtre des chaines traduites
     *
     * @param filter l'objet filtre a utiliser
     */
    public static void setFilter(I18nFilter filter) {
        I18n.filter = filter;
    }

    public static void setRecordFilePath(String recordFilePath) {
        I18n.recordFilePath = recordFilePath;
    }

    /**
     * Change extra urls to use in bundle discovering
     * <p/>
     * Note: <b>This method will close the i18n system.</b>
     *
     * @param extraURL new extra urls to use
     */
    public static void setExtraURL(URL[] extraURL) {
        I18n.extraURL = extraURL;
        // must reset loader urls
        I18nLoader.urls = null;
        // and close system
        close();
    }

    /**
     * Change the unique bunlde name to use
     * <p/>
     * Note: <b>This method will close the i18n system.</b>
     *
     * @param uniqueBundleName the new unique bundle pattern to use
     *
     * @since 1.0.6
     */
    public static void setUniqueBundleName(String uniqueBundleName) {
        I18n.uniqueBundleName = uniqueBundleName;
        // must reset loader urls
        I18nLoader.urls = null;
        // and close system
        close();
    }

    /**
     * close i18n caches, says the loader if exists
     * <p/>
     * This method should be called to reset all caches (languages, bundles,...)
     */
    public static void close() {
        if (loader != null) {
            loader.close();
            loader = null;
        }
    }

    /**
     * Applique le filtre s'il y en a un
     *
     * @param message le message qui devrait etre retourne avant application du
     *                filtre.
     * @return le message filtre
     */
    protected static String applyFilter(String message) {
        if (getFilter() != null) {
            return getFilter().applyFilter(message);
        }
        return message;
    }

    protected static I18nFilter getFilter() {
        return filter;
    }

    /**
     * Get the i18n loader.
     * <p/>
     * If no loader found, then instanciate a new one.
     * 
     * @return the instanciated i18n loader
     */
    public static synchronized I18nLoader getLoader() {

        if (loader == null) {
            loader = new I18nLoader(DEFAULT_LOCALE, uniqueBundleName);
        }
        return loader;
    }

    /**
     * Parse a list of {@link Locale} seperated by comma.
     *
     * Example : fr_FR,en_GB
     *
     * @param str the string representation of locale separated by comma
     * @return list of available locales
     * @throws IllegalArgumentException ia a locale is not valid
     */
    public static Locale[] parseLocales(String str) throws IllegalArgumentException {
        List<Locale> result = new java.util.ArrayList<Locale>();
        String[] bundlesToUse = str.split(",");
        for (int i = 0, j = bundlesToUse.length; i < j; i++) {
            String s = bundlesToUse[i].trim();
            // on devrait verifier que le bundle existe
            try {
                Locale l = ConverterUtil.convert(Locale.class, s);
                result.add(l);
            } catch (Exception e) {
                throw new IllegalArgumentException("bundle " + s + " is not a valid locale,e");
            }
        }
        return result.toArray(new Locale[result.size()]);
    }

    public static Locale newLocale(String str) {
        if (str == null) {
            // get use locale
            return newLocale(null, null);
        }
        try {
            return ConverterUtil.convert(Locale.class, str);
        } catch (Exception e) {
            Logger.getLogger("org.nuiton.i18n.I18n").warning("could not load locale '" + str + " for reason : " + e.getMessage());
            // use default locale
            return DEFAULT_LOCALE;
        }
    }

    public static Locale newLocale(String language, String country) {
        if (language == null) {
            // get user locale
            language = System.getProperty("user.language", DEFAULT_LOCALE.getLanguage());
            country = System.getProperty("user.country", DEFAULT_LOCALE.getCountry());
        }
        return newLocale(language + (country == null ? "" : '_' + country));
    }
} //I18n
