package org.nuiton.util;

/**
 * *##% Chorem-Commons
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>. ##%*
 */

import java.text.DateFormatSymbols;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;

/**
 * Library for manipulating dates.
 *
 * @author fdesbois
 * @version $Revision: 87 $
 *
 * Last update: $Date: 2009-12-01 14:26:40 +0100 (Tue, 01 Dec 2009) $
 * by : $Author: fdesbois $
 */
public class DateUtils {

    public static final String DEFAULT_PATTERN = "dd/MM/yyyy";

    public static final String MONTH_PATTERN = "MM/yyyy";

    /**
     * Format a date using the pattern in argument. The pattern is the same using
     * for DateFormat object.
     * @param date the date to format
     * @param pattern the pattern to use
     * @return a String corresponding to the date formatted
     * @see java.text.DateFormat
     */
    public static String formatDate(Date date, String pattern) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern);
        return simpleDateFormat.format(date);
    }
    
    public static String formatDate(Date date, String pattern, Locale locale) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern, locale);
        return simpleDateFormat.format(date);
    }

    /**
     * Parse a date using the pattern in argument. The pattern is the same using
     * for DateFormat object.
     * @param date the String to parse
     * @param pattern the pattern to use
     * @return a Date corresponding to the String argument parsed
     * @throws ParseException for parsing errors
     * @see java.text.DateFormat
     */
    public static Date parseDate(String date, String pattern) throws ParseException {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern);
        Date result = simpleDateFormat.parse(date);
        return result;
    }

    /**
     * Create a new date from day, month and year (French version).
     * The month is the real month of the year and not the one which is stored in Calendar object.     *
     * @param dd value of the day 1-31
     * @param mm value of the month 1-12
     * @param yy value of the year 0-9999
     * @return a new date
     */
    public static Date createDate(int dd, int mm, int yy) {
        Calendar calendar = new GregorianCalendar(yy,mm-1,dd);
        return calendar.getTime();
    }

    /**
     * Create a new date after the current date (today) with modification on day, month and year.
     * You can use negative values on arguments to have a date before today.
     * @param ddStep nb days you want to increase from the current date
     * @param mmStep nb months you want to increase from the current date
     * @param yyStep nb years you want to increase from the current date
     * @return a new date from the current date increase by days, months and years.
     */
    public static Date createDateAfterToday(int ddStep, int mmStep, int yyStep) {
        Calendar calendar = new GregorianCalendar();
        calendar.setTime(new Date());
        calendar.add(Calendar.DAY_OF_MONTH, ddStep);
        calendar.add(Calendar.MONTH,mmStep);
        calendar.add(Calendar.YEAR, yyStep);        
        return calendar.getTime();
    }

    /**
     * Set the last day of month to the date in argument.
     * The value depends on the month of the date. (30 april, 28 february, ...)
     * @param date Date to modify
     * @return the date with day of month modified
     */
    public static Date setLastDayOfMonth(Date date) {
        Calendar calendar = new GregorianCalendar();
        calendar.setTime(date);
        int maximum = calendar.getActualMaximum(Calendar.DAY_OF_MONTH);
        calendar.set(Calendar.DAY_OF_MONTH, maximum);
        return calendar.getTime();
    }

    /**
     * Set the first day of month to the date in argument.
     * @param date Date to modify
     * @return the date with day of month modified
     */
    public static Date setFirstDayOfMonth(Date date) {
        Calendar calendar = new GregorianCalendar();
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        return calendar.getTime();
    }

    /**
     * Check if the first date in argument is included between the two other dates.
     * The argument myDate can be equals to beforeDate or afterDate to validate the includes.
     * @param myDate the date to test
     * @param beforeDate the first date of the period to test
     * @param afterDate the second date of the period to test
     * @return true if myDate is included between beforeDate and afterDate
     */
    public static boolean between(Date myDate, Date beforeDate, Date afterDate) {
        if (myDate == null) {
            return false;
        }
        boolean result = true;    
        result &= myDate.after(beforeDate) || myDate.equals(beforeDate);
        result &= afterDate == null || myDate.before(afterDate) || myDate.equals(afterDate);
        return result;
    }

    /**
     * Check if the current date is between the two dates in argument.
     * @param beforeDate the first date of the period
     * @param afterDate the second date of the period
     * @return true if the current date is included between the two dates, false either
     * @see #between(java.util.Date, java.util.Date, java.util.Date)
     */
    public static boolean currentPeriod(Date beforeDate, Date afterDate) {
        return between(new Date(), beforeDate, afterDate);
    }

    /**
     * Get the month value from a date (between 0 and 11).
     * @param date the date to extract month
     * @return the month value of the date
     */
    public static int getMonth(Date date) {
        Calendar calendar = new GregorianCalendar();
        calendar.setTime(date);
        return calendar.get(Calendar.MONTH);
    }

    /**
     * Do the difference between the two dates in argument. The result is a number
     * of days between the two dates.
     * Ex : 28/01/2009 and 08/02/2009 return 11.
     * @param beginDate first date
     * @param endDate second date
     * @return a number of days between beginDate and endDate
     */
    public static int getDifferenceInDays(Date beginDate, Date endDate) {
        long begin = beginDate.getTime();
        long end = endDate.getTime();
         // 86400000 = 24 * 60 * 60 * 1000
        return (int) Math.ceil(((end - begin) / 86400000));
    }

    /**
     * Get libelle of the month corresponding to the number given in argument.
     * @param monthNumber between 1-12
     * @param locale Locale for language support
     * @return a String corresponding to the libelle of the month
     */
    public static String getMonthLibelle(int monthNumber, Locale locale) {
        return new DateFormatSymbols(locale).getMonths()[monthNumber-1];
    }

    /**
     * Get libelle of the month corresponding to the number given in argument.
     * @param monthNumber between 1-12
     * @return a String corresponding to the libelle of the month
     */
    public static String getMonthLibelle(int monthNumber) {
        return getMonthLibelle(monthNumber, Locale.getDefault());
    }

}
