/* *##% Nuiton utilities library
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*/

package org.nuiton.log;

import java.awt.GraphicsEnvironment;
import javax.swing.JOptionPane;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ListenerSet;

/**
 * Classe permettant d'afficher facillement des messages a l'utilisateur.
 * Si une interface graphique est disponible alors des boites de dialogue
 * sont affichees. Il est aussi possible d'indique des composants a utiliser
 * pour afficher les messages. N'importe quel composant ayant une methode
 * setText convient
 *
 * @author poussin
 * @version $Revision: 1637 $
 *
 * Last update: $Date: 2009-06-21 04:53:31 +0200 (dim., 21 juin 2009) $
 * by : $Author: tchemit $
 */
public class UserLog {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(UserLog.class);

    public static enum Level {
        DEBUG, INFO, WARN, ERROR, FATAL
    }

    protected static ListenerSet<Object> listeners = new ListenerSet<Object>();
    protected static boolean graphicUI = true;

    public static void addListeners(ListenerSet<Object> l) {
        listeners.add(l);
    }

    public static void removeListeners(ListenerSet<Object> l) {
        listeners.remove(l);
    }

    /**
     * Enable/disable UI log display.
     * 
     * @param graphicUI <tt>true</tt> to enable
     */
    public static void setGraphicUI(boolean graphicUI) {
        UserLog.graphicUI = graphicUI;
    }

    /**
     * Return true if UI login is enabled and available.
     * 
     * @return graphicUI enable
     */
    public static boolean isGraphicUI() {
        return graphicUI && !GraphicsEnvironment.isHeadless();
    }

    /**
     * Appel la methode setText de tous les listeners avec le message en
     * parametre et si {@link #isGraphicUI} retourne vrai affiche une boite
     * de dialogue avec le message
     * 
     * @param level le niveau du message
     * @param msg le message
     * @param cause l'exception s'il y en a une
     */
    protected static void show(Level level, String msg, Throwable cause) {
        try {
            listeners.fire("setText", msg);
        } catch (Exception e) {
            log.warn("Can't call setText on listener", e);
        }

        if (isGraphicUI()) {
            String title = level.name();
            int type = JOptionPane.PLAIN_MESSAGE;
            switch (level) {
            case DEBUG:
                type = JOptionPane.PLAIN_MESSAGE;
                break;
            case INFO:
                type = JOptionPane.INFORMATION_MESSAGE;
                break;
            case WARN:
                type = JOptionPane.WARNING_MESSAGE;
                break;
            case ERROR:
            case FATAL:
                type = JOptionPane.ERROR_MESSAGE;
                break;
            }
            JOptionPane.showMessageDialog(null, msg, title, type);
        }
    }

    /**
     * Log message using commons-logging and display message box if UI is
     * available and non disabled.
     * 
     * @param msg message to display
     */
    public static void debug(String msg) {
        if (log.isDebugEnabled()) {
            log.debug(msg);
        }
        show(Level.DEBUG, msg, null);
    }

    /**
     * Log message using commons-logging and display message box if UI is
     * available and non disabled.
     * 
     * @param msg message to display
     * @param cause exception cause
     */
    public static void debug(String msg, Throwable cause) {
        if (log.isDebugEnabled()) {
            log.debug(msg, cause);
        }
        show(Level.DEBUG, msg, cause);
    }

    /**
     * Log message using commons-logging and display message box if UI is
     * available and non disabled.
     * 
     * @param msg message to display
     */
    public static void info(String msg) {
        log.info(msg);
        show(Level.INFO, msg, null);
    }

    /**
     * Log message using commons-logging and display message box if UI is
     * available and non disabled.
     * 
     * @param msg message to display
     * @param cause exception cause
     */
    public static void info(String msg, Throwable cause) {
        log.info(msg, cause);
        show(Level.INFO, msg, cause);
    }

    /**
     * Log message using commons-logging and display message box if UI is
     * available and non disabled.
     * 
     * @param msg message to display
     */
    public static void warn(String msg) {
        log.warn(msg);
        show(Level.WARN, msg, null);
    }

    /**
     * Log message using commons-logging and display message box if UI is
     * available and non disabled.
     * 
     * @param msg message to display
     * @param cause exception cause
     */
    public static void warn(String msg, Throwable cause) {
        log.warn(msg, cause);
        show(Level.WARN, msg, cause);
    }

    /**
     * Log message using commons-logging and display message box if UI is
     * available and non disabled.
     * 
     * @param msg message to display
     */
    public static void error(String msg) {
        log.error(msg);
        show(Level.ERROR, msg, null);
    }

    /**
     * Log message using commons-logging and display message box if UI is
     * available and non disabled.
     * 
     * @param msg message to display
     * @param cause exception cause
     */
    public static void error(String msg, Throwable cause) {
        log.error(msg, cause);
        show(Level.ERROR, msg, cause);
    }

    /**
     * Log message using commons-logging and display message box if UI is
     * available and non disabled.
     * 
     * @param msg message to display
     */
    public static void fatal(String msg) {
        log.fatal(msg);
        show(Level.FATAL, msg, null);
    }

    /**
     * Log message using commons-logging and display message box if UI is
     * available and non disabled.
     * 
     * @param msg message to display
     * @param cause exception cause
     */
    public static void fatal(String msg, Throwable cause) {
        log.fatal(msg, cause);
        show(Level.FATAL, msg, cause);
    }
}
