/* *##%
 * Copyright (C) 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.nuiton.util.war;

import java.awt.AWTException;
import java.awt.Desktop;
import java.awt.Image;
import java.awt.MenuItem;
import java.awt.PopupMenu;
import java.awt.SystemTray;
import java.awt.TrayIcon;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.IOException;
import java.net.JarURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Map;

import javax.swing.ImageIcon;

import winstone.Launcher;

/**
 * War main class launcher (winstone based).
 * 
 * Based on winstone micro container.
 * 
 * To use it :
 *  java -jar app-xxx.war
 * 
 * @author chatellier
 * @version $Revision: 1716 $
 * @since 1.1.2
 * 
 * Last update : $Date: 2009-12-17 16:34:04 +0100 (jeu., 17 déc. 2009) $
 * By : $Author$
 */
public class WinstoneLauncher implements ActionListener, MouseListener {

    /** Winstone server instance. */
    protected Launcher winstone;

    /** Server URI. */
    protected URI serverUri;

    /**
     * Main method (used by war in manifest).
     * 
     * @param args args
     * @throws IOException
     */
    public static void main(String[] args) throws IOException {
        WinstoneLauncher launcher = new WinstoneLauncher();
        launcher.startServer(args);
        launcher.installSystemTray();
        launcher.openBrowser();
    }

    /**
     * Launch servlet container.
     * 
     * @param args args
     * @throws IOException
     */
    protected void startServer(String[] args) throws IOException {
        System.out.println("Starting server embedded mode...");

        String fqnLauncherFile = WinstoneLauncher.class.getName().replaceAll("\\.", "/") + ".class";
        System.out.println("Search for launcher class : " + fqnLauncherFile);

        URL classFile = WinstoneLauncher.class.getClassLoader().getResource(fqnLauncherFile);
        System.out.println(" - using classFile : " + classFile);

        // strange following line seams also work for jpnl launch
        File me = new File(((JarURLConnection) classFile.openConnection()).getJarFile().getName());
        System.out.println(" - using warfile file : " + me);

        // hashArgs map, initialized with command line args
        Map<String, String> hashArgs = Launcher.getArgsFromCommandLine(args);

        hashArgs.put("warfile", me.getAbsolutePath()); // or any other command line args, eg port

        System.out.println(" - using args : " + hashArgs);

        Launcher.initLogger(hashArgs);
        winstone = new Launcher(hashArgs); // spawns threads, so your application doesn't block

        // open browser
        int port = 8080;
        String configPort = hashArgs.get("httpPort");
        if (configPort != null && !configPort.isEmpty()) {
            try {
                port = Integer.parseInt(configPort);
            } catch (NumberFormatException e) {
                e.printStackTrace();
            }
        }

        // build server uri
        try {
            serverUri = new URI("http://localhost:" + port);
        } catch (URISyntaxException e) {
            e.printStackTrace();
        }
    }

    /**
     * Shutdown server.
     */
    protected void stopServer() {
        if (winstone != null) {
            winstone.shutdown();
            System.exit(0);
        }
    }

    /**
     * Install system tray to stop server.
     */
    protected void installSystemTray() {
        if (SystemTray.isSupported()) {
            // build menu
            PopupMenu menu = new PopupMenu();
            MenuItem browserItem = new MenuItem("Start browser");
            browserItem.addActionListener(this);
            browserItem.setActionCommand("browser");
            menu.add(browserItem);

            MenuItem stopItem = new MenuItem("Stop server");
            stopItem.addActionListener(this);
            stopItem.setActionCommand("stop");
            menu.add(stopItem);

            // build tray icon
            URL imageURL = WinstoneLauncher.class.getResource("/favicon.png");
            if (imageURL == null) {
                imageURL = WinstoneLauncher.class.getResource("/favicon.jpg");
            }
            if (imageURL == null) {
                System.out.println("No favicon.{png|jpg} found, skip systray installation");
            } else {
                Image image = new ImageIcon(imageURL).getImage();
                TrayIcon icon = new TrayIcon(image, "Server", menu);
                icon.setImageAutoSize(true);
                icon.addMouseListener(this);

                // System tray
                SystemTray systemTray = SystemTray.getSystemTray();
                try {
                    systemTray.add(icon);
                } catch (AWTException ex) {
                    throw new RuntimeException("Can't install tray icon", ex);
                }
            }
        }
    }

    /*
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    @Override
    public void actionPerformed(ActionEvent e) {
        if ("browser".equalsIgnoreCase(e.getActionCommand())) {
            openBrowser();
        } else if ("stop".equalsIgnoreCase(e.getActionCommand())) {
            stopServer();
        }
    }

    /**
     * Open browser.
     * 
     * @throws IOException 
     */
    protected void openBrowser() {
        if (Desktop.isDesktopSupported() && serverUri != null) {
            Desktop desktop = Desktop.getDesktop();
            if (desktop.isSupported(Desktop.Action.BROWSE)) {
                System.out.println("Opening browser at " + serverUri);
                try {
                    desktop.browse(serverUri);
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /*
     * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2) {
            openBrowser();
        }
    }

    /*
     * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
     */
    @Override
    public void mousePressed(MouseEvent e) {

    }

    /*
     * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseReleased(MouseEvent e) {

    }

    /*
     * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseEntered(MouseEvent e) {

    }

    /*
     * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseExited(MouseEvent e) {

    }
}
