/* *##% Nuiton utilities library
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */
package org.nuiton.util;

import org.apache.commons.beanutils.ConstructorUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Properties;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;

/**
 * <h1>To do</h1>
 * <p/>
 * <ul>
 * <li> ajout d'annotations sur les methodes
 * pour precisser plus de chose pour les options (pattern, min/max, alias,
 * description, ...)
 * <li> trouver un moyen de document les options et actions pour automatiquement
 * generer l'aide en ligne. Pour eviter de devoir maintenir une methode
 * dans lequel est ecrit l'aide en plus des options.
 * <li> prise en compte du flag {@link #useOnlyAliases}
 * <li> vu qu'en java on ne peut pas pointer une methode mais seulement une classe
 * il y a un bout des actions qui sont des chaines (nom de la methode). Il faudrait
 * faire un plugin maven qui check que l'action existe bien durant la compilation.
 * Il est simple de le faire a l'execution mais c trop tard :(
 * </ul>
 * <p/>
 * <h1>Usage</h1>
 * <li> create subclass of ApplicationConfig, where in constructor you call
 * addAliases, setConfigFileName, setDefaultActionPackage, setDefaultActionClass, setDefaultActionMethod
 * to have properly value.
 * <p/>
 * <li> conf = new MonAppConfig();
 * <li> conf.parse(args);
 * <li> here you can used conf.getOption(key);
 * <li> conf.doAction(0);
 * <li> ...
 * <li> conf.doAction(n);
 * <p/>
 * <h1>Lecture des fichiers de configuration</h1>
 * <p/>
 * <p/>
 * La lecture des fichiers de configuration se fait durant l'appel de la methode
 * {@link #parse} en utilisant la valeur de {@link #getConfigFileName} pour
 * trouver les fichiers (voir Les options de configuration pour l'ordre de
 * chargement des fichiers)
 * <p/>
 * <h1>La sauvegarde</h1>
 * <p/>
 * <p/>
 * La sauvegarde des options se fait via une des trois methodes disponible
 * <p/>
 * <ul>
 * <li> {@link #save(File, boolean,String[])} sauve les données dans le fichier demandé
 * <li> {@link #saveForSystem} sauvegarde les donnees dans /etc
 * <li> {@link #saveForUser} sauvegarde les donnees dans $HOME
 * </ul>
 * <p/>
 * Lors de l'utilisation de la methode {@link #saveForSystem(String[])} ou
 * {@link #saveForUser(String[])} seul les options lu dans un fichier ou modifier par
 * programmation ({@link #setOption} seront sauvegardees. Par exemple les
 * options passees sur la ligne de commande ne seront pas sauvees.
 * <p/>
 * <h1>Les options de configuration</h1>
 * <p/>
 * <p/>
 * Cette classe permet de lire les fichiers de configuration, utiliser les
 * variable d'environnement et de parser la ligne de commande. L'ordre de prise
 * en compte des informations trouvées est la suivante (le premier le plus
 * important).
 * <p/>
 * <ul>
 * <li>options ajoutees par programmation: {@link #setOption}(key, value)</li>
 * <li>ligne de commande</li>
 * <li>variable d'environnement de la JVM: java -Dkey=value</li>
 * <li>variable d'environnement; export key=value</li>
 * <li>fichier de configuration du repertoire courant: $user.dir/filename</li>
 * <li>fichier de configuration du repertoire home de l'utilisateur: $user.home/.filename</li>
 * <li>fichier de configuration du repertoire /etc: /etc/filename</li>
 * <li>fichier de configuration trouve dans le classpath: $CLASSPATH/filename</li>
 * <li>options ajoutees par programmation: {@link #defaults}.put(key, value)</li>
 * </ul>
 * <p/>
 * <p/>
 * Les options sur la ligne de commande sont de la forme:
 * <pre>
 * --option key value
 * --monOption key value1 value2
 * </pre>
 * <p/>
 * <ul>
 * <li>--option key value: est la syntaxe par defaut
 * <li>--monOption key value1 value2: est la syntaxe si vous avez ajouter une
 * methode setMonOption(key, value1, value2) sur votre classe de configuration
 * qui herite de {@link ApplicationConfig}. Dans ce cas vous pouvez mettre les
 * arguments que vous souhaitez du moment qu'ils soient convertibles de la
 * representation String vers le type que vous avez mis.
 * </ul>
 * <p/>
 * <h1>Les actions</h1>
 * <p/>
 * Les actions ne peuvent etre que sur la ligne de commande. Ils sont de la
 * forme:
 * <pre>
 * --le.package.LaClass#laMethode arg1 arg2 arg3 ... argN
 * </pre>
 * <p/>
 * <p/>
 * Une action est donc defini par le chemin complet vers la methode qui traitera
 * l'action. Cette methode peut-etre une methode static ou non. Si la methode
 * n'est pas static lors de l'instanciation de l'objet on essaie de passer en
 * parametre du constructeur la classe de configuration utilisee pour permettre
 * a l'action d'avoir a sa disposition les options de configuration. Si aucun
 * constructeur avec comme seul parametre une classe heritant de
 * {@link ApplicationConfig} n'existe alors le constructeur par defaut est
 * utilise (il doit etre accessible). Toutes methodes d'actions faisant
 * parties d'un meme objet utiliseront la meme instance de cette objet lors
 * de leur execution.
 * <p/>
 * <p/>
 * Si la methode utilise les arguments variants alors tous les arguments
 * jusqu'au prochain -- ou la fin de la ligne de commande sont utilises. Sinon
 * Le nombre exact d'argument necessaire a la methode sont utilises.
 * <p/>
 * <p/>
 * Les arguments sont automatiquement converti dans le bon type reclame par la
 * methode.
 * <p/>
 * <p/>
 * Si l'on veut des arguments optionnels le seul moyen actuellement est
 * d'utiliser une methode avec des arguments variants
 * <p/>
 * <p> Les actions ne sont pas execute mais seulement parsees. Pour les executer
 * il faut utiilser la methode {@link #doAction} qui prend en argument un numero
 * de 'step'. Par defaut toutes les actions sont de niveau 0 et sont executee
 * dans l'ordre d'apparition sur la ligne de commande. Si l'on souhaite
 * distinguer les actions il est possible d'utiliser l'annotation
 * {@link ApplicationConfig.Action.Step} sur la methode qui fera l'action en
 * precisant une autre valeur que 0.
 * <p/>
 * <pre>
 * doAction(0);
 * ... do something ...
 * doAction(1);
 * </pre>
 * <p/>
 * <p/>
 * dans cette exemple on fait un traitement entre l'execution des actions
 * de niveau 0 et les actions de niveau 1.
 * <p/>
 * <h1>Les arguments non parses</h1>
 * <p/>
 * <p/>
 * Tout ce qui n'est pas option ou action est considere comme non parse et peut
 * etre recupere par la methode {@link #getUnparsed}. Si l'on souhaite forcer
 * la fin du parsing de la ligne de commande il est possible de mettre --.
 * Par exemple:
 * <pre>
 * monProg "mon arg" --option k1 v1 -- --option k2 v2 -- autre
 * </pre>
 * <p/>
 * <p/>
 * Dans cet exemple seule la premiere option sera considere comme une option.
 * On retrouvera dans unparsed: "mon arg", "--option", "k2", "v2", "--", "autre"
 * <p/>
 * <h1>Les alias</h1>
 * <p/>
 * On voit qu'aussi bien pour les actions que pour les options, le nom de la
 * methode doit etre utilise. Pour eviter ceci il est possible de definir
 * des alias ce qui permet de creer des options courtes par exemple. Pour cela,
 * on utilise la methode {@link #addAlias}.
 * <p/>
 * <pre>
 * addAlias("-v", "--option", "verbose", "true");
 * addAlias("-o", "--option", "outputfile");
 * addAlias("-i", "--mon.package.MaClass#MaMethode", "import");
 * </pre>
 * <p/>
 * <p/>
 * En faite avant le parsing de la ligne de commande tous les alias trouves sont
 * automatiquement remplacer par leur correspondance. Il est donc possible
 * d'utiliser ce mecanisme pour autre chose par exemple:
 * <p/>
 * <pre>
 * addAlias("cl", "Code Lutin");
 * addAlias("bp", "Benjamin POUSSIN);
 * </pre>
 * <p/>
 * <p/>
 * Dans le premier exemple on simplifie une option de flags l'option -v n'attend
 * donc plus d'argument. Dans le second exemple on simplifie une option qui
 * attend encore un argment de type File. Enfin dans le troisieme exemple
 * on simplifie la syntaxe d'une action et on force le premier argument de
 * l'action a etre "import".
 * <p/>
 * <h1>Conversion de type</h1>
 * Pour la conversion de type nous utilisons common-beans. Les types supporte
 * sont:
 * <p/>
 * <ul>
 * <li> les primitif (byte, short, int, long, float, double, char, boolean)
 * <li> String
 * <li> File
 * <li> URL
 * <li> Class
 * <li> SqlDate
 * <li> SqlTime
 * <li> SqlTimestamps
 * <li> les tableaux d'un type primitif ou String. Chaque element doit etre
 * separe par une virgule
 * </ul>
 * <p/>
 * Pour suporter d'autre type, il vous suffit d'enregistrer de nouveau
 * converter dans commons-beans
 *
 * @author poussin
 * @version $Revision$
 * @since 0.30
 *        <p/>
 *        Last update $Date$
 *        by $Author$
 */
public class ApplicationConfig {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(ApplicationConfig.class);
    /**
     * used to know what is separator between la class et la method sur la
     * ligne de commande
     */
    static final private String CLASS_METHOD_SEPARATOR = "#";
    static final public String CONFIG_FILE_NAME = "config.file";
    protected boolean useOnlyAliases = false;
    protected Map<String, List<String>> aliases = new HashMap<String, List<String>>();
    /** file /etc/[filename] */
    String systemPath = File.separator + "etc" + File.separator;
    /** file $user.home/.[filename] */
    String userPath = getUserHome() + File.separator + ".";
    /** vrai si on est en train de parser les options de la ligne de commande */
    protected boolean inParseOptionPhase = false;
    protected Properties defaults = new Properties();
    protected Properties classpath = new Properties(defaults);
    protected Properties etcfile = new Properties(classpath);
    protected Properties homefile = new Properties(etcfile);
    protected Properties curfile = new Properties(homefile);
    protected Properties env = new Properties(curfile);
    protected Properties jvm = new Properties(env);
    protected Properties line = new Properties(jvm);
    protected Properties options = new Properties(line);
    protected Map<String, CacheItem<?>> cacheOption = new HashMap<String, CacheItem<?>>();
    protected Map<Class<?>, Object> cacheAction = new HashMap<Class<?>, Object>();
    /** contient apres l'appel de parse, la liste des arguments non utilises */
    protected List<String> unparsed = new ArrayList<String>();
    protected Map<Integer, List<Action>> actions = new HashMap<Integer, List<Action>>();
    /** suport of config modification */
    protected PropertyChangeSupport pcs = new PropertyChangeSupport(this);

    /**
     * Le contrat de marquage des options, on utilise cette interface pour
     * caracteriser une option de configuration.
     *
     * @since 1.0.0-rc-9
     */
    static public interface OptionDef {

        /**
         * @return la clef identifiant l'option
         */
        String getKey();

        /**
         * @return le type de l'option
         */
        Class<?> getType();

        /**
         * @return la clef i18n de description de l'option
         */
        String getDescription();

        /**
         * @return la valeur par defaut de l'option sous forme de chaine de
         * caracteres
         */
        String getDefaultValue();

        /**
         * @return <code>true</code> si l'option ne peut etre sauvegardee sur
         * disque (utile par exemple pour les mots de passe, ...)
         */
        boolean isTransient();

        /**
         * @return <code>true</code> si l'option n'est pas modifiable (utilise
         * par exemple pour la version de l'application, ...)
         */
        boolean isFinal();
    }

    static public class Action {

        @Retention(RetentionPolicy.RUNTIME)
        @Target(ElementType.METHOD)
        static public @interface Step {

            int value() default 0;
        }
        protected int step;
        protected Object o;
        protected Method m;
        protected String[] params;

        public Action(int step, Object o, Method m, String... params) {
            this.step = step;
            this.o = o;
            this.m = m;
            this.params = params;
        }

        public void doAction() throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, InstantiationException {
            ObjectUtil.call(o, m, params);
        }
    }

    /**
     * Item used for cacheOption
     *
     * @param <T>
     */
    static protected class CacheItem<T> {

        /** typed option value */
        public T item;
        /** hash of string representation */
        public int hash;

        public CacheItem(T item, int hash) {
            this.item = item;
            this.hash = hash;
        }
    }

    public ApplicationConfig() {
        setConfigFileName(this.getClass().getSimpleName());
        // init extra-converters
        ConverterUtil.initConverters();
    }

    static public String getUserHome() {
        String result = System.getProperty("user.home");
        return result;
    }

    public String getUsername() {
        String result = getOption("user.name");
        return result;
    }

    /**
     * Used to put default configuration option in config option. Those options
     * are used as fallback value.
     *
     * @param key   default property key
     * @param value default property value
     */
    protected void setDefaultOption(String key, String value) {
        defaults.setProperty(key, value);
    }

    /**
     * Save configuration, in specified file
     *
     * @param file     file where config will be writen
     * @param forceAll if true save all config option
     *                 (with defaults, classpath, env, command line)
     * @param excludeKeys optional list of keys to exclude from
     * @throws IOException if IO pb
     */
    public void save(File file, boolean forceAll, String... excludeKeys) throws IOException {
        Properties prop = new Properties();
        if (forceAll) {
            prop.putAll(defaults);
            prop.putAll(classpath);
        }
        prop.putAll(etcfile);
        prop.putAll(homefile);
        prop.putAll(curfile);
        if (forceAll) {
            prop.putAll(jvm);
            prop.putAll(env);
            prop.putAll(line);
        }
        prop.putAll(options);

        for (String excludeKey : excludeKeys) {
            prop.remove(excludeKey);
        }
        Writer writer = new FileWriter(file);
        prop.store(writer, "Last saved " + new java.util.Date());
    }

    /**
     * Save configuration, in system directory (/etc/) using the
     * {@link #getConfigFileName}. Default, env and commande line note saved.
     *
     * @param excludeKeys optional list of keys to exclude from
     */
    public void saveForSystem(String... excludeKeys) {
        File file = new File(systemPath + getConfigFileName());
        try {
            save(file, false, excludeKeys);
        } catch (IOException eee) {
            if (log.isWarnEnabled()) {
                log.warn(_("lutinutil.error.applicationconfig.save", file), eee);
            }
        }
    }

    /**
     * Save configuration, in user home directory using the
     * {@link #getConfigFileName}. Default, env and commande line note saved
     *
     * @param excludeKeys optional list of keys to exclude from
     */
    public void saveForUser(String... excludeKeys) {
        File file = new File(userPath + getConfigFileName());
        try {
            save(file, false, excludeKeys);
        } catch (IOException eee) {
            if (log.isWarnEnabled()) {
                log.warn(_("lutinutil.error.applicationconfig.save", file), eee);
            }
        }
    }

    /**
     * Return list of unparsed command line argument
     *
     * @return list of unparsed arguments
     */
    public List<String> getUnparsed() {
        return unparsed;
    }

    /**
     * Add action to list of action to do
     *
     * @param action action to add, can be null.
     */
    public void addAction(Action action) {
        if (action != null) {
            Integer step = action.step;
            List<Action> list = actions.get(step);
            if (list == null) {
                list = new LinkedList<ApplicationConfig.Action>();
                actions.put(step, list);
            }
            list.add(action);
        }
    }

    public void doAction(int step) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, InstantiationException {
        List<Action> list = actions.get(step);
        if (list != null) {
            for (Action a : list) {
                a.doAction();
            }
        }
    }

    public void setUseOnlyAliases(boolean useOnlyAliases) {
        this.useOnlyAliases = useOnlyAliases;
    }

    public boolean isUseOnlyAliases() {
        return useOnlyAliases;
    }

    /**
     * All argument in aliases as key is substitued by target
     *
     * @param alias  alias string as '-v'
     * @param target substitution as '--option verbose true'
     */
    public void addAlias(String alias, String... target) {
        aliases.put(alias, Arrays.asList(target));
    }

    /**
     * Add alias for action. This method put just -- front the actionMethod and
     * call {@link #addAlias(String, String[])}
     *
     * @param alias        the alias to add for the given method action
     * @param actionMethod must be fully qualified method path: package.Class.method
     */
    public void addActionAlias(String alias, String actionMethod) {
        addAlias(alias, "--" + actionMethod);
    }

    /**
     * Set name of file where options are read (in /etc, $HOME, $CURDIR)
     * This set used {@link #setDefaultOption(String, String)}
     *
     * @param name file name
     */
    public void setConfigFileName(String name) {
        // put in defaults, this permit user to overwrite it on commande line
        setDefaultOption(CONFIG_FILE_NAME, name);
    }

    public String getConfigFileName() {
        String result = options.getProperty(CONFIG_FILE_NAME);
        return result;
    }

    /**
     * Set option value
     *
     * @param key   property key
     * @param value property value
     */
    public void setOption(String key, String value) {
        if (inParseOptionPhase) {
            line.setProperty(key, value);
        } else {
            options.setProperty(key, value);
        }
    }

    /**
     * get option value as string
     *
     * @param key the option's key
     * @return String representation value
     */
    public String getOption(String key) {
        String value = options.getProperty(key);
        return value;
    }

    /**
     * Permet de recuperer l'ensemble des options commencant par une certaine
     * chaine
     * 
     * @param prefix debut de cle a recuperer
     * @return la liste des options filtrées 
     */
    public Properties getOptionStartsWith(String prefix) {
        Properties result = new Properties();

        for (String key : options.stringPropertyNames()) {
            if (key.startsWith(prefix)) {
                result.setProperty(key, options.getProperty(key));
            }
        }

        return result;
    }

    /**
     * get option value from a option definition
     *
     * @param key the definition of the option
     * @return the value for the given option 
     */
    public Object getOption(OptionDef key) {
        Object result = getOption(key.getType(), key.getKey());
        return result;
    }

    /**
     * get option value as typed value
     *
     * @param <T> type of the object wanted as return type
     * @param clazz type of object wanted as return type
     * @param key   the option's key
     * @return typed value
     */
    @SuppressWarnings("unchecked")
    public <T> T getOption(Class<T> clazz, String key) {
        T result = null;
        String cacheKey = key + "-" + clazz.getName();

        String value = options.getProperty(key);
        int hash = 0;
        if (value != null) {
            hash = value.hashCode();
        }
        CacheItem<T> cacheItem = (CacheItem<T>) cacheOption.get(cacheKey);
        // compute value if value don't exist in cacheOption or
        // if it's modified since last computation
        if (cacheItem == null || cacheItem.hash != hash) {
            // prefer use our convertert method (auto-register more converters)
            result = ConverterUtil.convert(clazz, value);
//            result = (T) ConvertUtils.convert(value, clazz);
            cacheItem = new CacheItem<T>(result, hash);
            cacheOption.put(cacheKey, cacheItem);
        } else {
            result = cacheItem.item;
        }

        return result;
    }

    /**
     * get option value as typed value
     *
     * @param key the option's key
     * @return typed value
     */
    public File getOptionAsFile(String key) {
        File result = getOption(File.class, key);
        result = result.getAbsoluteFile();
        return result;
    }

    /**
     * get option value as typed value
     *
     * @param key the option's key
     * @return typed value
     */
    public URL getOptionAsURL(String key) {
        URL result = getOption(URL.class, key);
        return result;
    }

    /**
     * get option value as typed value
     *
     * @param key the option's key
     * @return typed value
     */
    public Class<?> getOptionAsClass(String key) {
        Class<?> result = getOption(Class.class, key);
        return result;
    }

    /**
     * get option value as typed value
     *
     * @param key the option's key
     * @return typed value
     */
    public Date getOptionAsDate(String key) {
        Date result = getOption(Date.class, key);
        return result;
    }

    /**
     * get option value as typed value
     *
     * @param key the option's key
     * @return typed value
     */
    public Time getOptionAsTime(String key) {
        Time result = getOption(Time.class, key);
        return result;
    }

    /**
     * get option value as typed value
     *
     * @param key the option's key
     * @return typed value
     */
    public Timestamp getOptionAsTimestamp(String key) {
        Timestamp result = getOption(Timestamp.class, key);
        return result;
    }

    /**
     * get option value as typed value
     *
     * @param key the option's key
     * @return typed value
     */
    public int getOptionAsInt(String key) {
        Integer result = getOption(Integer.class, key);
        return result;
    }

    /**
     * get option value as typed value
     *
     * @param key the option's key
     * @return typed value
     */
    public double getOptionAsDouble(String key) {
        Double result = getOption(Double.class, key);
        return result;
    }

    /**
     * get option value as typed value
     *
     * @param key the option's key
     * @return typed value
     */
    public boolean getOptionAsBoolean(String key) {
        Boolean result = getOption(Boolean.class, key);
        return result;
    }

    /**
     * Get all set method on this object or super object
     *
     * @return map with method name without set and in lower case as key, and
     *         method as value
     */
    protected Map<String, Method> getMethods() {
        // looking for all methods set on ApplicationConfig
        Method[] allMethods = this.getClass().getMethods();
        Map<String, Method> methods = new HashMap<String, Method>();
        for (Method m : allMethods) {
            String methodName = m.getName();
            if (methodName.startsWith("set")) {
                methodName = methodName.substring(3).toLowerCase();
                methods.put(methodName, m);
            }
        }
        return methods;
    }

    /**
     * Take required argument for method in args. Argument used is removed from
     * args. If method has varArgs, we take all argument to next '--'
     *
     * @param m    the method to call
     * @param args iterator with many argument (equals or more than necessary
     * @return the arguments found for the given method
     */
    protected String[] getParams(Method m, ListIterator<String> args) {
        List<String> result = new ArrayList<String>();
        if (m.isVarArgs()) {
            while (args.hasNext()) {
                String p = args.next();
                if (p.startsWith("--")) {
                    // stop search
                    args.previous();
                    break;
                } else {
                    result.add(p);
                    args.remove();
                }
            }
        } else {
            int paramLenght = m.getParameterTypes().length;
            for (int i = 0; i < paramLenght; i++) {
                String p = args.next();
                args.remove(); // remove this arg because is used now
                result.add(p);
            }
        }
        return result.toArray(new String[result.size()]);
    }

    /**
     * Create action from string, string must be [package.][class][#][method]
     * if package, class or method missing, default is used
     *
     * @param name name of the action
     * @param args
     * @return the created action
     * @throws ArgumentsParserException
     * @throws IllegalAccessException
     * @throws IllegalArgumentException
     * @throws InstantiationException
     * @throws java.lang.reflect.InvocationTargetException
     */
    protected Action createAction(String name, ListIterator<String> args) throws ArgumentsParserException, InstantiationException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {
        Action result = null;

        Class<?> clazz;
        Method method = null;
        String className;
        String methodName;

        // looking for method name
        int sep = name.lastIndexOf(CLASS_METHOD_SEPARATOR);
        if (sep == -1) {
            throw new IllegalArgumentException(String.format(
                    "Can't find action method in %s", name));
        } else {
            className = name.substring(0, sep);
            methodName = name.substring(sep + 1);
        }

        // looking for class name
        try {
            clazz = Class.forName(className);
        } catch (ClassNotFoundException eee) {
            throw new IllegalArgumentException(String.format(
                    "Can't find action class %s", className));
        }

        List<Method> methods = ObjectUtil.getMethod(clazz, methodName, true);
        if (methods.size() > 0) {
            if (methods.size() > 1) {
                log.warn(String.format(
                        "More than one method found, used the first: %s", methods));
            }
            method = methods.get(0);
        }

        if (method != null) {
            args.remove(); // remove option from command line, because is used now

            // creation de l'object sur lequel on fera l'appel
            Object o = cacheAction.get(clazz);
            if (o == null && !Modifier.isStatic(method.getModifiers())) {
                try {
                    o = ConstructorUtils.invokeConstructor(clazz, this);
                } catch (NoSuchMethodException eee) {
                    log.debug(String.format(
                            "Use default constructor, because no constructor with Config parameter on class %s",
                            clazz.getName()));
                    o = clazz.newInstance();
                }
                cacheAction.put(clazz, o);
            }

            // recherche du step de l'action
            int step = 0;
            Action.Step annotation = method.getAnnotation(Action.Step.class);
            if (annotation != null) {
                step = annotation.value();
            }

            String[] params = getParams(method, args);
            result = new Action(step, o, method, params);
        }

        return result;
    }

    /**
     * Parse option and call set necessary method, read jvm, env variable,
     * Load configuration file and prepare Action.
     *
     * @param args argument as main(String[] args)
     * @throws ArgumentsParserException
     *
     */
    public void parse(String[] args) throws ArgumentsParserException {
        try {
            Map<String, Method> methods = getMethods();

            List<String> arguments = new ArrayList<String>(args.length);
            for (String arg : args) {
                if (aliases.containsKey(arg)) {
                    arguments.addAll(aliases.get(arg));
                } else {
                    arguments.add(arg);
                }
            }

            inParseOptionPhase = true;
            // first parse option
            for (ListIterator<String> i = arguments.listIterator(); i.hasNext();) {
                String arg = i.next();
                if (arg.equals("--")) {
                    // stop parsing
                    break;
                }
                if (arg.startsWith("--")) {
                    String optionName = arg.substring(2);
                    if (methods.containsKey(optionName)) {
                        i.remove(); // remove this arg because is used now
                        Method m = methods.get(optionName);
                        String[] params = getParams(m, i);
                        log.debug(String.format("Set option '%s' with method '%s %s'", optionName, m, Arrays.toString(params)));
                        ObjectUtil.call(this, m, params);
                    }
                }
            }
            inParseOptionPhase = false;

            //
            // second load options from all sources
            //
            // JVM
            jvm.putAll(System.getProperties());
            // ENV
            env.putAll(System.getenv());

            // classpath
            String filename = getConfigFileName();
            URL inClasspath = ClassLoader.getSystemClassLoader().getResource(filename);
            if (inClasspath == null) {
                inClasspath = ApplicationConfig.class.getResource(filename);
            }
            if (inClasspath != null) {
                log.info("Chargement du fichier de config: " + inClasspath);
                classpath.load(inClasspath.openStream());
            }

            File etcConfig = new File(systemPath + filename);
            if (etcConfig.exists()) {
                log.info("Chargement du fichier de config: " + etcConfig);
                etcfile.load(etcConfig.toURI().toURL().openStream());
            }

            File homeConfig = new File(userPath + filename);
            if (homeConfig.exists()) {
                log.info("Chargement du fichier de config: " + homeConfig);
                homefile.load(homeConfig.toURI().toURL().openStream());
            }

            // file $CURDIR/filename
            File config = new File(filename);
            if (config.exists()) {
                log.info("Chargement du fichier de config: " + config);
                curfile.load(config.toURI().toURL().openStream());
            }

            //
            // third parse action and do action
            //
            for (ListIterator<String> i = arguments.listIterator(); i.hasNext();) {
                String arg = i.next();
                if (arg.equals("--")) {
                    // stop parsing
                    break;
                }
                if (arg.startsWith("--")) {
                    String optionName = arg.substring(2);
                    Action action = createAction(optionName, i);
                    addAction(action);
                }
            }

            //
            // not used args added to unparsed
            //
            arguments.remove("--");
            unparsed.addAll(arguments);

        } catch (Exception eee) {
            eee.printStackTrace();
            throw new ArgumentsParserException("Can't parse argument", eee);
        }
    }

    /**
     * For debugging
     */
    public void printConfig() {
        System.out.println("-------------------Value-------------------------");
        System.out.println("defaults " + defaults);
        System.out.println("classpath " + classpath);
        System.out.println("etcfile " + etcfile);
        System.out.println("homefile " + homefile);
        System.out.println("curfile " + curfile);
        System.out.println("env " + env);
        System.out.println("jvm " + jvm);
        System.out.println("line " + line);
        System.out.println("options " + options);
        System.out.println("-------------------------------------------------");
    }

    protected void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
        pcs.firePropertyChange(propertyName, oldValue, newValue);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(propertyName, listener);
    }

    public synchronized boolean hasListeners(String propertyName) {
        return pcs.hasListeners(propertyName);
    }

    public synchronized PropertyChangeListener[] getPropertyChangeListeners(String propertyName) {
        return pcs.getPropertyChangeListeners(propertyName);
    }

    public synchronized PropertyChangeListener[] getPropertyChangeListeners() {
        return pcs.getPropertyChangeListeners();
    }
}
