/* *##% Nuiton utilities library
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */

/* *
 * FileUtil.java
 *
 * Created: 22 nov. 2004
 *
 * @author Benjamin Poussin <poussin@codelutin.com>
 * @version $Revision: 1643 $
 *
 * Mise a jour: $Date: 2009-06-24 17:03:57 +0200 (mer., 24 juin 2009) $
 * par : $Author: echatellier $
 */

package org.nuiton.util;

import org.apache.commons.logging.LogFactory;

import javax.swing.JFileChooser;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

public class FileUtil { // FileUtil

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private org.apache.commons.logging.Log log = LogFactory.getLog(FileUtil.class);

    /** Encoding par defaut utilisé si non spécifié */
    static public String ENCODING = "ISO-8859-1";
    static protected File currentDirectory = new File(".");

    static public void setCurrentDirectory(File dir) {
        currentDirectory = dir;
    }

    static public File getCurrentDirectory() {
        return currentDirectory;
    }

    static public class PatternChooserFilter extends javax.swing.filechooser.FileFilter {
        protected String pattern = null;
        protected String description = null;

        public PatternChooserFilter(String pattern, String description) {
            this.pattern = pattern;
            this.description = description;
        }

        @Override
        public boolean accept(File f) {
            return f.isDirectory() || f.getAbsolutePath().matches(pattern);
        }

        @Override
        public String getDescription() {
            return description;
        }

    }


    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param patternOrDescriptionFilters les filtres a utiliser, les chaines doivent etre données
     *                                    par deux, le pattern du filtre + la description du filtre
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     * @see #getFile(javax.swing.filechooser.FileFilter[])
     */
    static public File getFile(String... patternOrDescriptionFilters) {        
        File result;
        result = getFile(null, patternOrDescriptionFilters);
        return result;
    }

    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param filters les filtres a ajouter
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     */
    static public File getFile(javax.swing.filechooser.FileFilter... filters) {
        File result = getFile(null, filters);
        return result;
    }

     /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
      * @param parent le component parent du dialog
      * @param patternOrDescriptionFilters les filtres a utiliser, les chaines doivent etre données
     *                                    par deux, le pattern du filtre + la description du filtre
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     * @see #getFile(javax.swing.filechooser.FileFilter[])
     */
    static public File getFile(java.awt.Component parent, String... patternOrDescriptionFilters) {        
        File result;
        result = getFile("Ok", "Ok", parent, patternOrDescriptionFilters);
        return result;
    }

    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param title le titre de la boite de dialogue
     * @param approvalText le label du boutton d'acceptation
     * @param parent le component parent du dialog
     * @param patternOrDescriptionFilters les filtres a utiliser, les chaines doivent etre données
     *                                    par deux, le pattern du filtre + la description du filtre
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     * @see #getFile(javax.swing.filechooser.FileFilter[])
     */
    static public File getFile(String title, String approvalText,java.awt.Component parent, String... patternOrDescriptionFilters) {

        if (patternOrDescriptionFilters.length % 2 != 0) {
            throw new IllegalArgumentException("Arguments must be (pattern, description) couple");
        }
        javax.swing.filechooser.FileFilter[] filters = new javax.swing.filechooser.FileFilter[patternOrDescriptionFilters.length / 2];
        for (int i = 0; i < filters.length; i++) {
            String pattern = patternOrDescriptionFilters[i * 2];
            String description = patternOrDescriptionFilters[i * 2 + 1];
            filters[i] = new PatternChooserFilter(pattern, description);
        }
        File result;
        result = getFile(title, approvalText, parent, filters);
        return result;
    }

    
    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param parent le component parent du dialog
     * @param filters les filtres a ajouter
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     */
    static public File getFile(java.awt.Component parent, javax.swing.filechooser.FileFilter... filters) {
        File result = getFile("Ok", "Ok", parent, filters);
        return result;
    }

    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param title le titre de la boite de dialogue
     * @param approvalText le label du boutton d'acceptation
     * @param parent le component parent du dialog
     * @param filters les filtres a ajouter
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     */
    static public File getFile(String title, String approvalText, java.awt.Component parent, javax.swing.filechooser.FileFilter... filters) {
        try {
            JFileChooser chooser = new JFileChooser(currentDirectory);

            chooser.setDialogType(JFileChooser.CUSTOM_DIALOG);
            if (filters.length > 0) {
                if (filters.length == 1) {
                    chooser.setFileFilter(filters[0]);
                } else {
                    for (javax.swing.filechooser.FileFilter filter : filters) {
                        chooser.addChoosableFileFilter(filter);
                    }
                }
            }
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            chooser.setDialogTitle(title);
            int returnVal = chooser.showDialog(parent, approvalText);
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File theFile = chooser.getSelectedFile();
                if (theFile != null) {
                    currentDirectory = theFile;
                    return theFile.getAbsoluteFile();
                }
            }
        }
        catch (Exception eee) {
            log.warn("Erreur:", eee);
        }
        return null;
    }

    /**
     * @return le nom du repertoire entre dans la boite de dialogue.
     *         Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne
     *         null.
     */
    static public String getDirectory() {
        return getDirectory(null,"Ok", "Ok");
    }
    
    /**
     * @param title        le nom de la boite de dialogue
     * @param approvalText le texte de l'action d'acceptation du répertoire dans le file chooser
     * @return le nom du repertoire entre dans la boite de dialogue.
     *         Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne
     *         null.
     */
    static public String getDirectory(String title, String approvalText) {
        String result = getDirectory(null, title, approvalText);
        return result;
    }

    /**
     * @param parent le component parent du dialog
     * @param title        le nom de la boite de dialogue
     * @param approvalText le texte de l'action d'acceptation du répertoire dans le file chooser
     * @return le nom du repertoire entre dans la boite de dialogue.
     *         Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne
     *         null.
     */
    static public String getDirectory(java.awt.Component parent, String title, String approvalText) {
        try {
            JFileChooser chooser = new JFileChooser(currentDirectory);
            chooser.setDialogType(JFileChooser.CUSTOM_DIALOG);
            chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
            chooser.setDialogTitle(title);
            int returnVal = chooser.showDialog(parent, approvalText);
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File theFile = chooser.getSelectedFile();
                if (theFile != null) {
                    currentDirectory = theFile;
                    if (theFile.isDirectory()) {
                        return theFile.getAbsolutePath();
                    }
                }
            } else {
                return null;
            }
        } catch (Exception eee) {
            log.warn("Erreur:", eee);
        }
        return null;
    }

    /**
     * Permet de convertir un fichier en un tableau de byte
     *
     * @param file le fichier source à convertire
     * @return le contenu du fichier sous la forme d'un tableau de bytes.
     * @throws IOException if any io pb
     */
    static public byte[] fileToByte(File file) throws IOException {
        InputStream in = new BufferedInputStream(new FileInputStream(file));
        ByteArrayOutputStream result = new ByteArrayOutputStream();
        BufferedOutputStream tmp = new BufferedOutputStream(result);
        for (int b = in.read(); b != -1; b = in.read()) {
            tmp.write(b);
        }
        in.close();
        tmp.close();

        return result.toByteArray();
    }

    /**
     * Permet de recopier un stream dans un fichier
     *
     * @param src the incoming stream to grab
     * @param dst the dst file
     * @return the file filled by incoming input stream
     * @throws IOException if any io pb
     * @throws NullPointerException if src or dst parameter is null
     */
    static public File inputStreamToFile(InputStream src, File dst) throws IOException, NullPointerException {
        if (src == null) {
            throw new NullPointerException("parameter 'src' can not be null");
        }
        if (dst == null) {
            throw new NullPointerException("parameter 'dst' can not be null");
        }

        ByteArrayOutputStream result = new ByteArrayOutputStream();
        BufferedOutputStream tmp = new BufferedOutputStream(result);
        for (int b = src.read(); b != -1; b = src.read()) {
            tmp.write(b);
        }
        src.close();
        tmp.close();
        byteToFile(result.toByteArray(), dst);
        return dst;
    }

    /**
     * Permet de convertir des bytes en fichier, le fichier sera automatiquement
     * supprimé a la fin de la JVM.
     *
     * @param bytes the array of bytes to copy in dstination file
     * @return le fichier temporaire contenant les bytes
     * @throws IOException if any io pb
     */
    static public File byteToFile(byte[] bytes) throws IOException {
        File file = File.createTempFile("FileUtil-byteToFile", ".tmp");
        byteToFile(bytes, file);
        return file;
    }

    /**
     * Permet de convertir des bytes en fichier
     *
     * @param bytes the array of bytes to put in the given destination file
     * @param file  le fichier dans lequel il faut ecrire les bytes
     * @return le fichier passé en parametre
     * @throws IOException if any io pb
     */
    static public File byteToFile(byte[] bytes, File file) throws IOException {
        OutputStream out = new BufferedOutputStream(new FileOutputStream(file));
        out.write(bytes);
        out.close();
        return file;
    }

    /**
     * Retourne un Reader utilisant l'encoding par defaut {@link #ENCODING}.
     *
     * @param file the given reader
     * @return the reader on the given file
     * @throws IOException if any io pb
     */
    static public BufferedReader getReader(File file) throws IOException {
        return getReader(file, ENCODING);
    }

    /**
     * Retourne un reader utilisant l'encoding choisi et placé dans un
     * BufferedReader
     *
     * @param file     the given file
     * @param encoding (ISO-8859-1, UTF-8, ...)
     * @return the buffered reader in the given encoding
     * @throws IOException if any io pb
     */
    static public BufferedReader getReader(File file, String encoding) throws IOException {
        FileInputStream inf = new FileInputStream(file);
        InputStreamReader in = new InputStreamReader(inf, encoding);
        BufferedReader result = new BufferedReader(in);
        return result;
    }

    /**
     * Retourne un Writer utilisant l'encoding par defaut {@link #ENCODING}.
     *
     * @param file the given file
     * @return the writer on the given file
     * @throws IOException if any io pb
     */
    static public BufferedWriter getWriter(File file) throws IOException {
        return getWriter(file, ENCODING);
    }

    /**
     * Retourne un writer utilisant l'encoding choisi et placé dans un
     * BufferedWriter
     *
     * @param file     the given file
     * @param encoding (ISO-8859-1, UTF-8, ...)
     * @return the buffered writer on the given file with given encoding
     * @throws IOException if any io pb
     */
    static public BufferedWriter getWriter(File file, String encoding) throws IOException {
        FileOutputStream outf = new FileOutputStream(file);
        OutputStreamWriter out = new OutputStreamWriter(outf, encoding);
        BufferedWriter result = new BufferedWriter(out);
        return result;
    }


    /**
     * Permet de creer un nouveu repertoire temporaire, l'effacement du
     * répertoire est a la charge de l'appelant
     *
     * @param prefix le prefix du fichier
     * @param suffix le suffix du fichier
     * @param tmpdir le répertoire temporaire ou il faut creer le repertoire
     *               si null on utilise java.io.tmpdir
     * @return le fichier pointant sur le nouveau repertoire
     * @throws java.io.IOException if any io pb
     */
    static public File createTempDirectory(String prefix, String suffix, File tmpdir) throws IOException {
        if (tmpdir == null) {
            tmpdir = new File(System.getProperty("java.io.tmpdir"));
        }
        File result = new File(tmpdir, prefix + System.currentTimeMillis() + suffix);
        while (result.exists()) {
            result = new File(tmpdir, prefix + System.currentTimeMillis() + suffix);
        }
        if (!result.mkdirs()) {
            throw new IOException("Can't create temporary directory: " + result);
        }
        return result;
    }

    /**
     * Permet de creer un nouveu repertoire temporaire, l'effacement du
     * répertoire est a la charge de l'appelant
     *
     * @param prefix le prefix du repertoire a creer
     * @param suffix le suffix du repertoire a creer.
     * @return the temprary created file
     * @throws java.io.IOException if any io pb
     */
    static public File createTempDirectory(String prefix, String suffix) throws IOException {
        return createTempDirectory(prefix, suffix, null);
    }

    /**
     * Regarde si le fichier f1 est plus recent que le fichier f2
     *
     * @param f1 the first file
     * @param f2 the second file
     * @return vrai si f1 est plus recent que f2
     */
    static public boolean isNewer(File f1, File f2) {
        boolean result = f1.lastModified() > f2.lastModified();
        return result;
    }

    /**
     * Permet de lire un fichier et de retourner sont contenu sous forme d'une
     * chaine de carateres
     *
     * @param file le fichier a lire
     * @return le contenu du fichier
     * @throws IOException if any io pb
     */
    static public String readAsString(File file) throws IOException {
        StringBuffer result = new StringBuffer();
        char[] cbuf = new char[2000];
        BufferedReader in = getReader(file);
        int nb = in.read(cbuf);
        while (nb != -1) {
            result.append(cbuf, 0, nb);
            nb = in.read(cbuf);
        }
        in.close();
        return result.toString();
    }

    /**
     * Permet de sauver une chaine directement dans un fichier
     *
     * Use default enconding : {@link #ENCODING}.
     * 
     * @param file    Le fichier dans lequel il faut ecrire la chaine
     * @param content Le texte a ecrire dans le fichier
     * @throws IOException if any pb while writing
     */
    static public void writeString(File file, String content) throws IOException {
        //fixme on doit tester le retour de la méthode, car il se peut que le répertoire
        // ne puisse être crée.
        file.getParentFile().mkdirs();
        BufferedWriter out = getWriter(file);
        out.write(content);
        out.close();
    }

    /**
     * Permet de sauver une chaine directement dans un fichier
     *
     * @param file     Le fichier dans lequel il faut ecrire la chaine
     * @param content  Le texte a ecrire dans le fichier
     * @param encoding encoding to use
     * @throws IOException if any pb while writing
     */
    static public void writeString(File file, String content, String encoding) throws IOException {
        //fixme on doit tester le retour de la méthode, car il se peut que le répertoire
        // ne puisse être crée.
        file.getParentFile().mkdirs();
        BufferedWriter out = getWriter(file, encoding);
        out.write(content);
        out.close();
    }

    /**
     * Permet de donner une representation fichier pour une chaine de caractere.
     * Le fichier sera automatiquement effacé à la fin de la JVM.
     *
     * @param content le contenu du fichier temporaire
     * @return le fichier qui contient content
     * @throws IOException if any io pb
     */
    static public File getTempFile(String content) throws IOException {
        return getTempFile(content, "");
    }

    /**
     * Permet de donner une representation fichier pour une chaine de caractere.
     * Le fichier sera automatiquement effacé à la fin de la JVM.
     *
     * @param content    le contenu du fichier temporaire
     * @param fileSuffix l'extension du fichier créé
     * @return le fichier qui contient content
     * @throws IOException if any io pb
     */
    static public File getTempFile(String content, String fileSuffix) throws IOException {
        File result = File.createTempFile("tmp-" + FileUtil.class.getName(), fileSuffix);
        result.deleteOnExit();
        writeString(result, content);
        return result;
    }

    /**
     * Equivalent de la methode basename unix.
     * basename("/tmp/toto.xml", ".xml") -> "toto"
     *
     * @param file     le fichier dont on souhaite le nom sans le chemin
     * @param suffixes si present represente le suffixe a eliminer du fichier
     *                 s'il est trouvé
     * @return le nom du fichier sans le suffixe si trouvé.
     */
    static public String basename(File file, String... suffixes) {
        String result = file.getName();
        for (String suffixe : suffixes) {
            if (result.endsWith(suffixe)) {
                result = result.substring(0, result.length() - suffixe.length());
                break;
            }
        }
        return result;
    }

    /**
     * Permet de récupérer l'extension d'un fichier
     *
     * @param file     le fichier dont on souhaite l'extension
     * @param extchars la liste des caracteres pouvant former l'extension
     *                 dans l'ordre de preference. Si vide on utilise ".".
     * @return l'extension ou la chaine vide si le fichier n'a pas d'extension
     *         l'extension ne contient pas le chaine de delimitation
     */
    static public String extension(File file, String... extchars) {
        String result = "";
        String name = file.getName();

        if (extchars.length == 0) {
            extchars = new String[]{"."};
        }
        for (String extchar : extchars) {
            int pos = name.lastIndexOf(extchar);
            if (pos != -1) {
                result = name.substring(pos + extchar.length());
                break;
            }
        }
        return result;
    }

    static public interface FileAction {
        public boolean doAction(File f);
    }

    /**
     * Retourne tous les sous répertoires du répertoire passé en argument.
     *
     * @param directory un répertoire
     * @return une liste d'objet {@link File} de répertoires et ceci
     *         recursivement à partir de directory, si directory
     *         n'est pas un répertoire la liste est vide.
     */
    public static List<File> getSubDirectories(File directory) {
        class DirectoryFilter implements FileFilter {
            @Override
            public boolean accept(File f) {
                return f.isDirectory();
            }
        }
        return getFilteredElements(directory, new DirectoryFilter(), true);
    }

    /**
     * Retourne tous les fichiers du répertoire passé en argument.
     *
     * @param directory un répertoire
     * @return une liste d'objet {@link File} des fichiers et ceci
     *         recursivement à partir de directory, si directory n'est pas un
     *         répertoire la liste est vide
     */
    public static List<File> getFiles(File directory) {
        class NormalFileFilter implements FileFilter {
            @Override
            public boolean accept(File f) {
                return f.isFile();
            }
        }
        return getFilteredElements(directory, new NormalFileFilter(), true);
    }

    /**
     * Retourne les fichiers d'un répertoire qui s'attisfont un certain pattern.
     * La recherche est faite récursivement dans les sous répertoires
     *
     * @param directory   le répertoire à partir duquel il faut faire la recherche
     * @param pattern     le pattern que doit respecter le fichier pour être dans la
     *                    liste résultante
     * @param recursively flag pour indiquer si on doit descendre dans les sous répertoires
     * @return une liste d'objet {@link File} qui ont s'attisfait le
     *         pattern.
     */
    public static List<File> find(File directory, final String pattern, boolean recursively) {
        final String root = directory.getAbsolutePath();
        final int rootLength = root.length();

        return getFilteredElements(directory, new FileFilter() {
            @Override
            public boolean accept(File f) {
                String longFilename = f.getAbsolutePath();
                // + 1 to remove the first / or \
                String filename = longFilename.substring(rootLength + 1);
                return filename.matches(pattern);
            }
        }, recursively);
    }

    /**
     * Retourne la liste de toutes les fichiers ou répertoire qui s'attisfont
     * le filtre
     *
     * @param directory   repertoire à partir duquel il faut faire la recherche
     * @param ff          le filtre à appliquer pour savoir si le fichier parcouru doit
     *                    être conservé dans les résultats, ou null pour tous les fichiers
     * @param recursively un flag pour indiquer si on doit descendre dans les répertoires
     * @return une liste d'objet {@link File}, qui s'attisfont le filtre
     */
    public static List<File> getFilteredElements(File directory, FileFilter ff, boolean recursively) {
        ArrayList<File> result = new ArrayList<File>();
        LinkedList<File> todo = new LinkedList<File>();
        if (directory.isDirectory()) {
            todo.addAll(Arrays.asList(directory.listFiles()));
        }
        while (todo.size() > 0) {
            File file = todo.removeFirst();
            if (recursively && file.isDirectory()) {
                File[] childs = file.listFiles();
                if (childs != null) { // null if we don't have access to directory
                    todo.addAll(Arrays.asList(childs));
                }
            }
            if (ff == null || ff.accept(file)) {
                result.add(file);
            }
        }
        return result;
    }

    /**
     * Supprime recursivement tout le contenu d'un répertoire.
     *
     * @param directory le chemin du répertoire à supprimer
     * @return vrai si tout se passe bien, false si la suppression d'un élement
     *         se passe mal
     */
    public static boolean deleteRecursively(String directory) {
        return deleteRecursively(new File(directory));
    }

    /**
     * Supprime recursivement tout le contenu d'un répertoire.
     *
     * @param directory le répertoire à supprimer
     * @return vrai si tout se passe bien, false si la suppression d'un élement
     *         se passe mal
     */
    public static boolean deleteRecursively(File directory) {
        return walkBefore(directory, new FileAction() {
            @Override
            public boolean doAction(File f) {
                return f.delete();
            }
        });
    }

    /**
     * Permet de faire une action avant le parcours des fichiers, c-a-d que
     * l'on fera l'action sur les fichiers contenu dans un répertoire
     * après l'action sur le répertoire lui même.
     *
     * @param f          le fichier ou répertoire à partir duquel il faut commencer
     * @param fileAction l'action à effectuer sur chaque fichier
     * @return le résultat des fileAction executé sur les fichiers, chaque
     *         résultat de FileAction sont assemblé par un ET logique pour donner
     *         le résultat final
     */
    public static boolean walkAfter(File f, FileAction fileAction) {
        boolean result = fileAction.doAction(f);
        if (f.isDirectory()) {
            File list[] = f.listFiles();
            for (File aList : list) {
                result = result && walkAfter(aList, fileAction);
            }
        }
        return result;
    }

    /**
     * Permet de faire une action apès le parcours des fichiers, c-a-d que
     * l'on fera l'action sur les fichiers contenu dans un répertoire
     * avant l'action sur le répertoire lui même.
     *
     * @param f          le fichier ou répertoire à partir duquel il faut commencer
     * @param fileAction l'action à effectuer sur chaque fichier
     * @return le résultat des fileAction executé sur les fichiers, chaque
     *         résultat de FileAction sont assemblé par un ET logique pour donner
     *         le résultat final
     */
    public static boolean walkBefore(File f, FileAction fileAction) {
        boolean result = true;
        if (f.isDirectory()) {
            File list[] = f.listFiles();
            for (File aList : list) {
                result = result && walkBefore(aList, fileAction);
            }
        }
        return result && fileAction.doAction(f);
    }

    /**
     * Permet de copier le fichier source vers le fichier cible.
     *
     * @param source le fichier source
     * @param target le fichier cible
     * @throws IOException Erreur de copie
     */
    public static void copy(File source, File target) throws IOException {
        //fixme on doit tester le retour de la méthode, car il se peut que le répertoire
        // ne puisse être copié.
        target.getParentFile().mkdirs();
        FileChannel sourceChannel = new FileInputStream(source).getChannel();
        FileChannel targetChannel = new FileOutputStream(target).getChannel();
        sourceChannel.transferTo(0, sourceChannel.size(), targetChannel);
        // or
        //  targetChannel.transferFrom(sourceChannel, 0, sourceChannel.size());
        sourceChannel.close();
        targetChannel.close();
    }

    /**
     * Permet de copier le fichier source vers le fichier cible.
     *
     * @param source le fichier source
     * @param target le fichier cible
     * @throws IOException Erreur de copie
     */
    public static void copy(String source, String target) throws IOException {
        copy(new File(source), new File(target));
    }

    /**
     * Copie recursivement le repertoire source dans le repertoire destination
     * <p/>
     * copyRecursively("/truc/titi", "/var/tmp") donnera le repertoire
     * "/var/tmp/titi"
     *
     * @param srcDir          le répertoire source à copier
     * @param destDir         le répertoire destination où copier
     * @param includePatterns les patterns que doivent resperter les
     *                        fichiers/repertoires pour etre copié. Si vide alors tout est copié
     * @throws IOException if any io pb
     */
    static public void copyRecursively(File srcDir, File destDir, String... includePatterns) throws IOException {
        copyAndRenameRecursively(srcDir, destDir, null, null, includePatterns);
    }

    /**
     * Copie recursivement le repertoire source dans le repertoire destination
     * <p/>
     * copyRecursively("/truc/titi", "/var/tmp", "bidulle") donnera le repertoire
     * "/var/tmp/bidulle", 'bidulle' remplacant 'titi'
     *
     * @param srcDir          le répertoire source à copier
     * @param destDir         le répertoire destination où copier
     * @param renameFrom      pattern to permit rename file before uncompress it
     * @param renameTo        new name for file if renameFrom is applicable to it
     *                        you can use $1, $2, ... if you have '(' ')' in renameFrom
     * @param includePatterns les patterns que doivent resperter les
     *                        fichiers/repertoires pour etre copié. Si vide alors tout est copié
     * @throws IOException if any io pb
     */
    static public void copyAndRenameRecursively(File srcDir, File destDir,
                                                String renameFrom, String renameTo, String... includePatterns) throws IOException {
        copyAndRenameRecursively(srcDir, destDir, true, renameFrom, renameTo, false, includePatterns);
    }

    /**
     * Copie recursivement le repertoire source dans le repertoire destination
     * <p/>
     * copyRecursively("/truc/titi", "/var/tmp", "bidulle") donnera le repertoire
     * "/var/tmp/bidulle", 'bidulle' remplacant 'titi'
     *
     * @param srcDir          le répertoire source à copier
     * @param destDir         le répertoire destination où copier
     * @param includeSrcDir   si vrai alors le repertoire source est copie dans le
     *                        repertoire destination et non pas seulement les fichiers qu'il contient
     * @param renameFrom      pattern to permit rename file before uncompress it
     * @param renameTo        new name for file if renameFrom is applicable to it
     *                        you can use $1, $2, ... if you have '(' ')' in renameFrom
     * @param exclude         inverse include pattern interpretation
     * @param includePatterns les patterns que doivent resperter les
     *                        fichiers/repertoires pour etre copié. Si vide alors tout est copié
     * @throws IOException if any io pb
     */
    static public void copyAndRenameRecursively(File srcDir, File destDir,
                                                boolean includeSrcDir, String renameFrom, String renameTo, boolean exclude,
                                                String... includePatterns) throws IOException {
        String rootSrc;
        if (includeSrcDir) {
            rootSrc = srcDir.getParent();
        } else {
            rootSrc = srcDir.getPath();
        }
        List<File> files = getFilteredElements(srcDir, null, true);
        log.debug("copyRecursively: " + files);
        for (File file : files) {
            boolean doCopy = copyRecursivelyAccept(file, includePatterns);
            if (xor(exclude, doCopy)) {
                String path = file.getPath().substring(rootSrc.length());
                if (renameFrom != null && renameTo != null) {
                    String tmp = path.replaceAll(renameFrom, renameTo);
                    if (log.isDebugEnabled()) {
                        log.debug("rename " + path + " -> " + tmp);
                    }
                    path = tmp;
                }

                File destFile = new File(destDir, path);
                if (file.isDirectory()) {
                    log.debug("create directory: " + destFile);
                    //fixme on doit tester le retour de la méthode, car il se peut que le répertoire
                    // ne puisse être copié.
                    destFile.mkdirs();
                } else {
                    log.debug("copy " + path + " to " + destFile);
                    copy(file, destFile);
                }
            }
        }
    }

    /**
     * Get a ByteArrayOutputStream containing all data that could be read from the given InputStream
     * @param inputStream the stream to read
     * @param defaultBufferSize the buffer size
     * @return the input stream read for input
     * @throws IOException if any pb while reading or writing
     */
    public static ByteArrayOutputStream readBytesFrom(InputStream inputStream,
            int defaultBufferSize) throws IOException {

        ByteArrayOutputStream outputStream = new ByteArrayOutputStream(
                defaultBufferSize);
        byte[] buffer = new byte[defaultBufferSize];

        int readBytes = inputStream.read(buffer);
        while (readBytes > 0) {
            outputStream.write(buffer, 0, readBytes);
            readBytes = inputStream.read(buffer);
        }

        return outputStream;
    }

    /**
     * @param b first operande
     * @param c seconde operande
     * @return b^c
     * @deprecated Y'a un opérateur java qui fait ca très bien :) il s'agit de ^
     */
    static private boolean xor(boolean b, boolean c) {
        if (b) {
            return !c;
        } else {
            return c;
        }
    }

    /**
     * @param file            le fichier à tester.
     * @param includePatterns les patterns pour accepeter le fichier depuis son nom
     * @return <code>true</code> si le fichier est accepté, <code>false> autrement.
     */
    private static boolean copyRecursivelyAccept(File file, String[] includePatterns) {
        boolean result = includePatterns.length == 0;
        String filename = file.getAbsolutePath();
        for (String pattern : includePatterns) {
            result = filename.matches(pattern);
            if (result) {
                break;
            }
        }
        return result;
    }

} // FileUtil

