/*
 * #%L
 * Nuiton Utils :: Nuiton Profiling
 * %%
 * Copyright (C) 2006 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.profiling;

import java.util.Map.Entry;
import org.apache.commons.lang3.time.DurationFormatUtils;

import java.io.PrintStream;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.TreeSet;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.After;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;

/**
 * TODO poussin 20110824 check documentation: this documentation is for aspectwerkz, there are little
 * change for aspectj. see example file aop.xml
 *
 * Permet de tracer les appels aux methodes.
 * <p/>
 * Pour l'utiliser il faut définir un fichier XML qui intercepte les methodes
 * souhaiter.
 * <pre>
 * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 * &lt;!DOCTYPE aspectwerkz PUBLIC
 * "-//AspectWerkz//DTD//EN"
 * "http://aspectwerkz.codehaus.org/dtd/aspectwerkz2.dtd"&gt;
 * &lt;aspectwerkz&gt;
 * &lt;system id="sample"&gt;
 * &lt;aspect class="fr.ifremer.isisfish.aspect.TraceIsis"&gt;
 * &lt;pointcut name="executeMethod"&gt;
 * execution(* fr.ifremer.isisfish.datastore.ResultStorage.*(..))
 * || execution(* fr.ifremer.isisfish.aspect.Cache.*(..))
 * || execution(* fr.ifremer.isisfish.aspect.Trace.*(..))
 * || execution(* org.nuiton.topia..*(..))
 * || execution(* org.nuiton.math.matrix..*(..))
 * || execution(* fr.ifremer.isisfish.types..*(..))
 * || execution(* org.apache.commons.collections..*(..))
 * &lt;/pointcut&gt;
 * &lt;/aspect&gt;
 * &lt;/system&gt;
 * &lt;/aspectwerkz&gt;
 * </pre>
 * Ensuite il faut lancer la JVM avec deux options
 * <pre>
 * -javaagent:path/to/aspectwerkz-jdk5-2.0.jar
 * -Daspectwerkz.definition.file=path/to/trace-aop.xml
 * </pre>
 * Il est possible d'utiliser des noms normalisé et
 * trouvable dans le classpath a la place de -Daspectwerkz.definition.file
 * <li> /aspectwerkz.xml
 * <li> META-INF/aop.xml
 * <li> WEB-INF/aop.xml
 * <br/>
 * Ensuite pour afficher les statistiques dans votre programme
 * <li> log.info(NuitonTrace.getStatisticsAndClear());
 * <li> NuitonTrace.printStatistiqueAndClear();
 * Il doit être possible, plutot que d'écrire un fichier XML, de sous classer
 * NuitonTrace en ajoutant par exemple
 * <p/>
 * <pre>
 * \@Expression( "execution(* fr.ifremer.isisfish.datastore.ResultStorage.*(..))" +
 * "|| execution(* fr.ifremer.isisfish.aspect.Cache.*(..))" +
 * "|| execution(* fr.ifremer.isisfish.aspect.Trace.*(..))" +
 * "|| execution(* org.nuiton.topia..*(..))" +
 * "|| execution(* org.nuiton.math.matrix..*(..))" +
 * "|| execution(* fr.ifremer.isisfish.types..*(..))" +
 * "|| execution(* org.apache.commons.collections..*(..))"
 * )
 * Pointcut executeMethod;
 * </pre>
 * <p/>
 * Voir la classe de test {@code org.nuiton.profiling.NuitonTraceTest}.
 *
 * @see http://aspectwerkz.codehaus.org/definition_issues.html
 *
 * @author bpoussin <poussin@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 */
@Aspect
public abstract class NuitonTrace {

    static private final List<NuitonTrace> instances = new ArrayList<NuitonTrace>();

    // poussin 20110824: ne pas mettre de logger, car sinon, ca pose des problemes
    // avec l'AOP. NoClassDefFound :(
    /** to use log facility, just put in your code: log.info("..."); */
//    static private Log log = LogFactory.getLog(NuitonTrace.class);

    /** array : [call's numbers, call do, min time, max time, total time, total time with child] */
    protected Map<Method, Statistic> statistics = new LinkedHashMap<Method, Statistic>();

    protected Map<Method, Caller<Method>> callers = new HashMap<Method, Caller<Method>>();

    /**
     * array : [nest method call, start time, start time with child]
     * 
     * On ne melange pas les stack entre les threads, sinon les resultats
     * ne veulent plus rien dire car toutes les methodes des threads sont
     * melangees
     */
    protected static final ThreadLocal<Stack<Call>> callStack = new ThreadLocal<Stack<Call>> () {
        @Override
        protected Stack<Call> initialValue() {
            Stack<Call> result = new Stack<Call>();
            return result;
        }

    };

    public NuitonTrace() {
        instances.add(this);
    }

    /**
     * Retourne les statistiques d'appele pour la methode passee en parametre
     *
     * @param method
     * @return
     */
    public Statistic getStatistics(Method method) {
        Statistic result = statistics.get(method);
        if (result == null) {
            result = new Statistic(method);
            statistics.put(method, result);
        }
        return result;
    }

    /**
     * Retourne la liste des appelant pour la methode passee en parametre
     * 
     * @param method
     * @return
     */
    public Caller<Method> getCallers(Method method) {
        Caller<Method> result = callers.get(method);
        if (result == null) {
            result = new Caller<Method>();
            callers.put(method, result);
        }
        return result;
    }

    // abstract pointcut: no expression is defined
    @Pointcut
    abstract void executeMethod();

    @Before("executeMethod() && !within(org.nuiton.profiling.NuitonTrace.*)")
    public void traceBeforeExecute(JoinPoint jp) {
        // ajout dans le stack 
        Method method = ((MethodSignature) jp.getSignature()).getMethod();
        long current = System.nanoTime();

        Call stackItem = new Call(method, 0, current, current);
        callStack.get().push(stackItem);
    }

    @AfterThrowing("executeMethod() && !within(org.nuiton.profiling.NuitonTrace.*)")
    public void traceAfterThrowingExecute(JoinPoint jp) {
        // si une exeption est leve, il faut faire la meme chose
        traceAfterExecute(jp);
    }

    @After("executeMethod() && !within(org.nuiton.profiling.NuitonTrace.*)")
    public void traceAfterExecute(JoinPoint jp) {
        Method method = ((MethodSignature) jp.getSignature()).getMethod();

        long current = System.nanoTime();

        Stack<Call> callStack = this.callStack.get();

        if (callStack.isEmpty()) {
//            log.warn("Empty stack in afterExecute for method " + method.getName());
        } else {
            Call stackItem = callStack.pop();
            long timeSpent = current - stackItem.timeStart;
            long timeSpentNestMethod = current - stackItem.timeStartNestMethod;

            Statistic stat = getStatistics(method);
            stat.call++; // add +1 to call number
            stat.callNestMethod += stackItem.callNestMethod;
            stat.timeTotal += timeSpent;
            stat.timeTotalNestMethod += timeSpentNestMethod;
            if (stat.timeMin > timeSpent) {
                stat.timeMin = timeSpent;
            }
            if (stat.timeMax < timeSpent) {
                stat.timeMax = timeSpent;
            }

            if (!callStack.isEmpty()) {
                Call parent = callStack.peek();
                parent.callNestMethod++;  // add +1 to call number nest method
                parent.timeStart += timeSpentNestMethod; // remove to time all time spent in nest method (yes + to remove :)

                getCallers(method).add(parent.method);
            }
        }

        // ajouter le delta de temps dans le temps passé dans la méthod

        // il faud garder le temps passé dans l'appel d'autre methode de la stack
        // --> A
        //   =========
        //   --> B
        //   <-- B
        //   =========
        //   --> C
        //     --> D
        //     <-- D
        //   <-- C  
        //   =========
        // <-- A

        // le temps reellement passé dans A est representé par les =====
    }

    /** @return les statistiques in CSV format*/
    static public String getStatisticsCSVAndClear() {
        StringBuilder result = new StringBuilder();
        result.append("method;")
                .append("call;")
                        // TIme is in nano not millis, we must divide by 1000000
                .append("min;")
                .append("mean;")
                .append("max;")
                .append("total;")
                .append("call_nest;")
                .append("total_with_nest;")
                .append("callers;")
                .append("\n");
        
        for (NuitonTrace trace : instances) {
            for (Method method : trace.statistics.keySet()) {
                Statistic stat = trace.getStatistics(method);
                Caller<Method> caller = trace.getCallers(method);
                long meanTime = stat.timeTotal / stat.call;
                result.append(method)
                        .append(";").append(stat.call)
                        // Time is in nano not millis, we must divide by 1000000
                        .append(";").append(DurationFormatUtils.formatDuration(stat.timeMin / 1000000, "s'.'S"))
                        .append(";").append(DurationFormatUtils.formatDuration(meanTime / 1000000, "s'.'S"))
                        .append(";").append(DurationFormatUtils.formatDuration(stat.timeMax / 1000000, "s'.'S"))
                        .append(";").append(DurationFormatUtils.formatDuration(stat.timeTotal / 1000000, "s'.'S"))
                        .append(";").append(stat.callNestMethod)
                        .append(";").append(DurationFormatUtils.formatDuration(stat.timeTotalNestMethod / 1000000, "s'.'S"))
                        .append(";").append(caller)
                        .append("\n");
            }
            trace.statistics.clear();
            trace.callers.clear();
        }
        // Poussin 20110826 on ne vide pas les instances mais les stats
//        instances.clear();
        return result.toString();
    }

    /** @return les statistiques */
    static public String getStatisticsAndClear() {
        StringBuilder result = new StringBuilder();
        for (NuitonTrace trace : instances) {
            result.append("--- Statistics ---\n");
            for (Method method : trace.statistics.keySet()) {
                Statistic stat = trace.getStatistics(method);
                long meanTime = stat.timeTotal / stat.call;
                result.append(method).append("\t")
                        .append(" call: ").append(stat.call)
                        // TIme is in nano not millis, we must divide by 1000000
                        .append(" min: ").append(DurationFormatUtils.formatDuration(stat.timeMin / 1000000, "s'.'S"))
                        .append(" mean: ").append(DurationFormatUtils.formatDuration(meanTime / 1000000, "s'.'S"))
                        .append(" max: ").append(DurationFormatUtils.formatDuration(stat.timeMax / 1000000, "s'.'S"))
                        .append(" total: ").append(DurationFormatUtils.formatDuration(stat.timeTotal / 1000000, "s'.'S"))
                        .append(" call_nest: ").append(stat.callNestMethod)
                        .append(" total_with_nest: ").append(DurationFormatUtils.formatDuration(stat.timeTotalNestMethod / 1000000, "s'.'S"))
                        .append(" callers: ").append(trace.getCallers(method))
                        .append("\n");
            }
            result.append("--------------------\n");
            trace.statistics.clear();
            trace.callers.clear();
        }
        // Poussin 20110826 on ne vide pas les instances mais les stats
//        instances.clear();
        return result.toString();
    }

    static public void printStatisticsAndClear() {
        printStatisticsAndClear(System.out);
    }

    static public void printStatisticsAndClear(PrintStream printer) {
        String stat = getStatisticsAndClear();
        if (stat != null && !"".equals(stat)) {
            printer.println(stat);
        }
    }

    /**
     * Statistique sur un appele de methode
     */
    protected static class Statistic {
        Method method;
        /** nombre d'appel */
        long call;
        /** nombre d'appel vers une autre method depuis cette methode */
        long callNestMethod;
        /** temps mini d'execution de cette methode (sans le temps des autres methodes) */
        long timeMin;
        /** temps max d'execution de cette methode (sans le temps des autres methodes) */
        long timeMax;
        /** temps total d'execution de cette methode (sans le temps des autres methodes) */
        long timeTotal;
        /** temps total d'execution de cette methode (avec le temps des autres methodes) */
        long timeTotalNestMethod;

        public Statistic(Method method) {
            this.method = method;
        }

    }

    /**
     * Represente un call de methode
     */
    protected static class Call {
        Method method;
        /** nombre d'appel vers une autre method depuis cette methode */
        long callNestMethod;
        /** heure de depart de l'appel a la methode (sans le temps des autres methodes) */
        long timeStart;
        /** heure de depart de l'appel a la methode (avec le temps des autres methodes) */
        long timeStartNestMethod;

        public Call(Method method, long callNestMethod, long timeStart, long timeStartNestMethod) {
            this.method = method;
            this.callNestMethod = callNestMethod;
            this.timeStart = timeStart;
            this.timeStartNestMethod = timeStartNestMethod;
        }

    }

    /**
     * Classe qui permet d'ajouter un element et compte le nombre de fois
     * que l'on a ajoute cet element.
     * 
     * exemple d'utilisation: on souhaite compter le nombre de fois qu'une methode
     * est appeler par une autre, on ajoute a chaque fois que quelqu'un fait
     * appele a elle ce quelqu'un.
     * 
     */
    protected static class Caller<E> {

        protected Map<E, Integer> data = new HashMap<E, Integer>();
        
        /**
         * indique que l'on ajoute un element
         * @param e
         */
        public void add(E e) {
            Integer v = data.get(e);
            if (v == null) {
                v = 0;
            }
            data.put(e, v+1);
        }

        /**
         * Retourne la liste des objets ajouter avec le nombre de fois qu'ils
         * ont ete ajoute. Trie par le nombre de fois qu'ils ont ete ajoute,le
         * plus grand nombre en 1er
         * 
         * @return
         */
        @Override
        public String toString() {
            Comparator<Map.Entry<E, Integer>> comparator = new Comparator<Map.Entry<E, Integer>>() {
                public int compare(Entry<E, Integer> o1, Entry<E, Integer> o2) {
                    // on inverse, car on veut le plus grand en 1er
                    return o2.getValue().compareTo(o1.getValue());
                }
            };

            Set<Map.Entry<E, Integer>> entries = data.entrySet();
            Set<Map.Entry<E, Integer>> sorted = 
                    new TreeSet<Map.Entry<E, Integer>>(comparator);
            sorted.addAll(entries);

            StringBuilder result = new StringBuilder();

            for (Map.Entry<E, Integer> e : sorted) {
                result.append(e.getValue()).append("=[").append(e.getKey()).append("]").append(",");
            }
            if (result.length() > 0) {
                // on supprime la derniere virgule en trop
                result.deleteCharAt(result.length()-1);
            }

            return result.toString();
        }



    }

}
