/* *##% Nuiton Java-2-R library
 * Copyright (C) 2006 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*/
package org.nuiton.j2r.types;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.j2r.REngine;
import org.nuiton.j2r.RException;
import org.nuiton.j2r.RInstructions;

/**
 * Java implementation of the R data.frame
 *
 * @author couteau
 */
public class RDataFrame extends REXPAbstract implements REXP {

    private Log log = LogFactory.getLog(RDataFrame.class);

    //TODO implement the logger use !
    //Vector containing the names of the rows of the data.frame vectors
    private List<String> rowNames;
    //Vector containing the vectors of the data.frame
    private List<List<? extends Object>> data;

    /**
     * Constructor
     *
     * @param engine the engine in which the data frame will be located.
     */
    public RDataFrame(REngine engine) {
        super();
        this.names = new ArrayList<String>();
        this.rowNames = new ArrayList<String>();
        this.data = new ArrayList<List<? extends Object>>();
        this.variable = "";
        this.engine = engine;
        this.attributes = new HashMap<String, Object>();
    }

    /**
     * Constructor
     *
     * @param engine the engine in which the data frame will be located.
     * @param datatypes a table object determining the type of each column of
     * the data.frame.
     * @param y the length of each vector that compound the data.frame.
     */
    public RDataFrame(REngine engine, Object[] datatypes, int y) throws
            RException {
        super();
        this.names = new ArrayList<String>();
        this.rowNames = new ArrayList<String>();
        this.data = new ArrayList<List<? extends Object>>();
        for (int i = 0; i < datatypes.length; i++) {
            //create one column for each datatype element
            //check if type is supported
            checkType(datatypes[i]);
            //create the list of objects
            List<Object> column = new ArrayList<Object>();
            //Fill it with instances of the datatype
            for (int j = 0; j < y; j++) {
                if (datatypes[i] instanceof Double) {
                    column.add(0.0);
                } else if (datatypes[i] instanceof Integer) {
                    column.add(0);
                } else if (datatypes[i] instanceof Boolean) {
                    column.add(true);
                } else if (datatypes[i] instanceof String) {
                    column.add("");
                }
            }
            //add the column to the dataframe.
            data.add(column);
        }
        this.variable = "";
        this.engine = engine;
        this.attributes = new HashMap<String, Object>();
    }

    /**
     * Constructor
     *
     * @param engine the engine in which the data frame will be located.
     * @param names names of the the data.frame columns.
     * @param rowNames names of the data.frame rows.
     * @param data the data of the data.frame
     * @param variable the name of the data.frame in R.
     * @throws org.nuiton.j2r.RException if an error occur while trying to
     * initialize 
     */
    public RDataFrame(REngine engine, List<String> names,
            List<String> rowNames, List<List<? extends Object>> data,
            String variable) throws RException {
        super();
        this.names = names;
        this.rowNames = rowNames;
        this.data = data;
        this.variable = variable;
        this.engine = engine;
        this.attributes = new HashMap<String, Object>();
        try {
            engine.eval(this.toRString());
        } catch (RException eee) {
            throw new RException("Cannot initialize data.frame in R", eee);
        }
    }

    /**
     * Constructor, used for autoconversion from rexp to data.frame.
     *
     * @param engine the engine in which the data frame will be located.
     * @param names names of the the data.frame columns.
     * @param rowNames names of the data.frame rows.
     * @param data the data of the data.frame
     * @param variable the name of the data.frame in R.
     */
    public RDataFrame(REngine engine, String[] names,
            String[] rowNames, List<List<? extends Object>> data,
            String variable) {
        //Create the name ArrayList and fill it with the content of the name String[]
        String[] tempNames = {};
        if (names != null) {
            tempNames = names.clone();
        }
        String[] tempRowNames = {};
        if (rowNames != null) {
            tempRowNames = rowNames.clone();
        }
        this.names = new ArrayList<String>();
        for (int i = 0; i < tempNames.length; i++) {
            this.names.add(tempNames[i]);
        }
        //Create the rowNames ArrayList and fill it with the content of the rowNames
        //String[]
        this.rowNames = new ArrayList<String>();
        for (int i = 0; i < tempRowNames.length; i++) {
            this.rowNames.add(tempRowNames[i]);
        }
        this.data = data;
        this.variable = variable;
        this.engine = engine;
        this.attributes = new HashMap<String, Object>();
        try {
            engine.eval(this.toRString());
        } catch (RException eee) {
            //As for autoconversion,there is no variable name, this will always
            //be thrown. Catch it and log it instead so that it does not change
            //the behaviour and we can use this method with and without variable
            //name
            if (log.isDebugEnabled()) {
                log.debug("Cannot initialize data.frame in R : " +
                        eee.getMessage());
            }
        }
    }

    /**
     * Method to get the names of the rows of the R data.frame
     * 
     * @return a ArrayList of strings containing the names of each row of the R
     *         data.frame
     * @throws RException
     *             if an error occurs while getting back the names from R.
     * @throws IndexOutOfBoundsException
     *             when the row.names size get from R is bigger than the local
     *             data size.
     */
    public List<String> getRowNames() throws RException {
        if (engine.isAutoCommit()) {
            //Get back the names from R.
            String[] rowNamesArray = (String[]) engine.eval(String.format(
                    RInstructions.GET_ROW_NAMES, this.variable));

            //Check if size is correct, if yes, modify local data.
            if (rowNamesArray.length <= this.data.get(0).size()) {
                this.rowNames.clear();
                for (int i = 0; i < rowNamesArray.length; i++) {
                    rowNames.add(rowNamesArray[i]);
                }
                return rowNames;
            } else {
                //if size is not correct, throw a RException.
                throw new IndexOutOfBoundsException(String.format(
                        dataInconsistencyText, rowNamesArray.length,
                        this.data.get(0)));
            }
        } else {
            return this.rowNames;
        }

    }

    /**
     * Method to get the row name of the row index y+1 of the R data.frame.
     * 
     * @param y
     *            index of the row (0 to n-1)
     * 
     * @return the name of the ArrayList
     * 
     * @throws RException
     *             if an error occurs while getting back the row name from R.
     */
    public String getRowName(int y) throws RException {
        //check if the index is valid
        if (y < rowNames.size()) {
            if (engine.isAutoCommit()) {
                //Get back the names from R.
                String name = (String) engine.eval(String.format(
                        RInstructions.GET_ROW_NAME, this.variable, y + 1));

                //Check if the String is returned.
                if ((name != null) && (!name.equals(""))) {
                    this.rowNames.set(y, name);
                }
            }
        } else {
            //if String is not returned, throw an IndexOutOfBOundsException.
            throw new IndexOutOfBoundsException(String.format(indexExceptionText,
                    y, rowNames.size()));
        }
        return this.rowNames.get(y);

    }

    /**
     * Method to assign the names of the rows of the R data.frame
     * 
     * @param rowNames
     *            a ArrayList containing the names of the rows of the R
     *            data.frame
     * @throws RException
     *             if an error occur while in R.
     * 
     * @throws IndexOutOfBoundsException
     *             when the row names ArrayList is longer than the data.frame
     *             length
     */
    public void setRowNames(List<String> rowNames) throws RException {

        //Check if the size is correct, if yes, do the modifications and send them to R.
        if (rowNames.size() == data.get(0).size()) {
            this.rowNames = rowNames;

            //Create the r instruction (row.names(var)<-c("name 1",...,"name x")).
            String rowNamesString = "";
            for (int i = 0; i < this.rowNames.size(); i++) {
                if (i != 0) {
                    rowNamesString += ",";
                }
                rowNamesString += "\"" + rowNames.get(i) + "\"";
            }
            String rexp = String.format(RInstructions.SET_ROW_NAMES,
                    this.variable, rowNamesString);

            //Send the r instruction to the engine.
            engine.voidEval(rexp);
        } else {
            //if size is not correct, throw a RException.
            throw new IndexOutOfBoundsException(String.format(indexExceptionText,
                    rowNames.size(), data.get(0).size()));
        }
    }

    /**
     * Method to assign the name of the ArrayList at the index x.
     * 
     * @param x
     *            index of the ArrayList (0 to n-1)
     * @param name
     *            Name of the ArrayList.
     * @throws RException
     *             if an error occur while in R.
     * @throws IndexOutOfBoundsException
     *             when the index is out of the data.frame bounds.
     */
    public void setRowName(int x, String rowName) throws RException {
        //check if the index is valid
        if (x < rowNames.size()) {
            rowNames.set(x, rowName);

            //create the r instruction (row.names(var)[x]<-"rowName")
            String rexp = String.format(RInstructions.SET_ROW_NAME,
                    this.variable, x + 1, rowName);

            //Send the R instruction to the engine.
            engine.voidEval(rexp);
        } else {
            //if index is out of bounds, throw a RException
            throw new IndexOutOfBoundsException(String.format(indexExceptionText,
                    x, data.size()));
        }
    }

    /**
     * Method to export the data.frame in a String for evaluation in R.
     * 
     * @return a R string representation of the data.frame to create it in R.
     * @throws RException
     *             If no variable name is given
     */
    @Override
    public String toRString() throws RException {
        checkVariable();
        String returnString = this.variable + "<-data.frame(";
        if (!(this.data.isEmpty())) {
            for (int i = 0; i < data.size(); i++) {
                if (!(this.names.isEmpty())) {
                    returnString += this.names.get(i) + "=c(";
                } else {
                    returnString += "c(";
                }
                if (data.get(i).get(0) instanceof String) {
                    for (int j = 0; j < data.get(i).size(); j++) {
                        returnString += "\"" + data.get(i).get(j) + "\",";
                    }
                } else if (data.get(i).get(0) instanceof Boolean) {
                    for (int j = 0; j < data.get(i).size(); j++) {
                        if ((Boolean) data.get(i).get(j)) {
                            returnString += RInstructions.TRUE + ",";
                        } else {
                            returnString += RInstructions.FALSE + ",";
                        }

                    }
                } else if (data.get(i).get(0) instanceof Integer) {
                    for (int j = 0; j < data.get(i).size(); j++) {
                        returnString += String.format(RInstructions.AS_INTEGER,
                                data.get(i).get(j)) + ",";
                    }
                } else {
                    for (int j = 0; j < data.get(i).size(); j++) {
                        returnString += data.get(i).get(j) + ",";
                    }
                }
                returnString = returnString.substring(0,
                        returnString.length() - 1);
                returnString = returnString + "),";

            }
        }
        if (!(this.rowNames.isEmpty())) {
            returnString += "row.names=c(";
            for (int i = 0; i < rowNames.size(); i++) {
                returnString += "\"" + rowNames.get(i) + "\",";
            }
            returnString =
                    returnString.substring(0, returnString.length() - 1) +
                    "),stringsAsFactors=FALSE)";

        } else if (this.data.isEmpty()) {
            returnString += ")";
        } else {
            returnString += "stringsAsFactors=FALSE)";
        }
        return returnString;
    }

    /**
     * Method to set the value (x,y) to a Double value.
     * 
     * @param x
     *            x coordinate (from 0 to size-1)
     * @param y
     *            y coordinate (from 0 to size-1)
     * @param data
     *            value to set
     * @throws RException
     *             if no variable name has been given to the data.frame or the
     * data type is not allowed at this location.
     */
    public void set(int x, int y, Object data) throws RException {
        checkVariable();
        checkX(x);
        checkY(y);
        try {
            if (this.data.get(x).get(y) instanceof Double) {
                ((ArrayList<Double>) this.data.get(x)).set(y, (Double) data);
                engine.voidEval(String.format(RInstructions.SET_DATAFRAME_ITEM,
                        this.variable, y + 1,
                        x + 1,
                        data));
            } else if (this.data.get(x).get(y) instanceof Boolean) {
                ((ArrayList<Boolean>) this.data.get(x)).set(y, (Boolean) data);
                if ((Boolean) data) {
                    engine.voidEval(String.format(
                            RInstructions.SET_DATAFRAME_ITEM, this.variable,
                            y + 1, x + 1, RInstructions.TRUE));
                } else {
                    engine.voidEval(String.format(
                            RInstructions.SET_DATAFRAME_ITEM, this.variable,
                            y + 1, x + 1, RInstructions.FALSE));
                }
            } else if (this.data.get(x).get(y) instanceof String) {
                ((ArrayList<String>) this.data.get(x)).set(y, (String) data);
                engine.voidEval(String.format(RInstructions.SET_DATAFRAME_ITEM,
                        this.variable, y + 1, x + 1, "\"" + data + "\""));
            } else if (this.data.get(x).get(y) instanceof Integer) {
                ((ArrayList<Integer>) this.data.get(x)).set(y, (Integer) data);
                engine.voidEval(String.format(RInstructions.SET_DATAFRAME_ITEM,
                        this.variable, y + 1, x + 1, String.format(
                        RInstructions.AS_INTEGER, data)));
            } else {
                throw new ArrayStoreException(
                        "The data.frame does not accept this type on those coordinates : " +
                        data.getClass());
            }
        } catch (ClassCastException cce) {
            throw new RException(
                    "The data.frame does not accept this type on those coordinates : " +
                    data.getClass(), cce);
        }

    }

    /**
     * Method to get the object located at the [x,y] coordinates.
     * 
     * @param x
     *            x coordinates (between 0 and size-1)
     * @param y
     *            y coordinates (between 0 and size-1)
     * @return the Object located at the [x,y] coordinates
     * @throws RException
     *             if no variable name has been given to the data.frame
     * @throws IndexOutOfBoundsException
     *             if the x or y coordinates are not correct.
     */
    public Object get(int x, int y) throws RException {
        checkX(x);
        checkY(y);
        if (engine.isAutoCommit()) {
            Object returnObject = engine.eval(String.format(
                    RInstructions.GET_DATAFRAME_ITEM, this.variable, y + 1,
                    x + 1));
            if (returnObject instanceof String) {
                ((ArrayList<String>) this.data.get(x)).set(y,
                        (String) returnObject);
            } else if (returnObject instanceof Double) {
                ((ArrayList<Double>) this.data.get(x)).set(y,
                        (Double) returnObject);
            } else if (returnObject instanceof Integer) {
                ((ArrayList<Integer>) this.data.get(x)).set(y,
                        (Integer) returnObject);
            } else if (returnObject instanceof Boolean) {
                ((ArrayList<Boolean>) this.data.get(x)).set(y,
                        (Boolean) returnObject);
            }
        }
        return this.data.get(x).get(y);
    }

    /**
     * Method to get the ArrayLists of the R data.frame (there is no
     * synchronizing with R).
     * 
     * @return a ArrayList containing the ArrayLists of the R data.frame
     */
    public List<List<? extends Object>> getData() {
        //TODO manage the autocommit mode here.
        return data;
    }

    /**
     * Method to assign the data of the R data.frame (there is no synchronizing
     * with R, use the commit() method to send data to R.
     * 
     * @param data
     *            a ArrayList of ArrayLists, containing each ArrayList of the R
     *            data.frame
     * @param engine
     *            a REngine where the R session is located.
     * @throws RException
     */
    public void setData(List<List<? extends Object>> data) throws RException {
        this.data = data;
        engine.voidEval(toRString());
    }

    /**
     * Method to get a data.frame from a variable in R.
     * 
     * @param variable
     *            name of the data.frame in R
     * @param engine
     *            a REngine where the R session is located.
     * @throws RException
     */
    @Override
    public void getFrom(String variable) throws RException {
        this.variable = variable;
        if (engine.isAutoCommit()) {
            if (rowNames != null) {
                rowNames.clear();
            } else {
                rowNames = new ArrayList<String>();
            }
            if (names != null) {
                names.clear();
            } else {
                names = new ArrayList<String>();
            }
            if (data != null) {
                data.clear();
            } else {
                data = new ArrayList<List<? extends Object>>();
            }
            if (attributes != null) {
                attributes.clear();
            } else {
                attributes = new HashMap<String, Object>();
            }

            //update row names
            String[] rowNamesArray = (String[]) engine.eval(String.format(
                    RInstructions.GET_ROW_NAMES, this.variable));
            for (int i = 0; i < rowNamesArray.length; i++) {
                rowNames.add(rowNamesArray[i]);
            }

            //update names
            String[] namesArray = (String[]) engine.eval(String.format(
                    RInstructions.GET_NAMES, this.variable));
            for (int i = 0; i < namesArray.length; i++) {
                names.add(namesArray[i]);
            }

            //update data
            Integer dataframelength = (Integer) engine.eval(String.format(
                    RInstructions.LENGTH, this.variable));

            for (int i = 0; i < dataframelength; i++) {
                Integer arrayListLength = (Integer) engine.eval(String.format(
                        RInstructions.LENGTH_COLUMN, this.variable, i + 1));
                ArrayList<Serializable> thisColumn =
                        new ArrayList<Serializable>();
                for (int j = 0; j < arrayListLength; j++) {
                    thisColumn.add((Serializable) engine.eval(String.format(
                            RInstructions.GET_DATAFRAME_ITEM, this.variable,
                            j + 1, i + 1)));
                }
                data.add(thisColumn);

            }

            //update attributes
            Integer attributeslength = (Integer) engine.eval(String.format(
                    RInstructions.LENGTH_ATTRIBUTES, this.variable));

            for (int i = 0;
                    i < attributeslength; i++) {
                String key = (String) engine.eval(String.format(
                        RInstructions.GET_ATTRIBUTE_NAME, this.variable, i + 1));

                String attribute = (String) engine.eval("toString(" + String.format(
                        RInstructions.GET_ATTRIBUTE, this.variable, key) + ")");
                attributes.put(key, attribute);
            }
        }
    }

    /**
     * Export the data.frame into csv format.
     * @param outputFile the file to write
     * @param rowNames true/false to determine if the row names will be put on
     * the export.
     * @param names true/false to determine if the column names will be put on
     * the export.
     */
    public void exportCsv(File outputFile, boolean rowNames, boolean names)
            throws IOException {
        BufferedWriter file = new BufferedWriter(new FileWriter(outputFile));

        if (names) {
            if (rowNames) {
                file.write(";");
            }
            for (int i = 0; i < this.names.size(); i++) {
                file.write(this.names.get(i) + ";");
            }
            file.newLine();
        }

        for (int i = 0; i < this.data.get(0).size(); i++) {
            if (rowNames) {
                file.write(this.rowNames.get(i) + ";");
            }
            for (int j = 0; j < this.data.size(); j++) {
                file.write(this.data.get(j).get(i) + ";");
            }
            file.newLine();
        }
        file.close();

    }

    /**
     * Import a dataframe form a csv file. The dataframe will contain Strings.
     * Use this method if you don't know the type of data that is in the csv
     * file.
     * 
     * @param inputFile
     *            Csv file to import.
     * @param rowNames
     *            Does the csv file contains names of the rows.
     * @param names
     *            Does the csv file contain names of the columns.
     */
    public void importCsv(File inputFile, boolean rowNames, boolean names)
            throws FileNotFoundException, IOException {
        List<Object> tmp = new ArrayList<Object>();
        tmp.add("string");
        importCsv(inputFile, rowNames, names, (List) tmp);

    }

    /**
     * Import a dataframe form a csv file. The dataframe will contain Objects of
     * the same class than importType. Use this method if you know the type of
     * data that is in the csv file.
     * 
     * @param inputFile
     *            Csv file to import.
     * @param rowNames
     *            Does the csv file contains names of the rows.
     * @param names
     *            Does the csv file contain names of the columns.
     * @param importType
     *            Object of the class of the data imported (all the data have
     *            the same type). (Supported types : String, Double and Integer)
     */
    public void importCsv(File inputFile, boolean rowNames, boolean names,
            Object importType) throws FileNotFoundException, IOException {
        List<Object> tmp = new ArrayList<Object>();
        if (importType instanceof String) {
            tmp.add((String) importType);
            importCsv(inputFile, rowNames, names, (List) tmp);
        } else if (importType instanceof Double) {
            tmp.add((Double) importType);
            importCsv(inputFile, rowNames, names, tmp);
        } else if (importType instanceof Integer) {
            tmp.add((Integer) importType);
            importCsv(inputFile, rowNames, names, tmp);
        }
    }

    /**
     * Import a dataframe from a csv file. The dataframe will contain Objects of
     * the same class than each element of importTypes (one element = one
     * column). Use this method if you know the type of data that is in the csv
     * file.
     * 
     * @param inputFile
     *            Csv file to import.
     * @param rowNames
     *            Does the csv file contains names of the rows.
     * @param names
     *            Does the csv file contain names of the columns.
     * @param importTypes
     *            ArrayList of Object of the class of the data imported (the
     *            ArrayList match the data type of the columns). (Supported
     *            types : String, Double and Integer)
     */
    public void importCsv(File inputFile, boolean rowNames, boolean names,
            List<Object> importTypes) throws FileNotFoundException, IOException {
        //temporary String to read lines.
        String tmp;
        Integer dataSize = 0;
        //get the first line of the file
        BufferedReader br = new BufferedReader(new FileReader(inputFile));
        tmp = br.readLine();
        String[] splitted = tmp.split("\\;");
        //get the data size (number of columns)
        if (rowNames) {
            dataSize = splitted.length - 1;
        } else {
            dataSize = splitted.length;
        }
        //if data has already been initialized, clear it.
        if (this.data != null) {
            this.data.clear();
        }
        if ((rowNames) && (this.rowNames != null)) {
            this.rowNames.clear();

        }
        if (names) {
            //if names are present in the file, parse the first line to get
            //the names.
            if (this.names != null) {
                //if names has already been initialized, clear it.
                this.names.clear();
            }
            //get every name from the first line
            for (int i = 1; i < splitted.length; i++) {
                this.names.add(splitted[i]);
            }
        }

        //temporary data list.
        List<List<? extends Object>> tempData =
                new ArrayList<List<? extends Object>>();

        //Initialize all the data columns
        for (int i = 0; i < dataSize; i++) {
            List<Object> column = new ArrayList<Object>();
            tempData.add(column);
        }

        while ((tmp = br.readLine()) != null) {
            //parse each line
            splitted = tmp.split("\\;");
            //to determine the data index on the line.
            int index = 0;
            if (rowNames) {
                //if there are row names, extract the first item as the row name
                this.rowNames.add(splitted[0]);
                index = 1;
            }
            for (int i = 0 + index; i < splitted.length; i++) {
                //cast the data imported to the specified type.
                //test the size of the inputType list. If 1 import all the
                //columns as the type of the first element.
                if (importTypes.size() == 1) {
                    if (importTypes.get(0) instanceof String) {
                        ((ArrayList<Object>) tempData.get(i - index)).add(
                                (Serializable) splitted[i]);
                    } else if (importTypes.get(0) instanceof Double) {
                        ((ArrayList<Object>) tempData.get(i - index)).add(Double.valueOf(
                                splitted[i]));
                    } else if (importTypes.get(0) instanceof Integer) {
                        ((ArrayList<Object>) tempData.get(i - index)).add(Integer.valueOf(
                                splitted[i]));
                    }

                } else if (importTypes.get(i - index) instanceof String) {
                    ((ArrayList<Object>) tempData.get(i - index)).add(
                            (Serializable) splitted[i]);
                } else if (importTypes.get(i - index) instanceof Double) {
                    ((ArrayList<Object>) tempData.get(i - index)).add(Double.valueOf(
                            splitted[i]));
                } else if (importTypes.get(i - index) instanceof Integer) {
                    ((ArrayList<Object>) tempData.get(i - index)).add(Integer.valueOf(
                            splitted[i]));
                }
            }
        }
        br.close();
        this.data = tempData;
    }

    private void checkY(int y) {
        if (!(this.data.get(0).isEmpty()) && (y > this.data.get(0).size())) {
            throw new IndexOutOfBoundsException(String.format(indexExceptionText,
                    y, this.data.size()));
        }
    }

    /**
     * Check if the index is into the data.frame length throws a
     * IndexOutOfBoundsException if not the index is too big.
     *
     * @param x index
     */
    @Override
    public void checkX(int x) {
        if ((x > this.data.size()) && !(this.data.isEmpty())) {
            throw new IndexOutOfBoundsException(String.format(indexExceptionText,
                    x, this.data.size()));
        }
    }

    private void checkType(Object o) throws RException {
        if (!(o instanceof String) && !(o instanceof Double) &&
                !(o instanceof Integer) && !(o instanceof Boolean)) {
            throw new RException("Not supported type");
        }
    }
}