/*
 * #%L
 * Nuiton Utils :: Nuiton Csv
 * 
 * $Id: ImportToMap.java 2244 2011-11-30 17:12:49Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.4/nuiton-csv/src/main/java/org/nuiton/util/csv/ImportToMap.java $
 * %%
 * Copyright (C) 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.util.csv;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;

import static org.nuiton.i18n.I18n._;

/**
 * A extended {@link Import} to read csv lines into a single map.
 * <p/>
 * <strong>Warning:</strong> The map used to push values for a csv line is the
 * same for all lines, it means you have to copy to your own object.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.4
 */
public class ImportToMap extends Import<Map<String, Object>> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(ImportToMap.class);

    public static ImportToMap newImport(ImportModel<Map<String, Object>> model,
                                        InputStream inputStream) {
        return new ImportToMap(model, inputStream);
    }

    public static ImportToMap newImport(ImportModel<Map<String, Object>> model,
                                        Reader reader) {
        return new ImportToMap(model, reader);
    }

    @Override
    public Iterator<Map<String, Object>> iterator() {

        // obtain headers from csv input and validate the model
        prepareAndValidate();

        // read first line since first line is header
        readFirstLine();

        return new Iterator<Map<String, Object>>() {

            boolean hasNext = true;

            int lineNumber;

            Map<String, Object> element = new HashMap<String, Object>();

            @Override
            public boolean hasNext() {
                return hasNext;
            }

            @Override
            public Map<String, Object> next()
                    throws NoSuchElementException, ImportRuntimeException {

                if (!hasNext) {
                    throw new NoSuchElementException();
                }

                lineNumber += 1;

                // clean all values from the element
                element.clear();

                for (ImportableColumn<Map<String, Object>, Object> field :
                        getNonIgnoredHeaders()) {

                    // read value from csv cell
                    String value;
                    try {
                        value = reader.get(field.getHeaderName());
                    } catch (Exception e) {
                        reader.close();
                        throw new ImportRuntimeException(_("csv.import.error.unableToReadField",
                                                           field.getHeaderName(), lineNumber), e);
                    }

                    // contravariance ftw
                    Object parsedValue;
                    try {
                        parsedValue = field.parseValue(value);
                    } catch (Exception e) {
                        String message = _("csv.import.error.unableToParseValue",
                                           value, field.getHeaderName(), lineNumber)
                                         + "\n" + e.getMessage();
                        throw new ImportRuntimeException(message, e);
                    }

                    // set value to element
                    try {
                        field.setValue(element, parsedValue);
                    } catch (Exception e) {
                        String message = _("csv.import.error.unableToSetValue",
                                           parsedValue,
                                           element.toString(),
                                           lineNumber, field.getHeaderName());
                        if (log.isErrorEnabled()) {
                            log.error(message);
                        }
                        throw new ImportRuntimeException(message, e);
                    }
                }

                try {
                    hasNext = reader.readRecord();
                } catch (IOException e) {
                    reader.close();
                    throw new ImportRuntimeException(
                            _("csv.import.error.unableToReadLine", lineNumber + 1), e);
                }

                return element;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    protected ImportToMap(ImportModel<Map<String, Object>> mapImportModel,
                          InputStream inputStream) {
        super(mapImportModel, inputStream);
    }

    protected ImportToMap(ImportModel<Map<String, Object>> mapImportModel,
                          Reader reader) {
        super(mapImportModel, reader);
    }
}
